"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const RDFDM = require("@rdfjs/data-model");
const Expressions_1 = require("./Expressions");
const C = require("../util/Consts");
const Err = require("../util/Errors");
class Term {
    constructor() {
        this.expressionType = Expressions_1.ExpressionType.Term;
    }
    str() {
        throw new Err.InvalidArgumentTypes([this], C.RegularOperator.STR);
    }
    coerceEBV() {
        throw new Err.EBVCoercionError(this);
    }
}
exports.Term = Term;
// NamedNodes -----------------------------------------------------------------
class NamedNode extends Term {
    constructor(value) {
        super();
        this.value = value;
        this.termType = 'namedNode';
    }
    toRDF() {
        return RDFDM.namedNode(this.value);
    }
    str() {
        return this.value;
    }
}
exports.NamedNode = NamedNode;
// BlankNodes -----------------------------------------------------------------
class BlankNode extends Term {
    constructor(value) {
        super();
        this.value = value;
        this.termType = 'blankNode';
        this.value = value;
    }
    static nextID() {
        BlankNode._nextID += 1;
        return BlankNode.nextID.toString();
    }
    toRDF() {
        return RDFDM.blankNode(this.value);
    }
}
BlankNode._nextID = 0;
exports.BlankNode = BlankNode;
// Literals-- -----------------------------------------------------------------
class Literal extends Term {
    constructor(typedValue, typeURL, strValue, language) {
        super();
        this.typedValue = typedValue;
        this.typeURL = typeURL;
        this.strValue = strValue;
        this.language = language;
        this.termType = 'literal';
        this.type = C.type(typeURL.value);
    }
    toRDF() {
        return RDFDM.literal(this.strValue || this.str(), this.language || this.typeURL);
    }
    str() {
        return this.strValue || this.typedValue.toString();
    }
}
exports.Literal = Literal;
class NumericLiteral extends Literal {
    coerceEBV() {
        return !!this.typedValue;
    }
    toRDF() {
        const term = super.toRDF();
        if (!isFinite(this.typedValue)) {
            term.value = term.value.replace('Infinity', 'INF');
        }
        return term;
    }
    str() {
        return this.strValue
            || NumericLiteral.specificFormatters[this.type](this.typedValue);
    }
}
NumericLiteral.specificFormatters = {
    integer: (value) => value.toFixed(),
    float: (value) => value.toString(),
    decimal: (value) => value.toString(),
    // // Be consistent with float
    // decimal: (value) => {
    //   const jsDecimal = value.toString();
    //   return jsDecimal.match(/\./)
    //     ? jsDecimal
    //     : jsDecimal + '.0';
    // },
    // https://www.w3.org/TR/xmlschema-2/#double
    double: (value) => {
        const jsExponential = value.toExponential();
        const [jsMantisse, jsExponent] = jsExponential.split('e');
        // leading + must be removed for integer
        // https://www.w3.org/TR/xmlschema-2/#integer
        const exponent = jsExponent.replace(/\+/, '');
        // SPARQL test suite prefers trailing zero's
        const mantisse = jsMantisse.match(/\./)
            ? jsMantisse
            : jsMantisse + '.0';
        return `${mantisse}E${exponent}`;
    },
};
exports.NumericLiteral = NumericLiteral;
class BooleanLiteral extends Literal {
    constructor(typedValue, strValue) {
        super(typedValue, C.make(C.TypeURL.XSD_BOOLEAN), strValue);
        this.typedValue = typedValue;
        this.strValue = strValue;
    }
    coerceEBV() {
        return !!this.typedValue;
    }
}
exports.BooleanLiteral = BooleanLiteral;
class DateTimeLiteral extends Literal {
    // strValue is mandatory here because toISOString will always add
    // milliseconds, even if they were not present.
    constructor(typedValue, strValue) {
        super(typedValue, C.make(C.TypeURL.XSD_DATE_TIME), strValue);
        this.typedValue = typedValue;
        this.strValue = strValue;
    }
}
exports.DateTimeLiteral = DateTimeLiteral;
class LangStringLiteral extends Literal {
    constructor(typedValue, language) {
        super(typedValue, C.make(C.TypeURL.RDF_LANG_STRING), typedValue, language);
        this.typedValue = typedValue;
        this.language = language;
    }
    coerceEBV() {
        return this.strValue.length !== 0;
    }
}
exports.LangStringLiteral = LangStringLiteral;
// https://www.w3.org/TR/2004/REC-rdf-concepts-20040210/#dfn-plain-literal
// https://www.w3.org/TR/sparql11-query/#defn_SimpleLiteral
// https://www.w3.org/TR/sparql11-query/#func-strings
// This does not include language tagged literals
class StringLiteral extends Literal {
    constructor(typedValue) {
        super(typedValue, C.make(C.TypeURL.XSD_STRING), typedValue);
        this.typedValue = typedValue;
    }
    coerceEBV() {
        return this.strValue.length !== 0;
    }
}
exports.StringLiteral = StringLiteral;
/*
 * This class is used when a literal is parsed, and it's value is
 * an invalid lexical form for it's datatype. The spec defines value with
 * invalid lexical form are still valid terms, and as such we can not error
 * immediately. This class makes sure that the typedValue will remain undefined,
 * and the category 'nonlexical'. This way, only when operators apply to the
 * 'nonlexical' category, they will keep working, otherwise they will throw a
 * type error.
 * This seems to match the spec, except maybe for functions that accept
 * non-lexical values for their datatype.
 *
 * See:
 *  - https://www.w3.org/TR/xquery/#dt-type-error
 *  - https://www.w3.org/TR/rdf-concepts/#section-Literal-Value
 *  - https://www.w3.org/TR/xquery/#dt-ebv
 *  - ... some other more precise thing i can't find...
 */
class NonLexicalLiteral extends Literal {
    constructor(typedValue, typeURL, strValue, language) {
        super(typedValue, typeURL, strValue, language);
        this.typedValue = undefined;
        this.type = 'nonlexical';
        this.shouldBeCategory = C.type(typeURL.value);
    }
    coerceEBV() {
        const isNumericOrBool = C.PrimitiveNumericTypes.contains(this.shouldBeCategory)
            || this.shouldBeCategory === 'boolean';
        if (isNumericOrBool) {
            return false;
        }
        throw new Err.EBVCoercionError(this);
    }
}
exports.NonLexicalLiteral = NonLexicalLiteral;
//# sourceMappingURL=Term.js.map