import { List, Map } from 'immutable';
import * as E from '../expressions';
import * as C from '../util/Consts';
declare type Term = E.TermExpression;
export declare type OverloadMap = Map<List<ArgumentType>, E.SimpleApplication>;
export declare type ArgumentType = 'term' | E.TermType | C.Type;
export declare type OverloadedDefinition = {
    arity: number | number[];
    overloads: OverloadMap;
};
export declare abstract class BaseFunction<Operator> {
    operator: Operator;
    arity: number | number[];
    private overloads;
    constructor(operator: Operator, definition: OverloadedDefinition);
    /**
     * A function application works by monomorphing the function to a specific
     * instance depending on the runtime types. We then just apply this function
     * to the args.
     */
    apply: (args: E.TermExpression[]) => E.TermExpression;
    protected abstract handleInvalidTypes(args: Term[]): never;
    /**
     * We monomorph by checking the map of overloads for keys corresponding
     * to the runtime types. We start by checking for an implementation for the
     * most concrete types (integer, string, date, IRI), if we find none,
     * we consider their term types (literal, blank, IRI), and lastly we consider
     * all arguments as generic terms.
     *
     * Another option would be to populate the overloads with an implementation
     * for every concrete type when the function is generic over termtypes or
     * terms.
     */
    private monomorph;
}
/**
 * Varying kinds of functions take arguments of different types on which the
 * specific behaviour is dependant. Although their behaviour is often varying,
 * it is always relatively simple, and better suited for synced behaviour.
 * The types of their arguments are always terms, but might differ in
 * their term-type (eg: iri, literal),
 * their specific literal type (eg: string, integer),
 * their arity (see BNODE),
 * or even their specific numeric type (eg: integer, float).
 *
 * Examples include:
 *  - Arithmetic operations such as: *, -, /, +
 *  - Bool operators such as: =, !=, <=, <, ...
 *  - Functions such as: str, IRI
 *
 * See also: https://www.w3.org/TR/sparql11-query/#func-rdfTerms
 * and https://www.w3.org/TR/sparql11-query/#OperatorMapping
 */
export declare class RegularFunction extends BaseFunction<C.RegularOperator> {
    functionClass: 'regular';
    constructor(op: C.RegularOperator, definition: OverloadedDefinition);
    handleInvalidTypes(args: Term[]): never;
}
export declare class NamedFunction extends BaseFunction<C.NamedOperator> {
    functionClass: 'named';
    constructor(op: C.NamedOperator, definition: OverloadedDefinition);
    handleInvalidTypes(args: Term[]): never;
}
export declare class SpecialFunction {
    operator: C.SpecialOperator;
    functionClass: 'special';
    arity: number;
    applySync: E.SpecialApplicationSync;
    applyAsync: E.SpecialApplicationAsync;
    checkArity: (args: E.Expression[]) => boolean;
    constructor(operator: C.SpecialOperator, definition: SpecialDefinition);
}
export declare type SpecialDefinition = {
    arity: number;
    applyAsync: E.SpecialApplicationAsync;
    applySync: E.SpecialApplicationSync;
    checkArity?: (args: E.Expression[]) => boolean;
};
export declare function promote(left: C.PrimitiveNumericType, right: C.PrimitiveNumericType): C.PrimitiveNumericType;
export {};
