/**
 * These helpers provide a (albeit inflexible) DSL for writing function
 * definitions for the SPARQL functions.
 */
import { List, Record } from 'immutable';
import * as E from '../expressions';
import * as C from '../util/Consts';
import { ArgumentType, OverloadMap } from './Core';
declare type Term = E.TermExpression;
export declare function declare(): Builder;
export declare class Builder {
    private implementations;
    collect(): OverloadMap;
    log(): Builder;
    add(impl: Impl): Builder;
    set(argTypes: ArgumentType[], func: E.SimpleApplication): Builder;
    copy({ from, to }: {
        from: ArgumentType[];
        to: ArgumentType[];
    }): Builder;
    onUnary<T extends Term>(type: ArgumentType, op: (val: T) => Term): Builder;
    onUnaryTyped<T>(type: ArgumentType, op: (val: T) => Term): Builder;
    onBinary<L extends Term, R extends Term>(types: ArgumentType[], op: (left: L, right: R) => Term): Builder;
    onBinaryTyped<L, R>(types: ArgumentType[], op: (left: L, right: R) => Term): Builder;
    onTernaryTyped<A1, A2, A3>(types: ArgumentType[], op: (a1: A1, a2: A2, a3: A3) => Term): Builder;
    onTernary<A1 extends Term, A2 extends Term, A3 extends Term>(types: ArgumentType[], op: (a1: A1, a2: A2, a3: A3) => Term): Builder;
    onQuaternaryTyped<A1, A2, A3, A4>(types: ArgumentType[], op: (a1: A1, a2: A2, a3: A3, a4: A4) => Term): Builder;
    unimplemented(msg: string): Builder;
    onTerm1(op: (term: Term) => Term): Builder;
    onLiteral1<T>(op: (lit: E.Literal<T>) => Term): Builder;
    onBoolean1(op: (lit: E.BooleanLiteral) => Term): Builder;
    onBoolean1Typed(op: (lit: boolean) => Term): Builder;
    onString1(op: (lit: E.Literal<string>) => Term): Builder;
    onString1Typed(op: (lit: string) => Term): Builder;
    onLangString1(op: (lit: E.LangStringLiteral) => Term): Builder;
    onStringly1(op: (lit: E.Literal<string>) => Term): Builder;
    onStringly1Typed(op: (lit: string) => Term): Builder;
    onNumeric1(op: (val: E.NumericLiteral) => Term): Builder;
    onDateTime1(op: (date: E.DateTimeLiteral) => Term): Builder;
    /**
     * Arithmetic operators take 2 numeric arguments, and return a single numerical
     * value. The type of the return value is heavily dependant on the types of the
     * input arguments. In JS everything is a double, but in SPARQL it is not.
     *
     * {@link https://www.w3.org/TR/sparql11-query/#OperatorMapping}
     * {@link https://www.w3.org/TR/xpath-functions/#op.numeric}
     *
     * @param op the (simple) binary mathematical operator that
     */
    arithmetic(op: (left: number, right: number) => number): Builder;
    numberTest(test: (left: number, right: number) => boolean): Builder;
    stringTest(test: (left: string, right: string) => boolean): Builder;
    booleanTest(test: (left: boolean, right: boolean) => boolean): Builder;
    dateTimeTest(test: (left: Date, right: Date) => boolean): Builder;
    numeric(op: E.SimpleApplication): Builder;
    invalidLexicalForm(types: ArgumentType[], index: number): Builder;
    private chain;
}
/**
 * Immutable.js type definitions are pretty unsafe, and this is typo-prone work.
 * These helpers allow use to create OverloadMaps with more type-safety.
 * One entry in the OverloadMap is described by the record Impl;
 *
 * A list of Impl's then gets constructed into an Immutable.js Map.
 *
 * See:
 * https://medium.com/@alexxgent/enforcing-types-with-immutablejs-and-typescript-6ab980819b6a
 */
export declare type ImplType = {
    types: List<ArgumentType>;
    func: E.SimpleApplication;
};
declare const Impl_base: Record.Class;
export declare class Impl extends Impl_base {
    constructor(params: ImplType);
    get<T extends keyof ImplType>(value: T): ImplType[T];
    toPair(): [List<ArgumentType>, E.SimpleApplication];
}
export declare function map(implementations: Impl[]): OverloadMap;
export declare function bool(val: boolean): E.BooleanLiteral;
export declare function number(num: number, dt?: C.TypeURL): E.NumericLiteral;
export declare function numberFromString(str: string, dt?: C.TypeURL): E.NumericLiteral;
export declare function string(s: string): E.StringLiteral;
export declare function langString(s: string, lang: string): E.LangStringLiteral;
export declare function dateTime(date: Date, str: string): E.DateTimeLiteral;
export declare function log<T>(val: T, ...args: any[]): T;
export declare function typeCheckLit<T>(term: E.TermExpression, allowed: ArgumentType[], args: E.Expression[], op: C.Operator): E.Literal<T>;
export {};
