"use strict";
/**
 * These helpers provide a (albeit inflexible) DSL for writing function
 * definitions for the SPARQL functions.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const immutable_1 = require("immutable");
const E = require("../expressions");
const C = require("../util/Consts");
const Err = require("../util/Errors");
const Consts_1 = require("../util/Consts");
const Core_1 = require("./Core");
function declare() {
    return new Builder();
}
exports.declare = declare;
class Builder {
    constructor() {
        this.implementations = [];
    }
    collect() {
        return map(this.implementations);
    }
    log() {
        // tslint:disable-next-line:no-console
        console.log(this.implementations);
        return this;
    }
    add(impl) {
        this.implementations.push(impl);
        return this;
    }
    set(argTypes, func) {
        const types = immutable_1.List(argTypes);
        return this.add(new Impl({ types, func }));
    }
    copy({ from, to }) {
        const last = this.implementations.length - 1;
        const _from = immutable_1.List(from);
        for (let i = last; i >= 0; i--) {
            const impl = this.implementations[i];
            if (impl.get('types').equals(_from)) {
                return this.set(to, impl.get('func'));
            }
        }
        throw new Err.UnexpectedError('Tried to copy implementation, but types not found', { from, to });
    }
    onUnary(type, op) {
        return this.set([type], ([val]) => {
            return op(val);
        });
    }
    onUnaryTyped(type, op) {
        return this.set([type], ([val]) => {
            return op(val.typedValue);
        });
    }
    onBinary(types, op) {
        return this.set(types, ([left, right]) => {
            return op(left, right);
        });
    }
    onBinaryTyped(types, op) {
        return this.set(types, ([left, right]) => {
            return op(left.typedValue, right.typedValue);
        });
    }
    onTernaryTyped(types, op) {
        return this.set(types, ([a1, a2, a3]) => {
            return op(a1.typedValue, a2.typedValue, a3.typedValue);
        });
    }
    onTernary(types, op) {
        return this.set(types, ([a1, a2, a3]) => {
            return op(a1, a2, a3);
        });
    }
    onQuaternaryTyped(types, op) {
        return this.set(types, ([a1, a2, a3, a4]) => {
            return op(a1.typedValue, a2.typedValue, a3.typedValue, a4.typedValue);
        });
    }
    unimplemented(msg) {
        for (let arity = 0; arity <= 5; arity++) {
            const types = Array(arity).fill('term');
            const func = (_args) => { throw new Err.UnimplementedError(msg); };
            this.set(types, func);
        }
        return this;
    }
    onTerm1(op) {
        return this.set(['term'], ([term]) => op(term));
    }
    onLiteral1(op) {
        return this.set(['literal'], ([term]) => op(term));
    }
    onBoolean1(op) {
        return this
            .set(['boolean'], ([lit]) => op(lit));
    }
    onBoolean1Typed(op) {
        return this
            .set(['boolean'], ([lit]) => op(lit.typedValue));
    }
    onString1(op) {
        return this
            .set(['string'], ([lit]) => op(lit));
    }
    onString1Typed(op) {
        return this
            .set(['string'], ([lit]) => op(lit.typedValue));
    }
    onLangString1(op) {
        return this
            .set(['langString'], ([lit]) => op(lit));
    }
    onStringly1(op) {
        return this
            .set(['string'], ([lit]) => op(lit))
            .set(['langString'], ([lit]) => op(lit));
    }
    onStringly1Typed(op) {
        return this
            .set(['string'], ([lit]) => op(lit.typedValue))
            .set(['langString'], ([lit]) => op(lit.typedValue));
    }
    onNumeric1(op) {
        return this
            .set(['integer'], ([val]) => op(val))
            .set(['decimal'], ([val]) => op(val))
            .set(['float'], ([val]) => op(val))
            .set(['double'], ([val]) => op(val))
            .invalidLexicalForm(['nonlexical'], 1);
    }
    onDateTime1(op) {
        return this
            .set(['date'], ([val]) => op(val))
            .invalidLexicalForm(['nonlexical'], 1);
    }
    /**
     * Arithmetic operators take 2 numeric arguments, and return a single numerical
     * value. The type of the return value is heavily dependant on the types of the
     * input arguments. In JS everything is a double, but in SPARQL it is not.
     *
     * {@link https://www.w3.org/TR/sparql11-query/#OperatorMapping}
     * {@link https://www.w3.org/TR/xpath-functions/#op.numeric}
     *
     * @param op the (simple) binary mathematical operator that
     */
    arithmetic(op) {
        return this.numeric(([left, right]) => {
            const promotionType = Core_1.promote(left.type, right.type);
            const resultType = C.decategorize(promotionType);
            return number(op(left.typedValue, right.typedValue), resultType);
        });
    }
    numberTest(test) {
        return this.numeric(([left, right]) => {
            const result = test(left.typedValue, right.typedValue);
            return bool(result);
        });
    }
    stringTest(test) {
        return this
            .set(['string', 'string'], ([left, right]) => {
            const result = test(left.typedValue, right.typedValue);
            return bool(result);
        })
            .invalidLexicalForm(['nonlexical', 'string'], 1)
            .invalidLexicalForm(['string', 'nonlexical'], 2);
    }
    booleanTest(test) {
        return this
            .set(['boolean', 'boolean'], ([left, right]) => {
            const result = test(left.typedValue, right.typedValue);
            return bool(result);
        })
            .invalidLexicalForm(['nonlexical', 'boolean'], 1)
            .invalidLexicalForm(['boolean', 'nonlexical'], 2);
    }
    dateTimeTest(test) {
        return this
            .set(['date', 'date'], ([left, right]) => {
            const result = test(left.typedValue, right.typedValue);
            return bool(result);
        })
            .invalidLexicalForm(['nonlexical', 'date'], 1)
            .invalidLexicalForm(['date', 'nonlexical'], 2);
    }
    numeric(op) {
        return this
            .set(['integer', 'integer'], op)
            .set(['integer', 'decimal'], op)
            .set(['integer', 'float'], op)
            .set(['integer', 'double'], op)
            .invalidLexicalForm(['integer', 'nonlexical'], 2)
            .set(['decimal', 'integer'], op)
            .set(['decimal', 'decimal'], op)
            .set(['decimal', 'float'], op)
            .set(['decimal', 'double'], op)
            .invalidLexicalForm(['decimal', 'nonlexical'], 2)
            .set(['float', 'integer'], op)
            .set(['float', 'decimal'], op)
            .set(['float', 'float'], op)
            .set(['float', 'double'], op)
            .invalidLexicalForm(['float', 'nonlexical'], 2)
            .set(['double', 'integer'], op)
            .set(['double', 'decimal'], op)
            .set(['double', 'float'], op)
            .set(['double', 'double'], op)
            .invalidLexicalForm(['double', 'nonlexical'], 2)
            .invalidLexicalForm(['nonlexical', 'integer'], 1)
            .invalidLexicalForm(['nonlexical', 'decimal'], 1)
            .invalidLexicalForm(['nonlexical', 'float'], 1)
            .invalidLexicalForm(['nonlexical', 'double'], 1);
    }
    invalidLexicalForm(types, index) {
        return this.set(types, (args) => {
            throw new Err.InvalidLexicalForm(args[index - 1].toRDF());
        });
    }
    chain(impls) {
        this.implementations = this.implementations.concat(impls);
        return this;
    }
}
exports.Builder = Builder;
const implDefaults = {
    types: [],
    func() {
        const msg = 'Implementation not set yet declared as implemented';
        throw new Err.UnexpectedError(msg);
    },
};
class Impl extends immutable_1.Record(implDefaults) {
    constructor(params) { super(params); }
    get(value) {
        return super.get(value);
    }
    toPair() {
        return [this.get('types'), this.get('func')];
    }
}
exports.Impl = Impl;
function map(implementations) {
    const typeImplPair = implementations.map((i) => i.toPair());
    return immutable_1.Map(typeImplPair);
}
exports.map = map;
// ----------------------------------------------------------------------------
// Literal Construction helpers
// ----------------------------------------------------------------------------
function bool(val) {
    return new E.BooleanLiteral(val);
}
exports.bool = bool;
function number(num, dt) {
    return new E.NumericLiteral(num, C.make(dt || Consts_1.TypeURL.XSD_FLOAT), undefined);
}
exports.number = number;
function numberFromString(str, dt) {
    const num = Number(str);
    return new E.NumericLiteral(num, C.make(dt || Consts_1.TypeURL.XSD_FLOAT), undefined);
}
exports.numberFromString = numberFromString;
function string(s) {
    return new E.StringLiteral(s);
}
exports.string = string;
function langString(s, lang) {
    return new E.LangStringLiteral(s, lang);
}
exports.langString = langString;
function dateTime(date, str) {
    return new E.DateTimeLiteral(date, str);
}
exports.dateTime = dateTime;
// ----------------------------------------------------------------------------
// Util
// ----------------------------------------------------------------------------
// tslint:disable-next-line:no-any
function log(val, ...args) {
    // tslint:disable-next-line:no-console
    console.log(val, args);
    return val;
}
exports.log = log;
function typeCheckLit(term, allowed, args, op) {
    if (term.termType !== 'literal') {
        throw new Err.InvalidArgumentTypes(args, op);
    }
    // tslint:disable-next-line:no-any
    const lit = term;
    if (!allowed.includes(lit.type)) {
        throw new Err.InvalidArgumentTypes(args, op);
    }
    return lit;
}
exports.typeCheckLit = typeCheckLit;
//# sourceMappingURL=Helpers.js.map