"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const immutable_1 = require("immutable");
const Err = require("../util/Errors");
const Consts_1 = require("../util/Consts");
const Parsing_1 = require("../util/Parsing");
const Helpers_1 = require("./Helpers");
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// Begin definitions.
// ----------------------------------------------------------------------------
// XPath Constructor functions
// https://www.w3.org/TR/sparql11-query/#
// https://www.w3.org/TR/xpath-functions/#casting-from-primitive-to-primitive
// ----------------------------------------------------------------------------
const toString = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onTerm1((term) => Helpers_1.string(term.str()))
        .collect(),
};
const toFloat = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((val) => Helpers_1.number(val.typedValue))
        .onBoolean1Typed((val) => Helpers_1.number(val ? 1 : 0))
        .onUnary('string', (val) => {
        const result = Parsing_1.parseXSDFloat(val.str());
        if (!result) {
            throw new Err.CastError(val, Consts_1.TypeURL.XSD_FLOAT);
        }
        return Helpers_1.number(result);
    })
        .copy({ from: ['string'], to: ['nonlexical'] })
        .collect(),
};
const toDouble = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((val) => Helpers_1.number(val.typedValue, Consts_1.TypeURL.XSD_DOUBLE))
        .onBoolean1Typed((val) => Helpers_1.number(val ? 1 : 0, Consts_1.TypeURL.XSD_DOUBLE))
        .onUnary('string', (val) => {
        const result = Parsing_1.parseXSDFloat(val.str());
        if (!result) {
            throw new Err.CastError(val, Consts_1.TypeURL.XSD_DOUBLE);
        }
        return Helpers_1.number(result, Consts_1.TypeURL.XSD_DOUBLE);
    })
        .copy({ from: ['string'], to: ['nonlexical'] })
        .collect(),
};
const toDecimal = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((val) => {
        const result = Parsing_1.parseXSDDecimal(val.str());
        if (!result) {
            throw new Err.CastError(val, Consts_1.TypeURL.XSD_DECIMAL);
        }
        return Helpers_1.number(result, Consts_1.TypeURL.XSD_DECIMAL);
    })
        .copy({ from: ['integer'], to: ['string'] })
        .copy({ from: ['integer'], to: ['nonlexical'] })
        .onBoolean1Typed((val) => Helpers_1.number(val ? 1 : 0, Consts_1.TypeURL.XSD_DECIMAL))
        .collect(),
};
const toInteger = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onBoolean1Typed((val) => Helpers_1.number(val ? 1 : 0, Consts_1.TypeURL.XSD_INTEGER))
        .onNumeric1((val) => {
        const result = Parsing_1.parseXSDInteger(val.str());
        if (!result) {
            throw new Err.CastError(val, Consts_1.TypeURL.XSD_INTEGER);
        }
        return Helpers_1.number(result, Consts_1.TypeURL.XSD_INTEGER);
    })
        .copy({ from: ['integer'], to: ['string'] })
        .copy({ from: ['integer'], to: ['nonlexical'] })
        .collect(),
};
const toDatetime = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onUnary('date', (val) => val)
        .onUnary('string', (val) => {
        const date = new Date(val.str());
        if (isNaN(date.getTime())) {
            throw new Err.CastError(val, Consts_1.TypeURL.XSD_DATE_TIME);
        }
        return Helpers_1.dateTime(date, val.str());
    })
        .copy({ from: ['string'], to: ['nonlexical'] })
        .collect(),
};
const toBoolean = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((val) => Helpers_1.bool(val.coerceEBV()))
        .onUnary('boolean', (val) => val)
        .onUnary('string', (val) => {
        const str = val.str();
        if (str !== 'true' && str !== 'false') {
            throw new Err.CastError(val, Consts_1.TypeURL.XSD_BOOLEAN);
        }
        return Helpers_1.bool((str === 'true'));
    })
        .copy({ from: ['string'], to: ['nonlexical'] })
        .collect(),
};
// End definitions.
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
const _definitions = {
    // --------------------------------------------------------------------------
    // XPath Constructor functions
    // https://www.w3.org/TR/sparql11-query/#FunctionMapping
    // --------------------------------------------------------------------------
    [Consts_1.TypeURL.XSD_STRING]: toString,
    [Consts_1.TypeURL.XSD_FLOAT]: toFloat,
    [Consts_1.TypeURL.XSD_DOUBLE]: toDouble,
    [Consts_1.TypeURL.XSD_DECIMAL]: toDecimal,
    [Consts_1.TypeURL.XSD_INTEGER]: toInteger,
    [Consts_1.TypeURL.XSD_DATE_TIME]: toDatetime,
    [Consts_1.TypeURL.XSD_BOOLEAN]: toBoolean,
};
exports.namedDefinitions = immutable_1.Map(_definitions);
//# sourceMappingURL=NamedFunctions.js.map