"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const RDFDM = require("@rdfjs/data-model");
const hash = require("create-hash");
const uuid = require("uuid");
const immutable_1 = require("immutable");
const E = require("../expressions");
const C = require("../util/Consts");
const Err = require("../util/Errors");
const P = require("../util/Parsing");
const X = require("./XPathFunctions");
const Consts_1 = require("../util/Consts");
const Transformation_1 = require("../Transformation");
const Helpers_1 = require("./Helpers");
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// Begin definitions
// ----------------------------------------------------------------------------
// Operator Mapping
// https://www.w3.org/TR/sparql11-query/#OperatorMapping
// ----------------------------------------------------------------------------
const not = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onTerm1((val) => Helpers_1.bool(!val.coerceEBV()))
        .collect(),
};
const unaryPlus = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((val) => Helpers_1.number(val.typedValue, val.typeURL.value))
        .collect(),
};
const unaryMinus = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((val) => Helpers_1.number(-val.typedValue, val.typeURL.value))
        .collect(),
};
const multiplication = {
    arity: 2,
    overloads: Helpers_1.declare()
        .arithmetic((left, right) => left * right)
        .collect(),
};
const division = {
    arity: 2,
    overloads: Helpers_1.declare()
        .arithmetic((left, right) => left / right)
        .onBinaryTyped(['integer', 'integer'], (left, right) => {
        if (right === 0) {
            throw new Err.ExpressionError('Integer division by 0');
        }
        return Helpers_1.number(left / right, Consts_1.TypeURL.XSD_DECIMAL);
    })
        .collect(),
};
const addition = {
    arity: 2,
    overloads: Helpers_1.declare()
        .arithmetic((left, right) => left + right)
        .collect(),
};
const subtraction = {
    arity: 2,
    overloads: Helpers_1.declare()
        .arithmetic((left, right) => left - right)
        .collect(),
};
// https://www.w3.org/TR/sparql11-query/#func-RDFterm-equal
const equality = {
    arity: 2,
    overloads: Helpers_1.declare()
        .numberTest((left, right) => left === right)
        .stringTest((left, right) => left.localeCompare(right) === 0)
        .booleanTest((left, right) => left === right)
        .dateTimeTest((left, right) => left.getTime() === right.getTime())
        .set(['term', 'term'], ([left, right]) => Helpers_1.bool(RDFTermEqual(left, right)))
        .collect(),
};
function RDFTermEqual(_left, _right) {
    const left = _left.toRDF();
    const right = _right.toRDF();
    const val = left.equals(right);
    if ((left.termType === 'Literal') && (right.termType === 'Literal')) {
        throw new Err.RDFEqualTypeError([_left, _right]);
    }
    return val;
}
const inequality = {
    arity: 2,
    overloads: Helpers_1.declare()
        .numberTest((left, right) => left !== right)
        .stringTest((left, right) => left.localeCompare(right) !== 0)
        .booleanTest((left, right) => left !== right)
        .dateTimeTest((left, right) => left.getTime() !== right.getTime())
        .set(['term', 'term'], ([left, right]) => Helpers_1.bool(!RDFTermEqual(left, right)))
        .collect(),
};
const lesserThan = {
    arity: 2,
    overloads: Helpers_1.declare()
        .numberTest((left, right) => left < right)
        .stringTest((left, right) => left.localeCompare(right) === -1)
        .booleanTest((left, right) => left < right)
        .dateTimeTest((left, right) => left.getTime() < right.getTime())
        .collect(),
};
const greaterThan = {
    arity: 2,
    overloads: Helpers_1.declare()
        .numberTest((left, right) => left > right)
        .stringTest((left, right) => left.localeCompare(right) === 1)
        .booleanTest((left, right) => left > right)
        .dateTimeTest((left, right) => left.getTime() > right.getTime())
        .collect(),
};
const lesserThanEqual = {
    arity: 2,
    overloads: Helpers_1.declare()
        .numberTest((left, right) => left <= right)
        .stringTest((left, right) => left.localeCompare(right) !== 1)
        .booleanTest((left, right) => left <= right)
        .dateTimeTest((left, right) => left.getTime() <= right.getTime())
        .collect(),
};
const greaterThanEqual = {
    arity: 2,
    overloads: Helpers_1.declare()
        .numberTest((left, right) => left >= right)
        .stringTest((left, right) => left.localeCompare(right) !== -1)
        .booleanTest((left, right) => left >= right)
        .dateTimeTest((left, right) => left.getTime() >= right.getTime())
        .collect(),
};
// ----------------------------------------------------------------------------
// Functions on RDF Terms
// https://www.w3.org/TR/sparql11-query/#func-rdfTerms
// ----------------------------------------------------------------------------
const isIRI = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onTerm1((term) => Helpers_1.bool(term.termType === 'namedNode'))
        .collect(),
};
const isBlank = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onTerm1((term) => Helpers_1.bool(term.termType === 'blankNode'))
        .collect(),
};
const isLiteral = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onTerm1((term) => Helpers_1.bool(term.termType === 'literal'))
        .collect(),
};
const isNumeric = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((term) => Helpers_1.bool(true))
        .onTerm1((term) => Helpers_1.bool(false))
        .collect(),
};
const toString = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onTerm1((term) => Helpers_1.string(term.str()))
        .collect(),
};
const lang = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onLiteral1((lit) => Helpers_1.string(lit.language || ''))
        .collect(),
};
const datatype = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onLiteral1((lit) => new E.NamedNode(lit.typeURL.value))
        .collect(),
};
// See special operators
// const IRI = {};
// See special functions
// const BNODE = {};
const STRDT = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinary(['string', 'namedNode'], (str, iri) => {
        const lit = RDFDM.literal(str.typedValue, RDFDM.namedNode(iri.value));
        return Transformation_1.transformLiteral(lit);
    })
        .collect(),
};
const STRLANG = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (val, language) => new E.LangStringLiteral(val, language.toLowerCase()))
        .collect(),
};
const UUID = {
    arity: 0,
    overloads: Helpers_1.declare()
        .set([], () => new E.NamedNode(`urn:uuid:${uuid.v4()}`))
        .collect(),
};
const STRUUID = {
    arity: 0,
    overloads: Helpers_1.declare()
        .set([], () => Helpers_1.string(uuid.v4()))
        .collect(),
};
// ----------------------------------------------------------------------------
// Functions on strings
// https://www.w3.org/TR/sparql11-query/#func-forms
// ----------------------------------------------------------------------------
const STRLEN = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onStringly1((str) => Helpers_1.number(str.typedValue.length, Consts_1.TypeURL.XSD_INTEGER))
        .collect(),
};
const SUBSTR = {
    arity: [2, 3],
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'integer'], (source, startingLoc) => {
        return Helpers_1.string(source.substr(startingLoc - 1));
    })
        .onBinary(['langString', 'integer'], (source, startingLoc) => {
        const sub = source.typedValue.substr(startingLoc.typedValue - 1);
        return Helpers_1.langString(sub, source.language);
    })
        .onTernaryTyped(['string', 'integer', 'integer'], (source, startingLoc, length) => {
        return Helpers_1.string(source.substr(startingLoc - 1, length));
    })
        .onTernary(['langString', 'integer', 'integer'], (source, startingLoc, length) => {
        const sub = source.typedValue.substr(startingLoc.typedValue - 1, length.typedValue);
        return Helpers_1.langString(sub, source.language);
    })
        .collect(),
};
const UCASE = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((lit) => Helpers_1.string(lit.toUpperCase()))
        .onLangString1((lit) => Helpers_1.langString(lit.typedValue.toUpperCase(), lit.language))
        .collect(),
};
const LCASE = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((lit) => Helpers_1.string(lit.toLowerCase()))
        .onLangString1((lit) => Helpers_1.langString(lit.typedValue.toLowerCase(), lit.language))
        .collect(),
};
const STRSTARTS = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (arg1, arg2) => Helpers_1.bool(arg1.startsWith(arg2)))
        .onBinaryTyped(['langString', 'string'], (arg1, arg2) => Helpers_1.bool(arg1.includes(arg2)))
        .onBinary(['langString', 'langString'], (arg1, arg2) => {
        if (arg1.language !== arg2.language) {
            throw new Err.IncompatibleLanguageOperation(arg1, arg2);
        }
        return Helpers_1.bool(arg1.typedValue.startsWith(arg2.typedValue));
    })
        .collect(),
};
const STRENDS = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (arg1, arg2) => Helpers_1.bool(arg1.endsWith(arg2)))
        .onBinaryTyped(['langString', 'string'], (arg1, arg2) => Helpers_1.bool(arg1.includes(arg2)))
        .onBinary(['langString', 'langString'], (arg1, arg2) => {
        if (arg1.language !== arg2.language) {
            throw new Err.IncompatibleLanguageOperation(arg1, arg2);
        }
        return Helpers_1.bool(arg1.typedValue.endsWith(arg2.typedValue));
    })
        .collect(),
};
const CONTAINS = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (arg1, arg2) => Helpers_1.bool(arg1.includes(arg2)))
        .onBinaryTyped(['langString', 'string'], (arg1, arg2) => Helpers_1.bool(arg1.includes(arg2)))
        .onBinary(['langString', 'langString'], (arg1, arg2) => {
        if (arg1.language !== arg2.language) {
            throw new Err.IncompatibleLanguageOperation(arg1, arg2);
        }
        return Helpers_1.bool(arg1.typedValue.includes(arg2.typedValue));
    })
        .collect(),
};
const STRBEFORE = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (arg1, arg2) => Helpers_1.string(arg1.substr(0, arg1.indexOf(arg2))))
        .onBinary(['langString', 'string'], (arg1, arg2) => {
        const [a1, a2] = [arg1.typedValue, arg2.typedValue];
        const sub = arg1.typedValue.substr(0, a1.indexOf(a2));
        return (sub || !a2) ? Helpers_1.langString(sub, arg1.language) : Helpers_1.string(sub);
    })
        .onBinary(['langString', 'langString'], (arg1, arg2) => {
        if (arg1.language !== arg2.language) {
            throw new Err.IncompatibleLanguageOperation(arg1, arg2);
        }
        const [a1, a2] = [arg1.typedValue, arg2.typedValue];
        const sub = arg1.typedValue.substr(0, a1.indexOf(a2));
        return (sub || !a2) ? Helpers_1.langString(sub, arg1.language) : Helpers_1.string(sub);
    })
        .collect(),
};
const STRAFTER = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (arg1, arg2) => Helpers_1.string(arg1.substr(arg1.indexOf(arg2)).substr(arg2.length)))
        .onBinary(['langString', 'string'], (arg1, arg2) => {
        const [a1, a2] = [arg1.typedValue, arg2.typedValue];
        const sub = a1.substr(a1.indexOf(a2)).substr(a2.length);
        return (sub || !a2) ? Helpers_1.langString(sub, arg1.language) : Helpers_1.string(sub);
    })
        .onBinary(['langString', 'langString'], (arg1, arg2) => {
        if (arg1.language !== arg2.language) {
            throw new Err.IncompatibleLanguageOperation(arg1, arg2);
        }
        const [a1, a2] = [arg1.typedValue, arg2.typedValue];
        const sub = a1.substr(a1.indexOf(a2)).substr(a2.length);
        return (sub || !a2) ? Helpers_1.langString(sub, arg1.language) : Helpers_1.string(sub);
    })
        .collect(),
};
const ENCODE_FOR_URI = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onStringly1Typed((val) => Helpers_1.string(encodeURI(val))).collect(),
};
// See special operators
// const CONCAT = {}
const langmatches = {
    arity: 2,
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], (tag, range) => Helpers_1.bool(X.langMatches(tag, range))).collect(),
};
const regex2 = (text, pattern) => Helpers_1.bool(X.matches(text, pattern));
const regex3 = (text, pattern, flags) => Helpers_1.bool(X.matches(text, pattern, flags));
const REGEX = {
    arity: [2, 3],
    overloads: Helpers_1.declare()
        .onBinaryTyped(['string', 'string'], regex2)
        .onBinaryTyped(['langString', 'string'], regex2)
        .onTernaryTyped(['string', 'string', 'string'], regex3)
        .onTernaryTyped(['langString', 'string', 'string'], regex3)
        .collect(),
};
const REPLACE = {
    arity: [3, 4],
    overloads: Helpers_1.declare()
        .onTernaryTyped(['string', 'string', 'string'], (arg, pattern, replacement) => Helpers_1.string(X.replace(arg, pattern, replacement)))
        .set(['langString', 'string', 'string'], ([arg, pattern, replacement]) => {
        const result = X.replace(arg.typedValue, pattern.typedValue, replacement.typedValue);
        return Helpers_1.langString(result, arg.language);
    })
        .onQuaternaryTyped(['string', 'string', 'string', 'string'], (arg, pattern, replacement, flags) => Helpers_1.string(X.replace(arg, pattern, replacement, flags)))
        .set(['langString', 'string', 'string', 'string'], ([arg, pattern, replacement, flags]) => {
        const result = X.replace(arg.typedValue, pattern.typedValue, replacement.typedValue, flags.typedValue);
        return Helpers_1.langString(result, arg.language);
    })
        .collect(),
};
// ----------------------------------------------------------------------------
// Functions on numerics
// https://www.w3.org/TR/sparql11-query/#func-numerics
// ----------------------------------------------------------------------------
const abs = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((num) => Helpers_1.number(Math.abs(num.typedValue), num.typeURL.value))
        .collect(),
};
const round = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((num) => Helpers_1.number(Math.round(num.typedValue), num.typeURL.value))
        .collect(),
};
const ceil = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((num) => Helpers_1.number(Math.ceil(num.typedValue), num.typeURL.value))
        .collect(),
};
const floor = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onNumeric1((num) => Helpers_1.number(Math.floor(num.typedValue), num.typeURL.value))
        .collect(),
};
const rand = {
    arity: 0,
    overloads: Helpers_1.declare()
        .set([], () => Helpers_1.number(Math.random(), Consts_1.TypeURL.XSD_DOUBLE))
        .collect(),
};
// ----------------------------------------------------------------------------
// Functions on Dates and Times
// https://www.w3.org/TR/sparql11-query/#func-date-time
// ----------------------------------------------------------------------------
function parseDate(dateLit) {
    return P.parseXSDDateTime(dateLit.str());
}
// See special operators
// const now = {};
const year = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.number(Number(parseDate(date).year), Consts_1.TypeURL.XSD_INTEGER))
        .collect(),
};
const month = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.number(Number(parseDate(date).month), Consts_1.TypeURL.XSD_INTEGER))
        .collect(),
};
const day = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.number(Number(parseDate(date).day), Consts_1.TypeURL.XSD_INTEGER))
        .collect(),
};
const hours = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.number(Number(parseDate(date).hours), Consts_1.TypeURL.XSD_INTEGER))
        .collect(),
};
const minutes = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.number(Number(parseDate(date).minutes), Consts_1.TypeURL.XSD_INTEGER))
        .collect(),
};
const seconds = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.number(Number(parseDate(date).seconds), Consts_1.TypeURL.XSD_DECIMAL))
        .collect(),
};
const timezone = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => {
        const duration = X.formatDayTimeDuration(parseDate(date).timezone);
        if (!duration) {
            throw new Err.InvalidTimezoneCall(date.strValue);
        }
        return new E.Literal(duration, C.make(Consts_1.TypeURL.XSD_DAYTIME_DURATION), duration);
    })
        .collect(),
};
const tz = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onDateTime1((date) => Helpers_1.string(parseDate(date).timezone))
        .collect(),
};
// ----------------------------------------------------------------------------
// Hash functions
// https://www.w3.org/TR/sparql11-query/#func-hash
// ----------------------------------------------------------------------------
const MD5 = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((str) => Helpers_1.string(hash('md5').update(str).digest('hex')))
        .collect(),
};
const SHA1 = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((str) => Helpers_1.string(hash('sha1').update(str).digest('hex')))
        .collect(),
};
const SHA256 = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((str) => Helpers_1.string(hash('sha256').update(str).digest('hex')))
        .collect(),
};
const SHA384 = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((str) => Helpers_1.string(hash('sha384').update(str).digest('hex')))
        .collect(),
};
const SHA512 = {
    arity: 1,
    overloads: Helpers_1.declare()
        .onString1Typed((str) => Helpers_1.string(hash('sha512').update(str).digest('hex')))
        .collect(),
};
// End definitions.
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
/*
 * Collect all the definitions from above into an object
 */
const _definitions = {
    // --------------------------------------------------------------------------
    // Operator Mapping
    // https://www.w3.org/TR/sparql11-query/#OperatorMapping
    // --------------------------------------------------------------------------
    '!': not,
    'UPLUS': unaryPlus,
    'UMINUS': unaryMinus,
    '*': multiplication,
    '/': division,
    '+': addition,
    '-': subtraction,
    '=': equality,
    '!=': inequality,
    '<': lesserThan,
    '>': greaterThan,
    '<=': lesserThanEqual,
    '>=': greaterThanEqual,
    // --------------------------------------------------------------------------
    // Functions on RDF Terms
    // https://www.w3.org/TR/sparql11-query/#func-rdfTerms
    // --------------------------------------------------------------------------
    'isiri': isIRI,
    'isblank': isBlank,
    'isliteral': isLiteral,
    'isnumeric': isNumeric,
    'str': toString,
    'lang': lang,
    'datatype': datatype,
    // 'iri': IRI (see special operators),
    // 'uri': IRI (see special operators),
    // 'BNODE': BNODE (see special operators),
    'strdt': STRDT,
    'strlang': STRLANG,
    'uuid': UUID,
    'struuid': STRUUID,
    // --------------------------------------------------------------------------
    // Functions on strings
    // https://www.w3.org/TR/sparql11-query/#func-forms
    // --------------------------------------------------------------------------
    'strlen': STRLEN,
    'substr': SUBSTR,
    'ucase': UCASE,
    'lcase': LCASE,
    'strstarts': STRSTARTS,
    'strends': STRENDS,
    'contains': CONTAINS,
    'strbefore': STRBEFORE,
    'strafter': STRAFTER,
    'encode_for_uri': ENCODE_FOR_URI,
    // 'concat': CONCAT (see special operators)
    'langmatches': langmatches,
    'regex': REGEX,
    'replace': REPLACE,
    // --------------------------------------------------------------------------
    // Functions on numerics
    // https://www.w3.org/TR/sparql11-query/#func-numerics
    // --------------------------------------------------------------------------
    'abs': abs,
    'round': round,
    'ceil': ceil,
    'floor': floor,
    'rand': rand,
    // --------------------------------------------------------------------------
    // Functions on Dates and Times
    // https://www.w3.org/TR/sparql11-query/#func-date-time
    // --------------------------------------------------------------------------
    // 'now': now (see special operators),
    'year': year,
    'month': month,
    'day': day,
    'hours': hours,
    'minutes': minutes,
    'seconds': seconds,
    'timezone': timezone,
    'tz': tz,
    // --------------------------------------------------------------------------
    // Hash functions
    // https://www.w3.org/TR/sparql11-query/#func-hash
    // --------------------------------------------------------------------------
    'md5': MD5,
    'sha1': SHA1,
    'sha256': SHA256,
    'sha384': SHA384,
    'sha512': SHA512,
};
exports.definitions = immutable_1.Map(_definitions);
//# sourceMappingURL=RegularFunctions.js.map