"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const immutable_1 = require("immutable");
const URI = require("uri-js");
const uuid = require("uuid");
const E = require("../expressions");
const C = require("../util/Consts");
const Err = require("../util/Errors");
const Helpers_1 = require("./Helpers");
const index_1 = require("./index");
// ----------------------------------------------------------------------------
// Functional forms
// ----------------------------------------------------------------------------
function _bound({ args, mapping }) {
    const variable = args[0];
    if (variable.expressionType !== E.ExpressionType.Variable) {
        throw new Err.InvalidArgumentTypes(args, C.SpecialOperator.BOUND);
    }
    const val = mapping.has(variable.name) && !!mapping.get(variable.name);
    return Helpers_1.bool(val);
}
// BOUND ----------------------------------------------------------------------
const bound = {
    arity: 1,
    async applyAsync({ args, mapping }) {
        return _bound({ args, mapping });
    },
    applySync({ args, mapping }) {
        return _bound({ args, mapping });
    },
};
// IF -------------------------------------------------------------------------
const ifSPARQL = {
    arity: 3,
    async applyAsync({ args, mapping, evaluate }) {
        const valFirst = await evaluate(args[0], mapping);
        const ebv = valFirst.coerceEBV();
        return (ebv)
            ? evaluate(args[1], mapping)
            : evaluate(args[2], mapping);
    },
    applySync({ args, mapping, evaluate }) {
        const valFirst = evaluate(args[0], mapping);
        const ebv = valFirst.coerceEBV();
        return (ebv)
            ? evaluate(args[1], mapping)
            : evaluate(args[2], mapping);
    },
};
// COALESCE -------------------------------------------------------------------
const coalesce = {
    arity: Infinity,
    async applyAsync({ args, mapping, evaluate }) {
        const errors = [];
        for (const expr of args) {
            try {
                return await evaluate(expr, mapping);
            }
            catch (err) {
                errors.push(err);
            }
        }
        throw new Err.CoalesceError(errors);
    },
    applySync({ args, mapping, evaluate }) {
        const errors = [];
        for (const expr of args) {
            try {
                return evaluate(expr, mapping);
            }
            catch (err) {
                errors.push(err);
            }
        }
        throw new Err.CoalesceError(errors);
    },
};
// logical-or (||) ------------------------------------------------------------
// https://www.w3.org/TR/sparql11-query/#func-logical-or
const logicalOr = {
    arity: 2,
    async applyAsync({ args, mapping, evaluate }) {
        const [leftExpr, rightExpr] = args;
        try {
            const leftTerm = await evaluate(leftExpr, mapping);
            const left = leftTerm.coerceEBV();
            if (left) {
                return Helpers_1.bool(true);
            }
            const rightTerm = await evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            return Helpers_1.bool(right);
        }
        catch (leftErr) {
            const rightTerm = await evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            if (!right) {
                throw leftErr;
            }
            return Helpers_1.bool(true);
        }
    },
    applySync({ args, mapping, evaluate }) {
        const [leftExpr, rightExpr] = args;
        try {
            const leftTerm = evaluate(leftExpr, mapping);
            const left = leftTerm.coerceEBV();
            if (left) {
                return Helpers_1.bool(true);
            }
            const rightTerm = evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            return Helpers_1.bool(right);
        }
        catch (leftErr) {
            const rightTerm = evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            if (!right) {
                throw leftErr;
            }
            return Helpers_1.bool(true);
        }
    },
};
// logical-and (&&) -----------------------------------------------------------
// https://www.w3.org/TR/sparql11-query/#func-logical-and
const logicalAnd = {
    arity: 2,
    async applyAsync({ args, mapping, evaluate }) {
        const [leftExpr, rightExpr] = args;
        try {
            const leftTerm = await evaluate(leftExpr, mapping);
            const left = leftTerm.coerceEBV();
            if (!left) {
                return Helpers_1.bool(false);
            }
            const rightTerm = await evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            return Helpers_1.bool(right);
        }
        catch (leftErr) {
            const rightTerm = await evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            if (right) {
                throw leftErr;
            }
            return Helpers_1.bool(false);
        }
    },
    applySync({ args, mapping, evaluate }) {
        const [leftExpr, rightExpr] = args;
        try {
            const leftTerm = evaluate(leftExpr, mapping);
            const left = leftTerm.coerceEBV();
            if (!left) {
                return Helpers_1.bool(false);
            }
            const rightTerm = evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            return Helpers_1.bool(right);
        }
        catch (leftErr) {
            const rightTerm = evaluate(rightExpr, mapping);
            const right = rightTerm.coerceEBV();
            if (right) {
                throw leftErr;
            }
            return Helpers_1.bool(false);
        }
    },
};
// sameTerm -------------------------------------------------------------------
const sameTerm = {
    arity: 2,
    async applyAsync({ args, mapping, evaluate }) {
        const [leftExpr, rightExpr] = args.map((a) => evaluate(a, mapping));
        const left = await leftExpr;
        const right = await rightExpr;
        return Helpers_1.bool(left.toRDF().equals(right.toRDF()));
    },
    applySync({ args, mapping, evaluate }) {
        const [left, right] = args.map((a) => evaluate(a, mapping));
        return Helpers_1.bool(left.toRDF().equals(right.toRDF()));
    },
};
// IN -------------------------------------------------------------------------
const inSPARQL = {
    arity: Infinity,
    checkArity(args) { return args.length >= 1; },
    async applyAsync({ args, mapping, evaluate, context }) {
        const [leftExpr, ...remaining] = args;
        const left = await evaluate(leftExpr, mapping);
        return inRecursiveAsync(left, { args: remaining, mapping, evaluate, context }, []);
    },
    applySync({ args, mapping, evaluate, context }) {
        const [leftExpr, ...remaining] = args;
        const left = evaluate(leftExpr, mapping);
        return inRecursiveSync(left, { args: remaining, mapping, evaluate, context }, []);
    },
};
async function inRecursiveAsync(needle, { args, mapping, evaluate, context }, results) {
    if (args.length === 0) {
        const noErrors = results.every((v) => !v);
        return (noErrors) ? Helpers_1.bool(false) : Promise.reject(new Err.InError(results));
    }
    try {
        const next = await evaluate(args.shift(), mapping);
        const isEqual = index_1.regularFunctions.get(C.RegularOperator.EQUAL);
        if (isEqual.apply([needle, next]).typedValue === true) {
            return Helpers_1.bool(true);
        }
        else {
            return inRecursiveAsync(needle, { args, mapping, evaluate, context }, [...results, false]);
        }
    }
    catch (err) {
        return inRecursiveAsync(needle, { args, mapping, evaluate, context }, [...results, err]);
    }
}
function inRecursiveSync(needle, { args, mapping, evaluate, context }, results) {
    if (args.length === 0) {
        const noErrors = results.every((v) => !v);
        if (noErrors) {
            Helpers_1.bool(false);
        }
        else {
            throw new Err.InError(results);
        }
    }
    try {
        const next = evaluate(args.shift(), mapping);
        const isEqual = index_1.regularFunctions.get(C.RegularOperator.EQUAL);
        if (isEqual.apply([needle, next]).typedValue === true) {
            return Helpers_1.bool(true);
        }
        else {
            return inRecursiveSync(needle, { args, mapping, evaluate, context }, [...results, false]);
        }
    }
    catch (err) {
        return inRecursiveSync(needle, { args, mapping, evaluate, context }, [...results, err]);
    }
}
// NOT IN ---------------------------------------------------------------------
const notInSPARQL = {
    arity: Infinity,
    checkArity(args) { return args.length >= 1; },
    async applyAsync(context) {
        const _in = index_1.specialFunctions.get(C.SpecialOperator.IN);
        const isIn = await _in.applyAsync(context);
        return Helpers_1.bool(!isIn.typedValue);
    },
    applySync(context) {
        const _in = index_1.specialFunctions.get(C.SpecialOperator.IN);
        const isIn = _in.applySync(context);
        return Helpers_1.bool(!isIn.typedValue);
    },
};
// ----------------------------------------------------------------------------
// Annoying functions
// ----------------------------------------------------------------------------
// CONCAT
const concat = {
    arity: Infinity,
    async applyAsync({ args, evaluate, mapping }) {
        const pLits = args
            .map(async (expr) => evaluate(expr, mapping))
            .map(async (pTerm) => Helpers_1.typeCheckLit(await pTerm, ['string', 'langString'], args, C.SpecialOperator.CONCAT));
        const lits = await Promise.all(pLits);
        const strings = lits.map((lit) => lit.typedValue);
        const joined = strings.join('');
        const lang = langAllEqual(lits) ? lits[0].language : undefined;
        return (lang) ? Helpers_1.langString(joined, lang) : Helpers_1.string(joined);
    },
    applySync({ args, evaluate, mapping }) {
        const lits = args
            .map((expr) => evaluate(expr, mapping))
            .map((pTerm) => Helpers_1.typeCheckLit(pTerm, ['string', 'langString'], args, C.SpecialOperator.CONCAT));
        const strings = lits.map((lit) => lit.typedValue);
        const joined = strings.join('');
        const lang = langAllEqual(lits) ? lits[0].language : undefined;
        return (lang) ? Helpers_1.langString(joined, lang) : Helpers_1.string(joined);
    },
};
function langAllEqual(lits) {
    return lits.length > 0 && lits.every((lit) => lit.language === lits[0].language);
}
// ----------------------------------------------------------------------------
// Context dependant functions
// ----------------------------------------------------------------------------
const now = {
    arity: 0,
    async applyAsync({ context }) {
        return new E.DateTimeLiteral(context.now, context.now.toUTCString());
    },
    applySync({ context }) {
        return new E.DateTimeLiteral(context.now, context.now.toUTCString());
    },
};
// https://www.w3.org/TR/sparql11-query/#func-iri
const IRI = {
    arity: 1,
    async applyAsync({ args, evaluate, mapping, context }) {
        const input = await evaluate(args[0], mapping);
        return IRI_(input, context.baseIRI, args);
    },
    applySync({ args, evaluate, mapping, context }) {
        const input = evaluate(args[0], mapping);
        return IRI_(input, context.baseIRI, args);
    },
};
function IRI_(input, baseIRI, args) {
    const lit = (input.termType !== 'namedNode')
        ? Helpers_1.typeCheckLit(input, ['string'], args, C.SpecialOperator.IRI)
        : input;
    const iri = URI.resolve(baseIRI || '', lit.str());
    return new E.NamedNode(iri);
}
// https://www.w3.org/TR/sparql11-query/#func-bnode
// id has to be distinct over all id's in dataset
const BNODE = {
    arity: Infinity,
    checkArity(args) { return args.length === 0 || args.length === 1; },
    async applyAsync({ args, evaluate, mapping, context }) {
        const input = (args.length === 1)
            ? await evaluate(args[0], mapping)
            : undefined;
        const strInput = (input)
            ? Helpers_1.typeCheckLit(input, ['string'], args, C.SpecialOperator.BNODE).str()
            : undefined;
        if (context.bnode) {
            const bnode = await context.bnode(strInput);
            return new E.BlankNode(bnode.value);
        }
        return BNODE_(strInput);
    },
    applySync({ args, evaluate, mapping, context }) {
        const input = (args.length === 1)
            ? evaluate(args[0], mapping)
            : undefined;
        const strInput = (input)
            ? Helpers_1.typeCheckLit(input, ['string'], args, C.SpecialOperator.BNODE).str()
            : undefined;
        if (context.bnode) {
            const bnode = context.bnode(strInput);
            return new E.BlankNode(bnode.value);
        }
        return BNODE_(strInput);
    },
};
function BNODE_(input) {
    return new E.BlankNode('blank_' + uuid.v4());
}
const _specialDefinitions = {
    // --------------------------------------------------------------------------
    // Functional Forms
    // https://www.w3.org/TR/sparql11-query/#func-forms
    // --------------------------------------------------------------------------
    'bound': bound,
    'if': ifSPARQL,
    'coalesce': coalesce,
    '&&': logicalAnd,
    '||': logicalOr,
    'sameterm': sameTerm,
    'in': inSPARQL,
    'notin': notInSPARQL,
    // Annoying functions
    'concat': concat,
    // Context dependent functions
    'now': now,
    'iri': IRI,
    'uri': IRI,
    'BNODE': BNODE,
};
exports.specialDefinitions = immutable_1.Map(_specialDefinitions);
//# sourceMappingURL=SpecialFunctions.js.map