"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * This class of error will be thrown when an expression errors.
 * Various reasons this could happen are:
 *   - invalid types for the given operator
 *   - unbound variables
 *   - invalid lexical forms
 *   - ...
 *
 * The distinction is made so that one can catch this specific type
 * and handle it accordingly to the SPARQL spec (relevant for e.g. FILTER, EXTEND),
 * while others (programming errors) can be re-thrown.
 *
 * @see isExpressionError
 */
class ExpressionError extends Error {
}
exports.ExpressionError = ExpressionError;
/**
 * Checks whether a given error is an {@link ExpressionError}.
 * Also useful for mocking in tests for covering all branches.
 *
 * @see ExpressionError
 */
function isExpressionError(error) {
    return error instanceof ExpressionError;
}
exports.isExpressionError = isExpressionError;
/**
 * A literal has an invalid lexical form for the datatype it is accompanied by.
 * This error is only thrown when the term is as function argument that requires
 * a valid lexical form.
 */
class InvalidLexicalForm extends ExpressionError {
    constructor(arg) {
        super(`Invalid lexical form '${pp(arg)}'`);
        this.arg = arg;
    }
}
exports.InvalidLexicalForm = InvalidLexicalForm;
/**
 * A variable in the expression was not bound.
 */
class UnboundVariableError extends ExpressionError {
    constructor(variable, bindings) {
        super(`Unbound variable '${pp(variable)}'`);
        this.variable = variable;
        this.bindings = bindings;
    }
}
exports.UnboundVariableError = UnboundVariableError;
/**
 * An invalid term was being coerced to an Effective Boolean Value.
 *
 * See the {@link https://www.w3.org/TR/sparql11-query/#ebv | SPARQL docs}
 * on EBVs.
 */
class EBVCoercionError extends ExpressionError {
    constructor(arg) {
        super(`Cannot coerce term to EBV '${pp(arg)}'`);
        this.arg = arg;
    }
}
exports.EBVCoercionError = EBVCoercionError;
/**
 * An equality test was done on literals with unsupported datatypes.
 *
 * See {@link https://www.w3.org/TR/sparql11-query/#func-RDFterm-equal | term equality spec}.
 */
class RDFEqualTypeError extends ExpressionError {
    constructor(args) {
        super('Equality test for literals with unsupported datatypes');
        this.args = args;
    }
}
exports.RDFEqualTypeError = RDFEqualTypeError;
/**
 * All the expressions in a COALESCE call threw errors.
 */
class CoalesceError extends ExpressionError {
    constructor(errors) {
        super('All COALESCE arguments threw errors');
        this.errors = errors;
    }
}
exports.CoalesceError = CoalesceError;
/**
 * No arguments to an IN call where equal, and at least one threw an error.
 */
class InError extends ExpressionError {
    constructor(errors) {
        super('Some argument to IN errorred and none where equal. ' +
            errors.map((err) => `(${err.toString()}) `).join('and '));
        this.errors = errors;
    }
}
exports.InError = InError;
/**
 * Literals were passed to an operator that doesn't support their datatypes.
 */
class InvalidArgumentTypes extends ExpressionError {
    constructor(args, op) {
        super(`Argument types not valid for operator: '${pp(op)}' with '${pp(args)}`);
        this.args = args;
        this.op = op;
    }
}
exports.InvalidArgumentTypes = InvalidArgumentTypes;
/**
 * An invalid typecast happened.
 */
class CastError extends ExpressionError {
    constructor(arg, cast) {
        super(`Invalid cast: '${pp(arg)}' to '${pp(cast)}'`);
        this.arg = arg;
    }
}
exports.CastError = CastError;
class InvalidTimezoneCall extends ExpressionError {
    constructor(dateString) {
        super(`TIMEZONE call on ${dateString} which has no timezone`);
        this.dateString = dateString;
    }
}
exports.InvalidTimezoneCall = InvalidTimezoneCall;
class IncompatibleLanguageOperation extends ExpressionError {
    constructor(arg1, arg2) {
        super(`Operation on incompatible language literals '${pp(arg1)}' and '${pp(arg2)}'`);
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
}
exports.IncompatibleLanguageOperation = IncompatibleLanguageOperation;
class EmptyAggregateError extends ExpressionError {
    constructor() {
        super('Empty aggregate expression');
    }
}
exports.EmptyAggregateError = EmptyAggregateError;
// Non Expression Errors ------------------------------------------------------
/**
 * An error that arises when we detect a 'should-be-impossible' state.
 * Given that this error is thrown, it clearly wasn't impossible, and some
 * mistake has been made.
 */
class UnexpectedError extends Error {
    constructor(message, payload) {
        super(`Programmer Error '${message}'`);
        this.payload = payload;
    }
}
exports.UnexpectedError = UnexpectedError;
/**
 * An Error that signals a feature or function is yet unimplemented.
 */
class UnimplementedError extends Error {
    constructor(feature) {
        super(`Unimplemented feature '${feature}!'`);
    }
}
exports.UnimplementedError = UnimplementedError;
class InvalidArity extends Error {
    constructor(args, op) {
        super(`The amount of args don't match the arity of the operator '${pp(op)}'.`);
        this.args = args;
        this.op = op;
    }
}
exports.InvalidArity = InvalidArity;
class InvalidExpression extends Error {
    constructor(expr) {
        super(`Invalid SPARQL Expression '${pp(expr)}'`);
    }
}
exports.InvalidExpression = InvalidExpression;
class InvalidExpressionType extends Error {
    constructor(expr) {
        super(`Invalid expression type for SPARQL Expression '${pp(expr)}'`);
        this.expr = expr;
    }
}
exports.InvalidExpressionType = InvalidExpressionType;
class InvalidTermType extends Error {
    constructor(term) {
        super(`Invalid term type for term '${pp(term)}'`);
        this.term = term;
    }
}
exports.InvalidTermType = InvalidTermType;
class UnknownOperator extends Error {
    constructor(name) {
        super(`Unknown operator: '${pp(name)}`);
    }
}
exports.UnknownOperator = UnknownOperator;
class UnknownNamedOperator extends Error {
    constructor(name) {
        super(`Unknown named operator: '${pp(name)}'`);
    }
}
exports.UnknownNamedOperator = UnknownNamedOperator;
class NoAggregator extends Error {
    constructor(name) {
        super(`Aggregate expression ${pp(name)} found, but no aggregate hook provided.`);
    }
}
exports.NoAggregator = NoAggregator;
class NoExistenceHook extends Error {
    constructor() {
        super('EXISTS found, but no existence hook provided.');
    }
}
exports.NoExistenceHook = NoExistenceHook;
function pp(o) {
    return JSON.stringify(o);
}
//# sourceMappingURL=Errors.js.map