/// <reference types="node" />
import * as RDF from "rdf-js";
/**
 * Parser for the SPARQL 1.1 Query Results JSON format.
 * @see https://www.w3.org/TR/sparql11-results-json/
 */
export declare class SparqlJsonParser {
    private readonly dataFactory;
    private readonly prefixVariableQuestionMark?;
    constructor(settings?: ISettings);
    /**
     * Convert a SPARQL JSON bindings response to an array of bindings objects.
     * @param sparqlResponse A SPARQL JSON response.
     * @return {IBindings[]} An array of bindings.
     */
    parseJsonResults(sparqlResponse: any): IBindings[];
    /**
     * Convert a SPARQL JSON bindings response stream to a stream of bindings objects.
     *
     * The bindings stream will emit a 'variables' event that will contain
     * the array of variables (as RDF.Variable[]), as defined in the response head.
     *
     * @param {NodeJS.ReadableStream} sparqlResponseStream A SPARQL JSON response stream.
     * @return {NodeJS.ReadableStream} A stream of bindings.
     */
    parseJsonResultsStream(sparqlResponseStream: NodeJS.ReadableStream): NodeJS.ReadableStream;
    /**
     * Convert a SPARQL JSON result binding to a bindings object.
     * @param rawBindings A SPARQL JSON result binding.
     * @return {IBindings} A bindings object.
     */
    parseJsonBindings(rawBindings: any): IBindings;
    /**
     * Convert a SPARQL JSON boolean response to a boolean.
     * This will throw an error if the given reponse was not a valid boolean response.
     * @param sparqlResponse A SPARQL JSON response.
     * @return {IBindings[]} An array of bindings.
     */
    parseJsonBoolean(sparqlResponse: any): boolean;
    /**
     * Convert a SPARQL JSON boolean response stream to a promise resolving to a boolean.
     * This will reject if the given reponse was not a valid boolean response.
     * @param {NodeJS.ReadableStream} sparqlResponseStream A SPARQL JSON response stream.
     * @return {NodeJS.ReadableStream} A stream of bindings.
     */
    parseJsonBooleanStream(sparqlResponseStream: NodeJS.ReadableStream): Promise<boolean>;
}
/**
 * Constructor settings object interface for {@link SparqlJsonParser}.
 */
export interface ISettings {
    /**
     * A custom datafactory.
     */
    dataFactory?: RDF.DataFactory;
    /**
     * If variable names should be prefixed with a quotation mark.
     */
    prefixVariableQuestionMark?: boolean;
}
/**
 * A bindings object.
 */
export interface IBindings {
    [key: string]: RDF.Term;
}
