"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const DefaultDataFactory = require("@rdfjs/data-model");
const SparqlJsonBindingsTransformer_1 = require("./SparqlJsonBindingsTransformer");
/**
 * Parser for the SPARQL 1.1 Query Results JSON format.
 * @see https://www.w3.org/TR/sparql11-results-json/
 */
class SparqlJsonParser {
    constructor(settings) {
        settings = settings || {};
        this.dataFactory = settings.dataFactory || DefaultDataFactory;
        this.prefixVariableQuestionMark = !!settings.prefixVariableQuestionMark;
    }
    /**
     * Convert a SPARQL JSON bindings response to an array of bindings objects.
     * @param sparqlResponse A SPARQL JSON response.
     * @return {IBindings[]} An array of bindings.
     */
    parseJsonResults(sparqlResponse) {
        return sparqlResponse.results.bindings.map((rawBindings) => this.parseJsonBindings(rawBindings));
    }
    /**
     * Convert a SPARQL JSON bindings response stream to a stream of bindings objects.
     *
     * The bindings stream will emit a 'variables' event that will contain
     * the array of variables (as RDF.Variable[]), as defined in the response head.
     *
     * @param {NodeJS.ReadableStream} sparqlResponseStream A SPARQL JSON response stream.
     * @return {NodeJS.ReadableStream} A stream of bindings.
     */
    parseJsonResultsStream(sparqlResponseStream) {
        sparqlResponseStream.on('error', (error) => resultStream.emit('error', error));
        const variables = [];
        sparqlResponseStream
            .pipe(require('JSONStream').parse('head.vars.*'))
            .on('data', (variable) => variables.push(this.dataFactory.variable(variable)))
            .on('end', () => resultStream.emit('variables', variables));
        const resultStream = sparqlResponseStream
            .pipe(require('JSONStream').parse('results.bindings.*'))
            .pipe(new SparqlJsonBindingsTransformer_1.SparqlJsonBindingsTransformer(this));
        return resultStream;
    }
    /**
     * Convert a SPARQL JSON result binding to a bindings object.
     * @param rawBindings A SPARQL JSON result binding.
     * @return {IBindings} A bindings object.
     */
    parseJsonBindings(rawBindings) {
        const bindings = {};
        for (const key in rawBindings) {
            const rawValue = rawBindings[key];
            let value = null;
            switch (rawValue.type) {
                case 'bnode':
                    value = this.dataFactory.blankNode(rawValue.value);
                    break;
                case 'literal':
                    if (rawValue['xml:lang']) {
                        value = this.dataFactory.literal(rawValue.value, rawValue['xml:lang']);
                    }
                    else if (rawValue.datatype) {
                        value = this.dataFactory.literal(rawValue.value, this.dataFactory.namedNode(rawValue.datatype));
                    }
                    else {
                        value = this.dataFactory.literal(rawValue.value);
                    }
                    break;
                case 'typed-literal':
                    // Virtuoso uses this non-spec-compliant way of defining typed literals
                    value = this.dataFactory.literal(rawValue.value, this.dataFactory.namedNode(rawValue.datatype));
                    break;
                default:
                    value = this.dataFactory.namedNode(rawValue.value);
                    break;
            }
            bindings[this.prefixVariableQuestionMark ? ('?' + key) : key] = value;
        }
        return bindings;
    }
    /**
     * Convert a SPARQL JSON boolean response to a boolean.
     * This will throw an error if the given reponse was not a valid boolean response.
     * @param sparqlResponse A SPARQL JSON response.
     * @return {IBindings[]} An array of bindings.
     */
    parseJsonBoolean(sparqlResponse) {
        if ('boolean' in sparqlResponse) {
            return sparqlResponse.boolean;
        }
        throw new Error('No valid ASK response was found.');
    }
    /**
     * Convert a SPARQL JSON boolean response stream to a promise resolving to a boolean.
     * This will reject if the given reponse was not a valid boolean response.
     * @param {NodeJS.ReadableStream} sparqlResponseStream A SPARQL JSON response stream.
     * @return {NodeJS.ReadableStream} A stream of bindings.
     */
    parseJsonBooleanStream(sparqlResponseStream) {
        return new Promise((resolve, reject) => {
            sparqlResponseStream.on('error', reject);
            sparqlResponseStream.pipe(require('JSONStream').parse('boolean'))
                .on('data', resolve)
                .on('end', () => reject(new Error('No valid ASK response was found.')));
        });
    }
}
exports.SparqlJsonParser = SparqlJsonParser;
//# sourceMappingURL=SparqlJsonParser.js.map