/// <reference types="node" />
import * as RDF from "rdf-js";
/**
 * Parser for the SPARQL Query Results XML format.
 * @see https://www.w3.org/TR/rdf-sparql-XMLres/
 */
export declare class SparqlXmlParser {
    private readonly dataFactory;
    private readonly prefixVariableQuestionMark?;
    constructor(settings?: ISettings);
    /**
     * Convert a SPARQL XML bindings response stream to a stream of bindings objects.
     *
     * The bindings stream will emit a 'variables' event that will contain
     * the array of variables (as RDF.Variable[]), as defined in the response head.
     *
     * @param {NodeJS.ReadableStream} sparqlResponseStream A SPARQL XML response stream.
     * @return {NodeJS.ReadableStream} A stream of bindings.
     */
    parseXmlResultsStream(sparqlResponseStream: NodeJS.ReadableStream): NodeJS.ReadableStream;
    /**
     * Convert a SPARQL XML result binding to a bindings object.
     * @param rawBindings A SPARQL XML result binding.
     * @return {IBindings} A bindings object.
     */
    parseXmlBindings(rawBindings: any): IBindings;
    /**
     * Convert a SPARQL XML boolean response stream to a promise resolving to a boolean.
     * This will reject if the given response was not a valid boolean response.
     * @param {NodeJS.ReadableStream} sparqlResponseStream A SPARQL XML response stream.
     * @return {NodeJS.ReadableStream} A stream of bindings.
     */
    parseXmlBooleanStream(sparqlResponseStream: NodeJS.ReadableStream): Promise<boolean>;
}
/**
 * Constructor settings object interface for {@link SparqlXmlParser}.
 */
export interface ISettings {
    /**
     * A custom datafactory.
     */
    dataFactory?: RDF.DataFactory;
    /**
     * If variable names should be prefixed with a quotation mark.
     */
    prefixVariableQuestionMark?: boolean;
}
/**
 * A bindings object.
 */
export interface IBindings {
    [key: string]: RDF.Term;
}
