import _Promise from 'babel-runtime/core-js/promise';
import logFactory from '../utils/logger';
var log = logFactory('splitio-client');
import evaluator from '../engine/evaluator';
import ImpressionTracker from '../trackers/impression';
import ImpressionsTracker from '../trackers/impressions';
import tracker from '../utils/timeTracker';
import thenable from '../utils/promise/thenable';
import { matching, bucketing } from '../utils/key/factory';
/* asynchronous validations that live on the client. */
import { validateSplitExistance, validateTrafficTypeExistance } from '../utils/inputValidation';
import { SDK_NOT_READY } from '../utils/labels';
import { CONTROL } from '../utils/constants';

function queueEventsCallback(_ref, tracked) {
  var eventTypeId = _ref.eventTypeId,
      trafficTypeName = _ref.trafficTypeName,
      key = _ref.key,
      value = _ref.value,
      timestamp = _ref.timestamp,
      properties = _ref.properties;

  // Logging every prop would be too much.
  var msg = 'event of type "' + eventTypeId + '" for traffic type "' + trafficTypeName + '". Key: ' + key + '. Value: ' + value + '. Timestamp: ' + timestamp + '. ' + (properties ? 'With properties.' : 'With no properties.');

  if (tracked) {
    log.info('Successfully qeued ' + msg);
  } else {
    log.warn('Failed to queue ' + msg);
  }

  return tracked;
}

function ClientFactory(context) {
  var storage = context.get(context.constants.STORAGE);
  var metricCollectors = context.get(context.constants.COLLECTORS);
  var impressionTracker = ImpressionTracker(context);
  var impressionsTracker = ImpressionsTracker(context);

  function getTreatment(key, splitName, attributes) {
    var withConfig = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

    var taskToBeTracked = tracker.TaskNames[withConfig ? 'SDK_GET_TREATMENT_WITH_CONFIG' : 'SDK_GET_TREATMENT'];
    var stopLatencyTracker = tracker.start(taskToBeTracked, metricCollectors);
    var evaluation = evaluator(key, splitName, attributes, storage);

    if (thenable(evaluation)) {
      return evaluation.then(function (res) {
        return processEvaluation(res, splitName, key, attributes, stopLatencyTracker, impressionTracker.track, withConfig, 'getTreatment' + (withConfig ? 'withConfig' : ''));
      });
    } else {
      return processEvaluation(evaluation, splitName, key, attributes, stopLatencyTracker, impressionTracker.track, withConfig, 'getTreatment' + (withConfig ? 'withConfig' : ''));
    }
  }

  function getTreatmentWithConfig(key, splitName, attributes) {
    return getTreatment(key, splitName, attributes, true);
  }

  function getTreatments(key, splitNames, attributes) {
    var withConfig = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

    var taskToBeTracked = tracker.TaskNames[withConfig ? 'SDK_GET_TREATMENTS_WITH_CONFIG' : 'SDK_GET_TREATMENTS'];
    var stopLatencyTracker = tracker.start(taskToBeTracked, metricCollectors);
    var results = {};
    var thenables = [];
    var i = void 0;

    var _loop = function _loop() {
      var splitName = splitNames[i];
      var evaluation = evaluator(key, splitName, attributes, storage);

      if (thenable(evaluation)) {
        // If treatment returns a promise as it is being evaluated, save promise for progress tracking.
        thenables.push(evaluation);
        evaluation.then(function (res) {
          // set the treatment on the cb;
          results[splitName] = processEvaluation(res, splitName, key, attributes, false, impressionsTracker.queue, withConfig, 'getTreatments' + (withConfig ? 'withConfig' : ''));
        });
      } else {
        results[splitName] = processEvaluation(evaluation, splitName, key, attributes, false, impressionsTracker.queue, withConfig, 'getTreatments' + (withConfig ? 'withConfig' : ''));
      }
    };

    for (i = 0; i < splitNames.length; i++) {
      _loop();
    }

    var wrapUp = function wrapUp() {
      impressionsTracker.track();
      stopLatencyTracker();
      // After all treatments are resolved, we return the mapping object.
      return results;
    };

    if (thenables.length) {
      return _Promise.all(thenables).then(wrapUp);
    } else {
      return wrapUp();
    }
  }

  function getTreatmentsWithConfig(key, splitNames, attributes) {
    return getTreatments(key, splitNames, attributes, true);
  }

  // Internal function
  function processEvaluation(evaluation, splitName, key, attributes) {
    var stopLatencyTracker = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : false;
    var impressionsTracker = arguments[5];
    var withConfig = arguments[6];
    var invokingMethodName = arguments[7];

    var isSdkReady = context.get(context.constants.READY, true);
    var matchingKey = matching(key);
    var bucketingKey = bucketing(key);

    // If the SDK was not ready, treatment may be incorrect due to having Splits but not segments data.
    if (!isSdkReady) {
      evaluation = { treatment: CONTROL, label: SDK_NOT_READY };
    }

    var _evaluation = evaluation,
        treatment = _evaluation.treatment,
        label = _evaluation.label,
        changeNumber = _evaluation.changeNumber,
        _evaluation$config = _evaluation.config,
        config = _evaluation$config === undefined ? null : _evaluation$config;

    log.info('Split: ' + splitName + '. Key: ' + matchingKey + '. Evaluation: ' + treatment + '. Label: ' + label);

    if (validateSplitExistance(context, splitName, label, invokingMethodName)) {
      log.info('Queueing corresponding impression.');
      impressionsTracker({
        feature: splitName,
        keyName: matchingKey,
        treatment: treatment,
        time: Date.now(),
        bucketingKey: bucketingKey,
        label: label,
        changeNumber: changeNumber
      }, attributes);
    }

    stopLatencyTracker && stopLatencyTracker();

    if (withConfig) {
      return {
        treatment: treatment,
        config: config
      };
    }

    return treatment;
  }

  function track(key, trafficTypeName, eventTypeId) {
    var value = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : null;
    var properties = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : null;
    var size = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 1024;

    var matchingKey = matching(key);
    var timestamp = Date.now();
    var eventData = {
      eventTypeId: eventTypeId,
      trafficTypeName: trafficTypeName,
      value: value,
      timestamp: timestamp,
      key: matchingKey,
      properties: properties
    };

    // This may be async but we only warn, we don't actually care if it is valid or not in terms of queueing the event.
    validateTrafficTypeExistance(trafficTypeName, context, 'track');

    var tracked = storage.events.track(eventData, size);

    if (thenable(tracked)) {
      return tracked.then(queueEventsCallback.bind(null, eventData));
    } else {
      return queueEventsCallback(eventData, tracked);
    }
  }

  return {
    getTreatment: getTreatment, getTreatmentWithConfig: getTreatmentWithConfig,
    getTreatments: getTreatments, getTreatmentsWithConfig: getTreatmentsWithConfig,
    track: track
  };
}

export default ClientFactory;