import _objectWithoutProperties from 'babel-runtime/helpers/objectWithoutProperties';
/**
Copyright 2016 Split Software

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
**/

import { get } from '../utils/lang';
import parser from './parser';
import keyParser from '../utils/key/parser';
import thenable from '../utils/promise/thenable';
import * as LabelsConstants from '../utils/labels';
import { CONTROL } from '../utils/constants';

function defaults(inst) {
  // in case we don't have a default treatment in the instanciation, use 'control'
  if (typeof inst.baseInfo.defaultTreatment !== 'string') {
    inst.baseInfo.defaultTreatment = CONTROL;
  }
}

function evaluationResult(result, defaultTreatment) {
  return {
    treatment: get(result, 'treatment', defaultTreatment),
    label: get(result, 'label', LabelsConstants.NO_CONDITION_MATCH)
  };
}

function Split(baseInfo, evaluator) {
  if (!(this instanceof Split)) {
    return new Split(baseInfo, evaluator);
  }

  this.baseInfo = baseInfo;
  this.evaluator = evaluator;

  defaults(this);
}

Split.parse = function parse(splitFlatStructure, storage) {
  var conditions = splitFlatStructure.conditions,
      baseInfo = _objectWithoutProperties(splitFlatStructure, ['conditions']);

  var evaluator = parser(conditions, storage);

  return new Split(baseInfo, evaluator);
};

Split.prototype.getKey = function getKey() {
  return this.baseInfo.name;
};

Split.prototype.getTreatment = function getTreatment(key, attributes, splitEvaluator) {
  var _baseInfo = this.baseInfo,
      killed = _baseInfo.killed,
      seed = _baseInfo.seed,
      defaultTreatment = _baseInfo.defaultTreatment,
      trafficAllocation = _baseInfo.trafficAllocation,
      trafficAllocationSeed = _baseInfo.trafficAllocationSeed,
      algo = _baseInfo.algo;

  var parsedKey = void 0;
  var treatment = void 0;
  var label = void 0;

  try {
    parsedKey = keyParser(key);
  } catch (err) {
    return {
      treatment: CONTROL,
      label: LabelsConstants.EXCEPTION
    };
  }

  if (this.isGarbage()) {
    treatment = CONTROL;
    label = LabelsConstants.SPLIT_ARCHIVED;
  } else if (killed) {
    treatment = defaultTreatment;
    label = LabelsConstants.SPLIT_KILLED;
  } else {
    var evaluation = this.evaluator(parsedKey, seed, trafficAllocation, trafficAllocationSeed, attributes, algo, splitEvaluator);

    // Evaluation could be async, so we should handle that case checking for a
    // thenable object
    if (thenable(evaluation)) {
      return evaluation.then(function (result) {
        return evaluationResult(result, defaultTreatment);
      });
    } else {
      return evaluationResult(evaluation, defaultTreatment);
    }
  }

  return {
    treatment: treatment,
    label: label
  };
};

Split.prototype.isGarbage = function isGarbage() {
  return this.baseInfo.status === 'ARCHIVED';
};

Split.prototype.getChangeNumber = function getChangeNumber() {
  return this.baseInfo.changeNumber;
};

export default Split;