import _Object$keys from 'babel-runtime/core-js/object/keys';
/**
Copyright 2016 Split Software

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
**/

import fs from 'fs';
import path from 'path';
import yaml from 'js-yaml';
import logFactory from '../../../utils/logger';
import { isString, endsWith, find, forOwn, uniq } from '../../../utils/lang';
import parseCondition from './parseCondition';
var log = logFactory('splitio-offline:splits-fetcher');

var DEFAULT_FILENAME = '.split';

function configFilesPath() {
  var config = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

  var configFilePath = config.features;

  if (configFilePath === DEFAULT_FILENAME || !isString(configFilePath)) {
    var root = process.env.HOME;

    if (process.env.SPLIT_CONFIG_ROOT) root = process.env.SPLIT_CONFIG_ROOT;

    if (!root) throw 'Missing split mock configuration root.';

    configFilePath = path.join(root, DEFAULT_FILENAME);
  }

  // Validate the extensions
  if (!(endsWith(configFilePath, '.yaml', true) || endsWith(configFilePath, '.yml', true) || endsWith(configFilePath, '.split', true))) throw 'Invalid extension specified for Splits mock file. Accepted extensions are ".yml" and ".yaml". Your specified file is ' + configFilePath;

  if (!fs.existsSync(configFilePath)) throw 'Split configuration not found in ' + configFilePath + ' - Please review your Split file location.';

  return configFilePath;
}

// Parse `.split` configuration file and return a map of "Split Objects"
function readSplitConfigFile(filePath) {
  var SPLIT_POSITION = 0;
  var TREATMENT_POSITION = 1;
  var data = void 0;

  try {
    data = fs.readFileSync(filePath, 'utf-8');
  } catch (e) {
    log.error(e.message);

    return [];
  }

  var splitObjects = data.split(/\r?\n/).reduce(function (accum, line, index) {
    var tuple = line.trim();

    if (tuple === '' || tuple.charAt(0) === '#') {
      log.debug('Ignoring empty line or comment at #' + index);
    } else {
      tuple = tuple.split(/\s+/);

      if (tuple.length !== 2) {
        log.debug('Ignoring line since it does not have exactly two columns #' + index);
      } else {
        var splitName = tuple[SPLIT_POSITION];
        var condition = parseCondition({ treatment: tuple[TREATMENT_POSITION] });
        accum[splitName] = { conditions: [condition], configurations: {} };
      }
    }

    return accum;
  }, {});

  return splitObjects;
}

// Parse `.yml` or `.yaml` configuration files and return a map of "Split Objects"
function readYAMLConfigFile(filePath) {
  var yamldoc = null;

  try {
    yamldoc = yaml.safeLoad(fs.readFileSync(filePath, 'utf8'));
  } catch (e) {
    log.error(e);

    return {};
  }

  // Each entry will be mapped to a condition, but we'll also keep the configurations map.
  var mocksData = yamldoc.reduce(function (accum, splitEntry) {
    var splitName = _Object$keys(splitEntry)[0];

    if (!splitName || !isString(splitEntry[splitName].treatment)) log.error('Ignoring entry on YAML since the format is incorrect.');

    var mockData = splitEntry[splitName];

    // "Template" for each split accumulated data
    if (!accum[splitName]) {
      accum[splitName] = {
        configurations: {}, conditions: [], treatments: [], trafficTypeName: 'localhost'
      };
    }

    // Assign the config if there is one on the mock
    if (mockData.config) accum[splitName].configurations[mockData.treatment] = mockData.config;
    // Parse the condition from the entry.
    var condition = parseCondition(mockData);
    accum[splitName].conditions[condition.conditionType === 'ROLLOUT' ? 'push' : 'unshift'](condition);
    // Also keep track of the treatments, will be useful for manager functionality.
    accum[splitName].treatments.push(mockData.treatment);

    return accum;
  }, {});

  arrangeConditions(mocksData);

  return mocksData;
}

// This function is not pure nor meant to be. Here we apply modifications to cover
//  for behaviour that's ensured by the BE.
function arrangeConditions(mocksData) {
  // Iterate through each Split data
  forOwn(mocksData, function (data) {
    var conditions = data.conditions;

    // On the manager, as the split jsons come with all treatments on the partitions prop,
    // we'll add all the treatments to the first condition.
    var firstRolloutCondition = find(conditions, function (cond) {
      return cond.conditionType === 'ROLLOUT';
    });
    // Malformed mocks may have
    var treatments = uniq(data.treatments);
    // If they're only specifying a whitelist we add the treatments there.
    var allTreatmentsCondition = firstRolloutCondition ? firstRolloutCondition : conditions[0];

    var fullyAllocatedTreatment = allTreatmentsCondition.partitions[0].treatment;

    treatments.forEach(function (treatment) {
      if (treatment !== fullyAllocatedTreatment) {
        allTreatmentsCondition.partitions.push({
          treatment: treatment, size: 0
        });
      }
    });

    // Don't need these anymore
    delete data.treatments;
  });
}

// Load the content of a configuration file into an Object
function getSplitConfigForFile(settings) {
  var filePath = configFilesPath(settings);
  var mockData = null;

  // If we have a filePath, it means the extension is correct, choose the parser.
  if (endsWith(filePath, '.split')) {
    log.warn('.split mocks will be deprecated soon in favor of YAML files, which provide more targeting power. Take a look in our documentation.');
    mockData = readSplitConfigFile(filePath);
  } else {
    mockData = readYAMLConfigFile(filePath);
  }

  return mockData;
}

export default getSplitConfigForFile;