import _Promise from 'babel-runtime/core-js/promise';
import _Set from 'babel-runtime/core-js/set';
import _Object$getPrototypeOf from 'babel-runtime/core-js/object/get-prototype-of';
import _toConsumableArray from 'babel-runtime/helpers/toConsumableArray';
import _classCallCheck from 'babel-runtime/helpers/classCallCheck';
import _createClass from 'babel-runtime/helpers/createClass';
import _possibleConstructorReturn from 'babel-runtime/helpers/possibleConstructorReturn';
import _inherits from 'babel-runtime/helpers/inherits';
import ioredis from 'ioredis';
import { merge, isString } from '../utils/lang';
import thenable from '../utils/promise/thenable';
import timeout from '../utils/promise/timeout';

import LogFactory from '../utils/logger';
var log = LogFactory('splitio-storage:redis-adapter');

// If we ever decide to fully wrap every method, there's a Commander.getBuiltinCommands from ioredis.
var METHODS_TO_PROMISE_WRAP = ['set', 'exec', 'del', 'get', 'keys', 'sadd', 'srem', 'sismember', 'smembers', 'incr', 'rpush', 'pipeline', 'expire'];

// Not part of the settings since it'll vary on each storage. We should be removing storage specific logic from elsewhere.
var DEFAULT_OPTIONS = {
  connectionTimeout: 10000,
  operationTimeout: 5000
};
// Library specifics.
var DEFAULT_LIBRARY_OPTIONS = {
  enableOfflineQueue: false,
  connectTimeout: DEFAULT_OPTIONS.connectionTimeout,
  lazyConnect: false
};

/**
 * Redis adapter on top of the library of choice (written with ioredis) for some extra control.
 */

var RedisAdapter = function (_ioredis) {
  _inherits(RedisAdapter, _ioredis);

  function RedisAdapter(storageSettings) {
    var _ref;

    _classCallCheck(this, RedisAdapter);

    var options = RedisAdapter._defineOptions(storageSettings);
    // Call the ioredis constructor

    var _this = _possibleConstructorReturn(this, (_ref = RedisAdapter.__proto__ || _Object$getPrototypeOf(RedisAdapter)).call.apply(_ref, [this].concat(_toConsumableArray(RedisAdapter._defineLibrarySettings(options)))));

    _this._options = options;
    _this._notReadyCommandsQueue = [];
    _this._runningCommands = new _Set();
    _this._listenToEvents();
    _this._setTimeoutWrappers();
    _this._setDisconnectWrapper();
    return _this;
  }

  _createClass(RedisAdapter, [{
    key: '_listenToEvents',
    value: function _listenToEvents() {
      var _this2 = this;

      this.once('ready', function () {
        var commandsCount = _this2._notReadyCommandsQueue ? _this2._notReadyCommandsQueue.length : 0;
        log.info('Redis connection established. Queued commands: ' + commandsCount + '.');
        commandsCount && _this2._notReadyCommandsQueue.forEach(function (queued) {
          log.info('Executing queued ' + queued.name + ' command.');
          queued.command().then(queued.resolve).catch(queued.reject);
        });
        // After the SDK is ready for the first time we'll stop queueing commands. This is just so we can keep handling BUR for them.
        _this2._notReadyCommandsQueue = false;
      });
      this.once('close', function () {
        log.info('Redis connection closed.');
      });
    }
  }, {
    key: '_setTimeoutWrappers',
    value: function _setTimeoutWrappers() {
      var instance = this;

      METHODS_TO_PROMISE_WRAP.forEach(function (method) {
        var originalMethod = instance[method];

        instance[method] = function () {
          var params = arguments;

          function commandWrapper() {
            log.debug('Executing ' + method + '.');
            // Return original method
            var result = originalMethod.apply(instance, params);

            if (thenable(result)) {
              // For handling pending commands on disconnect, add to the set and remove once finished.
              // On sync commands there's no need, only thenables.
              instance._runningCommands.add(result);
              var cleanUpRunningCommandsCb = function cleanUpRunningCommandsCb(res) {
                instance._runningCommands.delete(result);
                return res;
              };
              // Both success and error remove from queue.
              result.then(cleanUpRunningCommandsCb, cleanUpRunningCommandsCb);

              return timeout(instance._options.operationTimeout, result).catch(function (err) {
                log.error(method + ' operation threw an error or exceeded configured timeout of ' + instance._options.operationTimeout + 'ms. Message: ' + err);
                // Handling is not the adapter responsibility.
                throw err;
              });
            }

            return result;
          }

          if (instance._notReadyCommandsQueue) {
            return new _Promise(function (res, rej) {
              instance._notReadyCommandsQueue.unshift({
                resolve: res,
                reject: rej,
                command: commandWrapper,
                name: method.toUpperCase()
              });
            });
          } else {
            return commandWrapper();
          }
        };
      });
    }
  }, {
    key: '_setDisconnectWrapper',
    value: function _setDisconnectWrapper() {
      var instance = this;
      var originalMethod = instance.disconnect;

      instance.disconnect = function disconnect() {
        var params = arguments;

        setTimeout(function deferedDisconnect() {
          if (instance._runningCommands.size > 0) {
            log.info('Attempting to disconnect but there are ' + instance._runningCommands.size + ' commands still waiting for resolution. Defering disconnection until those finish.');

            _Promise.all(instance._runningCommands.values()).then(function () {
              log.debug('Pending commands finished successfully, disconnecting.');
              originalMethod.apply(instance, params);
            }).catch(function (e) {
              log.warn('Pending commands finished with error: ' + e + '. Proceeding with disconnection.');
              originalMethod.apply(instance, params);
            });
          } else {
            log.debug('No commands pending execution, disconnect.');
            // Nothing pending, just proceed.
            originalMethod.apply(instance, params);
          }
        }, 10);
      };
    }

    /**
     * Receives the options and returns an array of parameters for the ioredis constructor.
     * Keeping both redis setup options for backwards compatibility.
     */

  }], [{
    key: '_defineLibrarySettings',
    value: function _defineLibrarySettings(options) {
      var opts = merge({}, DEFAULT_LIBRARY_OPTIONS);
      var result = [opts];

      if (!isString(options.url)) {
        merge(opts, { // If it's not the string URL, merge the params separately.
          host: options.host,
          port: options.port,
          db: options.db,
          password: options.pass
        });
      } else {
        // If it IS the string URL, that'll be the first param for ioredis.
        result.unshift(options.url);
      }

      return result;
    }

    /**
     * Parses the options into what we care about.
     */

  }, {
    key: '_defineOptions',
    value: function _defineOptions(_ref2) {
      var connectionTimeout = _ref2.connectionTimeout,
          operationTimeout = _ref2.operationTimeout,
          url = _ref2.url,
          host = _ref2.host,
          port = _ref2.port,
          db = _ref2.db,
          pass = _ref2.pass;

      var parsedOptions = {
        connectionTimeout: connectionTimeout, operationTimeout: operationTimeout, url: url, host: host, port: port, db: db, pass: pass
      };

      return merge({}, DEFAULT_OPTIONS, parsedOptions);
    }
  }]);

  return RedisAdapter;
}(ioredis);

export default RedisAdapter;