import _Number$isNaN from 'babel-runtime/core-js/number/is-nan';
import _Promise from 'babel-runtime/core-js/promise';
import _classCallCheck from 'babel-runtime/helpers/classCallCheck';
import _createClass from 'babel-runtime/helpers/createClass';
import _slicedToArray from 'babel-runtime/helpers/slicedToArray';
import { isFinite } from '../../utils/lang';
import logFactory from '../../utils/logger';
var log = logFactory('splitio-storage:redis');

/**
 * Discard errors for an answer of multiple operations.
 */
var processPipelineAnswer = function processPipelineAnswer(results) {
  return results.reduce(function (accum, _ref) {
    var _ref2 = _slicedToArray(_ref, 2),
        err = _ref2[0],
        value = _ref2[1];

    if (err === null) accum.push(value);
    return accum;
  }, []);
};

var SplitCacheInRedis = function () {
  function SplitCacheInRedis(keys, redis) {
    var _this = this;

    _classCallCheck(this, SplitCacheInRedis);

    this.redis = redis;
    this.keys = keys;
    this.redisError = false;

    this.redis.on('error', function (e) {
      _this.redisError = e;
    });

    this.redis.on('connect', function () {
      _this.redisError = false;
    });
  }

  _createClass(SplitCacheInRedis, [{
    key: 'addSplit',
    value: function addSplit(splitName, split) {
      return this.redis.set(this.keys.buildSplitKey(splitName), split).then(function (status) {
        return status === 'OK';
      });
    }
  }, {
    key: 'addSplits',
    value: function addSplits(entries) {
      var _this2 = this;

      if (entries.length) {
        var cmds = entries.map(function (_ref3) {
          var _ref4 = _slicedToArray(_ref3, 2),
              key = _ref4[0],
              value = _ref4[1];

          return ['set', _this2.keys.buildSplitKey(key), value];
        });

        return this.redis.pipeline(cmds).exec().then(processPipelineAnswer).then(function (answers) {
          return answers.map(function (status) {
            return status === 'OK';
          });
        });
      } else {
        return [true];
      }
    }

    /**
     * Remove a given split from Redis. Returns the number of deleted keys.
     */

  }, {
    key: 'removeSplit',
    value: function removeSplit(splitName) {
      return this.redis.del(this.keys.buildSplitKey(splitName));
    }

    /**
     * Bulk delete of splits from Redis. Returns the number of deleted keys.
     */

  }, {
    key: 'removeSplits',
    value: function removeSplits(names) {
      var _this3 = this;

      if (names.length) {
        return this.redis.del(names.map(function (n) {
          return _this3.keys.buildSplitKey(n);
        }));
      } else {
        return _Promise.resolve(0);
      }
    }

    /**
     * Get split definition or null if it's not defined.
     */

  }, {
    key: 'getSplit',
    value: function getSplit(splitName) {
      if (this.redisError) {
        log.error(this.redisError);

        throw this.redisError;
      }

      return this.redis.get(this.keys.buildSplitKey(splitName));
    }

    /**
     * Set till number.
     *
     * @TODO pending error handling
     */

  }, {
    key: 'setChangeNumber',
    value: function setChangeNumber(changeNumber) {
      return this.redis.set(this.keys.buildSplitsTillKey(), changeNumber + '').then(function (status) {
        return status === 'OK';
      });
    }

    /**
     * Get till number or null if it's not defined.
     *
     * @TODO pending error handling
     */

  }, {
    key: 'getChangeNumber',
    value: function getChangeNumber() {
      return this.redis.get(this.keys.buildSplitsTillKey()).then(function (value) {
        var i = parseInt(value, 10);

        return _Number$isNaN(i) ? -1 : i;
      });
    }

    /**
     * @TODO we need to benchmark which is the maximun number of commands we could
     *       pipeline without kill redis performance.
     */

  }, {
    key: 'getAll',
    value: function getAll() {
      var _this4 = this;

      return this.redis.keys(this.keys.searchPatternForSplitKeys()).then(function (listOfKeys) {
        return _this4.redis.pipeline(listOfKeys.map(function (k) {
          return ['get', k];
        })).exec();
      }).then(processPipelineAnswer);
    }
  }, {
    key: 'getKeys',
    value: function getKeys() {
      var _this5 = this;

      return this.redis.keys(this.keys.searchPatternForSplitKeys()).then(function (listOfKeys) {
        return listOfKeys.map(_this5.keys.extractKey);
      });
    }
  }, {
    key: 'trafficTypeExists',
    value: function trafficTypeExists(trafficType) {
      // If there is a number there should be > 0, otherwise the TT is considered as not existent.
      return this.redis.get(this.keys.buildTrafficTypeKey(trafficType)).then(function (ttCount) {
        ttCount = parseInt(ttCount, 10);
        if (!isFinite(ttCount) || ttCount < 0) {
          log.info('Could not validate traffic type existance of ' + trafficType + ' due to data corruption of some sorts.');
          return false;
        }

        return ttCount > 0;
      }).catch(function (e) {
        log.error('Could not validate traffic type existance of ' + trafficType + ' due to an error: ' + e + '.');
        // If there is an error, bypass the validation so the event can get tracked.
        return true;
      });
    }

    // noop, just keeping the interface. This is used by client-side implementations only.

  }, {
    key: 'usesSegments',
    value: function usesSegments() {
      return true;
    }

    /**
     * Delete everything in the current database.
     *
     * @NOTE documentation says it never fails.
     */

  }, {
    key: 'flush',
    value: function flush() {
      return this.redis.flushdb().then(function (status) {
        return status === 'OK';
      });
    }
  }]);

  return SplitCacheInRedis;
}();

export default SplitCacheInRedis;