'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _promise = require('babel-runtime/core-js/promise');

var _promise2 = _interopRequireDefault(_promise);

var _logger = require('../utils/logger');

var _logger2 = _interopRequireDefault(_logger);

var _evaluator = require('../engine/evaluator');

var _evaluator2 = _interopRequireDefault(_evaluator);

var _impression = require('../trackers/impression');

var _impression2 = _interopRequireDefault(_impression);

var _impressions = require('../trackers/impressions');

var _impressions2 = _interopRequireDefault(_impressions);

var _timeTracker = require('../utils/timeTracker');

var _timeTracker2 = _interopRequireDefault(_timeTracker);

var _thenable = require('../utils/promise/thenable');

var _thenable2 = _interopRequireDefault(_thenable);

var _factory = require('../utils/key/factory');

var _inputValidation = require('../utils/inputValidation');

var _labels = require('../utils/labels');

var _constants = require('../utils/constants');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var log = (0, _logger2.default)('splitio-client');
/* asynchronous validations that live on the client. */


function queueEventsCallback(_ref, tracked) {
  var eventTypeId = _ref.eventTypeId,
      trafficTypeName = _ref.trafficTypeName,
      key = _ref.key,
      value = _ref.value,
      timestamp = _ref.timestamp,
      properties = _ref.properties;

  // Logging every prop would be too much.
  var msg = 'event of type "' + eventTypeId + '" for traffic type "' + trafficTypeName + '". Key: ' + key + '. Value: ' + value + '. Timestamp: ' + timestamp + '. ' + (properties ? 'With properties.' : 'With no properties.');

  if (tracked) {
    log.info('Successfully qeued ' + msg);
  } else {
    log.warn('Failed to queue ' + msg);
  }

  return tracked;
}

function ClientFactory(context) {
  var storage = context.get(context.constants.STORAGE);
  var metricCollectors = context.get(context.constants.COLLECTORS);
  var impressionTracker = (0, _impression2.default)(context);
  var impressionsTracker = (0, _impressions2.default)(context);

  function getTreatment(key, splitName, attributes) {
    var withConfig = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

    var taskToBeTracked = _timeTracker2.default.TaskNames[withConfig ? 'SDK_GET_TREATMENT_WITH_CONFIG' : 'SDK_GET_TREATMENT'];
    var stopLatencyTracker = _timeTracker2.default.start(taskToBeTracked, metricCollectors);
    var evaluation = (0, _evaluator2.default)(key, splitName, attributes, storage);

    if ((0, _thenable2.default)(evaluation)) {
      return evaluation.then(function (res) {
        return processEvaluation(res, splitName, key, attributes, stopLatencyTracker, impressionTracker.track, withConfig, 'getTreatment' + (withConfig ? 'withConfig' : ''));
      });
    } else {
      return processEvaluation(evaluation, splitName, key, attributes, stopLatencyTracker, impressionTracker.track, withConfig, 'getTreatment' + (withConfig ? 'withConfig' : ''));
    }
  }

  function getTreatmentWithConfig(key, splitName, attributes) {
    return getTreatment(key, splitName, attributes, true);
  }

  function getTreatments(key, splitNames, attributes) {
    var withConfig = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : false;

    var taskToBeTracked = _timeTracker2.default.TaskNames[withConfig ? 'SDK_GET_TREATMENTS_WITH_CONFIG' : 'SDK_GET_TREATMENTS'];
    var stopLatencyTracker = _timeTracker2.default.start(taskToBeTracked, metricCollectors);
    var results = {};
    var thenables = [];
    var i = void 0;

    var _loop = function _loop() {
      var splitName = splitNames[i];
      var evaluation = (0, _evaluator2.default)(key, splitName, attributes, storage);

      if ((0, _thenable2.default)(evaluation)) {
        // If treatment returns a promise as it is being evaluated, save promise for progress tracking.
        thenables.push(evaluation);
        evaluation.then(function (res) {
          // set the treatment on the cb;
          results[splitName] = processEvaluation(res, splitName, key, attributes, false, impressionsTracker.queue, withConfig, 'getTreatments' + (withConfig ? 'withConfig' : ''));
        });
      } else {
        results[splitName] = processEvaluation(evaluation, splitName, key, attributes, false, impressionsTracker.queue, withConfig, 'getTreatments' + (withConfig ? 'withConfig' : ''));
      }
    };

    for (i = 0; i < splitNames.length; i++) {
      _loop();
    }

    var wrapUp = function wrapUp() {
      impressionsTracker.track();
      stopLatencyTracker();
      // After all treatments are resolved, we return the mapping object.
      return results;
    };

    if (thenables.length) {
      return _promise2.default.all(thenables).then(wrapUp);
    } else {
      return wrapUp();
    }
  }

  function getTreatmentsWithConfig(key, splitNames, attributes) {
    return getTreatments(key, splitNames, attributes, true);
  }

  // Internal function
  function processEvaluation(evaluation, splitName, key, attributes) {
    var stopLatencyTracker = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : false;
    var impressionsTracker = arguments[5];
    var withConfig = arguments[6];
    var invokingMethodName = arguments[7];

    var isSdkReady = context.get(context.constants.READY, true);
    var matchingKey = (0, _factory.matching)(key);
    var bucketingKey = (0, _factory.bucketing)(key);

    // If the SDK was not ready, treatment may be incorrect due to having Splits but not segments data.
    if (!isSdkReady) {
      evaluation = { treatment: _constants.CONTROL, label: _labels.SDK_NOT_READY };
    }

    var _evaluation = evaluation,
        treatment = _evaluation.treatment,
        label = _evaluation.label,
        changeNumber = _evaluation.changeNumber,
        _evaluation$config = _evaluation.config,
        config = _evaluation$config === undefined ? null : _evaluation$config;

    log.info('Split: ' + splitName + '. Key: ' + matchingKey + '. Evaluation: ' + treatment + '. Label: ' + label);

    if ((0, _inputValidation.validateSplitExistance)(context, splitName, label, invokingMethodName)) {
      log.info('Queueing corresponding impression.');
      impressionsTracker({
        feature: splitName,
        keyName: matchingKey,
        treatment: treatment,
        time: Date.now(),
        bucketingKey: bucketingKey,
        label: label,
        changeNumber: changeNumber
      }, attributes);
    }

    stopLatencyTracker && stopLatencyTracker();

    if (withConfig) {
      return {
        treatment: treatment,
        config: config
      };
    }

    return treatment;
  }

  function track(key, trafficTypeName, eventTypeId) {
    var value = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : null;
    var properties = arguments.length > 4 && arguments[4] !== undefined ? arguments[4] : null;
    var size = arguments.length > 5 && arguments[5] !== undefined ? arguments[5] : 1024;

    var matchingKey = (0, _factory.matching)(key);
    var timestamp = Date.now();
    var eventData = {
      eventTypeId: eventTypeId,
      trafficTypeName: trafficTypeName,
      value: value,
      timestamp: timestamp,
      key: matchingKey,
      properties: properties
    };

    // This may be async but we only warn, we don't actually care if it is valid or not in terms of queueing the event.
    (0, _inputValidation.validateTrafficTypeExistance)(trafficTypeName, context, 'track');

    var tracked = storage.events.track(eventData, size);

    if ((0, _thenable2.default)(tracked)) {
      return tracked.then(queueEventsCallback.bind(null, eventData));
    } else {
      return queueEventsCallback(eventData, tracked);
    }
  }

  return {
    getTreatment: getTreatment, getTreatmentWithConfig: getTreatmentWithConfig,
    getTreatments: getTreatments, getTreatmentsWithConfig: getTreatmentsWithConfig,
    track: track
  };
}

exports.default = ClientFactory;
module.exports = exports.default;