'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SplitFactory = SplitFactory;

var _manager = require('./manager');

var _manager2 = _interopRequireDefault(_manager);

var _storage = require('./storage');

var _storage2 = _interopRequireDefault(_storage);

var _readiness2 = require('./readiness');

var _readiness3 = _interopRequireDefault(_readiness2);

var _settings = require('./utils/settings');

var _settings2 = _interopRequireDefault(_settings);

var _context = require('./utils/context');

var _context2 = _interopRequireDefault(_context);

var _parser = require('./utils/key/parser');

var _parser2 = _interopRequireDefault(_parser);

var _logger = require('./utils/logger');

var _logger2 = _interopRequireDefault(_logger);

var _timeTracker = require('./utils/timeTracker');

var _timeTracker2 = _interopRequireDefault(_timeTracker);

var _online = require('./factory/online');

var _online2 = _interopRequireDefault(_online);

var _offline = require('./factory/offline');

var _offline2 = _interopRequireDefault(_offline);

var _statusManager = require('./readiness/statusManager');

var _statusManager2 = _interopRequireDefault(_statusManager);

var _constants = require('./utils/constants');

var _inputValidation = require('./utils/inputValidation');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var log = (0, _logger2.default)('splitio');


var buildInstanceId = function buildInstanceId(key, trafficType) {
  return (key.matchingKey ? key.matchingKey : key) + '-' + (key.bucketingKey ? key.bucketingKey : key) + '-' + (trafficType !== undefined ? trafficType : '');
};

function SplitFactory(config) {
  // Cache instances created per factory.
  var clientInstances = {};

  // Tracking times. We need to do it here because we need the storage created.
  var readyLatencyTrackers = {
    splitsReadyTracker: _timeTracker2.default.start(_timeTracker2.default.TaskNames.SPLITS_READY),
    segmentsReadyTracker: _timeTracker2.default.start(_timeTracker2.default.TaskNames.SEGMENTS_READY),
    sdkReadyTracker: _timeTracker2.default.start(_timeTracker2.default.TaskNames.SDK_READY)
  };
  var context = new _context2.default();

  // Put settings config within context
  var settings = (0, _settings2.default)(config);
  context.put(context.constants.SETTINGS, settings);

  // We will just log and allow for the SDK to end up throwing an SDK_TIMEOUT event for devs to handle.
  (0, _inputValidation.validateApiKey)(settings.core.authorizationKey);

  // Put storage config within context
  var storage = (0, _storage2.default)(context);
  var gateFactory = (0, _readiness3.default)();
  context.put(context.constants.STORAGE, storage);

  // Define which type of factory to use
  var splitFactory = settings.mode === _constants.LOCALHOST_MODE ? _offline2.default : _online2.default;

  // Put readiness config within context
  var readiness = gateFactory(settings.startup.readyTimeout);
  context.put(context.constants.READINESS, readiness);
  var statusManager = (0, _statusManager2.default)(context);
  context.put(context.constants.STATUS_MANAGER, statusManager);

  var _splitFactory = splitFactory(context, readyLatencyTrackers),
      mainClientInstance = _splitFactory.api,
      mainClientMetricCollectors = _splitFactory.metricCollectors;

  // It makes no sense to have multiple instances of the manager.


  var managerInstance = (0, _manager2.default)(storage.splits, context);

  var parsedDefaultKey = (0, _parser2.default)(settings.core.key);
  var defaultInstanceId = buildInstanceId(parsedDefaultKey, settings.core.trafficType);
  clientInstances[defaultInstanceId] = mainClientInstance;

  log.info('New Split SDK instance created.');

  return {
    // Split evaluation and event tracking engine
    client: function client(key, trafficType) {
      if (key === undefined) {
        log.debug('Retrieving default SDK client.');
        return mainClientInstance;
      }

      if (typeof storage.shared != 'function') {
        throw 'Shared Client not supported by the storage mechanism. Create isolated instances instead.';
      }

      // Validate the key value
      var validKey = (0, _inputValidation.validateKey)(key, 'Shared Client instantiation');
      if (validKey === false) {
        throw 'Shared Client needs a valid key.';
      }

      var validTrafficType = void 0;
      if (trafficType !== undefined) {
        validTrafficType = (0, _inputValidation.validateTrafficType)(trafficType, 'Shared Client instantiation');
        if (validTrafficType === false) {
          throw 'Shared Client needs a valid traffic type or no traffic type at all.';
        }
      }
      var instanceId = buildInstanceId(validKey, validTrafficType);

      if (!clientInstances[instanceId]) {
        var sharedSettings = settings.overrideKeyAndTT(validKey, validTrafficType);
        var sharedContext = new _context2.default();

        sharedContext.put(context.constants.READY, true); // For SDK inner workings it's supposed to be ready.
        var _readiness = gateFactory(sharedSettings.startup.readyTimeout);
        sharedContext.put(context.constants.READINESS, _readiness);
        sharedContext.put(sharedContext.constants.STATUS_MANAGER, (0, _statusManager2.default)(sharedContext, true));
        sharedContext.put(context.constants.SETTINGS, sharedSettings);
        sharedContext.put(context.constants.STORAGE, storage.shared(sharedSettings));

        // As shared clients reuse all the storage information, we don't need to check here if we
        // will use offline or online mode. We should stick with the original decision.
        clientInstances[instanceId] = splitFactory(sharedContext, false, mainClientMetricCollectors).api;
        // The readiness should depend on the readiness of the parent, instead of showing ready by default.
        clientInstances[instanceId].ready = mainClientInstance.ready;

        log.info('New shared client instance created.');
      } else {
        log.debug('Retrieving existing SDK client.');
      }

      return clientInstances[instanceId];
    },


    // Manager API to explore available information
    manager: function manager() {
      log.info('Manager instance retrieved.');
      return managerInstance;
    },


    // Logger wrapper API
    Logger: _logger.API,

    // Expose SDK settings
    settings: settings
  };
}