'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _jsYaml = require('js-yaml');

var _jsYaml2 = _interopRequireDefault(_jsYaml);

var _logger = require('../../../utils/logger');

var _logger2 = _interopRequireDefault(_logger);

var _lang = require('../../../utils/lang');

var _parseCondition = require('./parseCondition');

var _parseCondition2 = _interopRequireDefault(_parseCondition);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
Copyright 2016 Split Software

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
**/

var log = (0, _logger2.default)('splitio-offline:splits-fetcher');

var DEFAULT_FILENAME = '.split';

function configFilesPath() {
  var config = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

  var configFilePath = config.features;

  if (configFilePath === DEFAULT_FILENAME || !(0, _lang.isString)(configFilePath)) {
    var root = process.env.HOME;

    if (process.env.SPLIT_CONFIG_ROOT) root = process.env.SPLIT_CONFIG_ROOT;

    if (!root) throw 'Missing split mock configuration root.';

    configFilePath = _path2.default.join(root, DEFAULT_FILENAME);
  }

  // Validate the extensions
  if (!((0, _lang.endsWith)(configFilePath, '.yaml', true) || (0, _lang.endsWith)(configFilePath, '.yml', true) || (0, _lang.endsWith)(configFilePath, '.split', true))) throw 'Invalid extension specified for Splits mock file. Accepted extensions are ".yml" and ".yaml". Your specified file is ' + configFilePath;

  if (!_fs2.default.existsSync(configFilePath)) throw 'Split configuration not found in ' + configFilePath + ' - Please review your Split file location.';

  return configFilePath;
}

// Parse `.split` configuration file and return a map of "Split Objects"
function readSplitConfigFile(filePath) {
  var SPLIT_POSITION = 0;
  var TREATMENT_POSITION = 1;
  var data = void 0;

  try {
    data = _fs2.default.readFileSync(filePath, 'utf-8');
  } catch (e) {
    log.error(e.message);

    return [];
  }

  var splitObjects = data.split(/\r?\n/).reduce(function (accum, line, index) {
    var tuple = line.trim();

    if (tuple === '' || tuple.charAt(0) === '#') {
      log.debug('Ignoring empty line or comment at #' + index);
    } else {
      tuple = tuple.split(/\s+/);

      if (tuple.length !== 2) {
        log.debug('Ignoring line since it does not have exactly two columns #' + index);
      } else {
        var splitName = tuple[SPLIT_POSITION];
        var condition = (0, _parseCondition2.default)({ treatment: tuple[TREATMENT_POSITION] });
        accum[splitName] = { conditions: [condition], configurations: {} };
      }
    }

    return accum;
  }, {});

  return splitObjects;
}

// Parse `.yml` or `.yaml` configuration files and return a map of "Split Objects"
function readYAMLConfigFile(filePath) {
  var yamldoc = null;

  try {
    yamldoc = _jsYaml2.default.safeLoad(_fs2.default.readFileSync(filePath, 'utf8'));
  } catch (e) {
    log.error(e);

    return {};
  }

  // Each entry will be mapped to a condition, but we'll also keep the configurations map.
  var mocksData = yamldoc.reduce(function (accum, splitEntry) {
    var splitName = (0, _keys2.default)(splitEntry)[0];

    if (!splitName || !(0, _lang.isString)(splitEntry[splitName].treatment)) log.error('Ignoring entry on YAML since the format is incorrect.');

    var mockData = splitEntry[splitName];

    // "Template" for each split accumulated data
    if (!accum[splitName]) {
      accum[splitName] = {
        configurations: {}, conditions: [], treatments: [], trafficTypeName: 'localhost'
      };
    }

    // Assign the config if there is one on the mock
    if (mockData.config) accum[splitName].configurations[mockData.treatment] = mockData.config;
    // Parse the condition from the entry.
    var condition = (0, _parseCondition2.default)(mockData);
    accum[splitName].conditions[condition.conditionType === 'ROLLOUT' ? 'push' : 'unshift'](condition);
    // Also keep track of the treatments, will be useful for manager functionality.
    accum[splitName].treatments.push(mockData.treatment);

    return accum;
  }, {});

  arrangeConditions(mocksData);

  return mocksData;
}

// This function is not pure nor meant to be. Here we apply modifications to cover
//  for behaviour that's ensured by the BE.
function arrangeConditions(mocksData) {
  // Iterate through each Split data
  (0, _lang.forOwn)(mocksData, function (data) {
    var conditions = data.conditions;

    // On the manager, as the split jsons come with all treatments on the partitions prop,
    // we'll add all the treatments to the first condition.
    var firstRolloutCondition = (0, _lang.find)(conditions, function (cond) {
      return cond.conditionType === 'ROLLOUT';
    });
    // Malformed mocks may have
    var treatments = (0, _lang.uniq)(data.treatments);
    // If they're only specifying a whitelist we add the treatments there.
    var allTreatmentsCondition = firstRolloutCondition ? firstRolloutCondition : conditions[0];

    var fullyAllocatedTreatment = allTreatmentsCondition.partitions[0].treatment;

    treatments.forEach(function (treatment) {
      if (treatment !== fullyAllocatedTreatment) {
        allTreatmentsCondition.partitions.push({
          treatment: treatment, size: 0
        });
      }
    });

    // Don't need these anymore
    delete data.treatments;
  });
}

// Load the content of a configuration file into an Object
function getSplitConfigForFile(settings) {
  var filePath = configFilesPath(settings);
  var mockData = null;

  // If we have a filePath, it means the extension is correct, choose the parser.
  if ((0, _lang.endsWith)(filePath, '.split')) {
    log.warn('.split mocks will be deprecated soon in favor of YAML files, which provide more targeting power. Take a look in our documentation.');
    mockData = readSplitConfigFile(filePath);
  } else {
    mockData = readYAMLConfigFile(filePath);
  }

  return mockData;
}

exports.default = getSplitConfigForFile;
module.exports = exports.default;