'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateSplit = validateSplit;
exports.validateSplitExistance = validateSplitExistance;

var _lang = require('../lang');

var _labels = require('../labels');

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var log = (0, _logger2.default)('', {
  displayAllErrors: true
});
// include BOM and nbsp
var TRIMMABLE_SPACES_REGEX = /^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/;

function validateSplit(maybeSplit, method) {
  if (maybeSplit == undefined) {
    // eslint-disable-line eqeqeq
    log.error(method + ': you passed a null or undefined split name, split name must be a non-empty string.');
  } else if (!(0, _lang.isString)(maybeSplit)) {
    log.error(method + ': you passed an invalid split name, split name must be a non-empty string.');
  } else {
    if (TRIMMABLE_SPACES_REGEX.test(maybeSplit)) {
      log.warn(method + ': split name "' + maybeSplit + '" has extra whitespace, trimming.');
      maybeSplit = maybeSplit.trim();
    }

    if (maybeSplit.length > 0) {
      return maybeSplit;
    } else {
      log.error(method + ': you passed an empty split name, split name must be a non-empty string.');
    }
  }

  return false;
}

/**
 * This is defined here and in this format mostly because of the logger and the fact that it's considered a validation at product level.
 * But it's not going to run on the input validation layer. In any case, the most compeling reason to use it as we do is to avoid going to Redis and get a split twice.
 */
function validateSplitExistance(context, splitName, labelOrSplitObj, method) {
  if (context.get(context.constants.READY, true)) {
    // Only if it's ready we validate this, otherwise it may just be that the SDK is not ready yet.
    if (labelOrSplitObj === _labels.SPLIT_NOT_FOUND || labelOrSplitObj == null) {
      log.warn(method + ': you passed "' + splitName + '" that does not exist in this environment, please double check what Splits exist in the web console.');
      return false;
    }
  }

  return true;
}