'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTrafficType = validateTrafficType;
exports.validateTrafficTypeExistance = validateTrafficTypeExistance;

var _lang = require('../lang');

var _thenable = require('../promise/thenable');

var _thenable2 = _interopRequireDefault(_thenable);

var _constants = require('../constants');

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var log = (0, _logger2.default)('', {
  displayAllErrors: true
});

var CAPITAL_LETTERS_REGEX = /[A-Z]/;

function validateTrafficType(maybeTT, method) {
  if (maybeTT == undefined) {
    // eslint-disable-line eqeqeq
    log.error(method + ': you passed a null or undefined traffic_type_name, traffic_type_name must be a non-empty string.');
  } else if (!(0, _lang.isString)(maybeTT)) {
    log.error(method + ': you passed an invalid traffic_type_name, traffic_type_name must be a non-empty string.');
  } else {
    if (maybeTT.length === 0) {
      log.error(method + ': you passed an empty traffic_type_name, traffic_type_name must be a non-empty string.');
    } else {
      if (CAPITAL_LETTERS_REGEX.test(maybeTT)) {
        log.warn(method + ': traffic_type_name should be all lowercase - converting string to lowercase.');
        maybeTT = maybeTT.toLowerCase();
      }

      return maybeTT;
    }
  }

  return false;
}

function logTTExistanceWarning(method, ttName) {
  log.warn(method + ': Traffic Type ' + ttName + ' does not have any corresponding Splits in this environment, make sure you\'re tracking your events to a valid traffic type defined in the Split console.');
}

/**
 * Separated from the previous method since on some cases it'll be async.
 */
function validateTrafficTypeExistance(maybeTT, context, method) {
  var isReady = context.get(context.constants.READY, true);
  var settings = context.get(context.constants.SETTINGS);
  var splitsStorage = context.get(context.constants.STORAGE).splits;

  // If not ready or in localhost mode, we won't run the validation
  if (!isReady || settings.mode === _constants.LOCALHOST_MODE) return true;

  var res = splitsStorage.trafficTypeExists(maybeTT);

  if ((0, _thenable2.default)(res)) {
    res.then(function (isValid) {
      if (!isValid) logTTExistanceWarning(method, maybeTT);

      return isValid; // propagate result
    });
  } else {
    if (!res) logTTExistanceWarning(method, maybeTT);
  }

  return res;
}