import { Component, Input, Output, ElementRef, EventEmitter, ViewEncapsulation, forwardRef } from '@angular/core';
import { NG_VALUE_ACCESSOR } from '@angular/forms';
import * as CodeMirror from 'codemirror-minified';
import * as _$ from 'jquery';
const $ = _$;
// CodeMirror extensions
import 'codemirror-minified/mode/meta';
import 'codemirror-minified/addon/lint/lint';
import 'codemirror-minified/addon/hint/show-hint';
// import 'codemirror-minified/addon/mode/loadmode';
import 'codemirror-minified/addon/edit/matchbrackets';
import 'codemirror-minified/addon/edit/closebrackets';
import 'codemirror-minified/addon/display/placeholder';
import 'codemirror-minified/addon/scroll/annotatescrollbar';
import 'codemirror-minified/addon/scroll/simplescrollbars';
// Lint support
// Unclear how to import this dynamically...
// import 'codemirror-minified/addon/lint/javascript-lint';
// import 'codemirror-minified/addon/lint/json-lint';
// import 'codemirror-minified/addon/lint/yaml-lint';
// TODO: use dynamic import with JS7 in the future. CM autoLoad cannot load it properly - thinks its AMD
// Supported languages until dynamic loading
// import 'codemirror-minified/mode/groovy/groovy';
// import 'codemirror-minified/mode/javascript/javascript';
// import 'codemirror-minified/mode/python/python';
// import 'codemirror-minified/mode/ruby/ruby';
// import 'codemirror-minified/mode/clike/clike';
// import 'codemirror-minified/mode/yaml/yaml';
export class CodeEditorComponent {
    constructor(element) {
        this.element = element;
        this._dsl = '';
        this._lint = false;
        this.lineNumbers = false;
        this.lineWrapping = false;
        this.dslChange = new EventEmitter();
        this.focus = new EventEmitter();
        this.blur = new EventEmitter();
        this.editor = new EventEmitter();
        this._dslChangedHandler = () => {
            this._dsl = this.doc.getValue();
            this.dslChange.emit(this._dsl);
            if (this._onChangeHandler) {
                this._onChangeHandler(this._dsl);
            }
        };
    }
    set dsl(dsl) {
        this._dsl = dsl;
        if (this.doc && this._dsl !== this.doc.getValue()) {
            let cursorPosition = this.doc.getCursor();
            this.doc.setValue(this._dsl || '');
            this.doc.setCursor(cursorPosition);
        }
    }
    set language(_language) {
        if (this._language !== _language) {
            this._language = _language;
            this.loadEditorMode();
        }
    }
    ngOnInit() {
        let options = {
            value: this._dsl || '',
            gutters: ['CodeMirror-lint-markers'],
            extraKeys: { 'Ctrl-Space': 'autocomplete' },
            lineNumbers: this.lineNumbers,
            lineWrapping: this.lineWrapping,
            matchBrackets: true,
            autoCloseBrackets: true,
        };
        if (this.scrollbarStyle) {
            options.scrollbarStyle = this.scrollbarStyle;
        }
        if (this._lint) {
            options.lint = this._lint;
        }
        this.doc = CodeMirror.fromTextArea($('#code-editor-host', this.element.nativeElement)[0], options);
        if (this.placeholder) {
            this.doc.setOption('placeholder', this.placeholder);
        }
        // Turns out "value" in the option doesn't set it.
        this.doc.setValue(this._dsl || '');
        this.doc.on('change', this._dslChangedHandler);
        this.doc.on('focus', () => {
            this.focus.emit();
            if (this._onTouchHandler) {
                this._onTouchHandler();
            }
        });
        this.doc.on('blur', () => this.blur.emit());
        this.warningRuler = this.doc.annotateScrollbar('CodeMirror-vertical-ruler-warning');
        this.errorRuler = this.doc.annotateScrollbar('CodeMirror-vertical-ruler-error');
        this.loadEditorMode();
        this.editor.emit(this.doc);
    }
    loadEditorMode() {
        // CodeMirror doc object must be initialized
        if (!this.doc) {
            return;
        }
        const info = this._language ? CodeMirror.findModeByName(this._language) : undefined;
        // Set proper editor mode
        if (info) {
            this.doc.setOption('mode', info.mime);
            // (<any>CodeMirror).autoLoadMode(this.doc, info.mode);
        }
        else {
            this.doc.setOption('mode', 'text/plain');
        }
        // Set proper Lint mode
        this.doc.setOption('lint', this.getLintOptions());
    }
    writeValue(obj) {
        this.dsl = obj;
    }
    registerOnChange(fn) {
        this._onChangeHandler = fn;
    }
    registerOnTouched(fn) {
        this._onTouchHandler = fn;
    }
    getLintOptions() {
        switch (this._language) {
            case 'javascript':
            case 'json':
            case 'coffeescript':
            case 'yaml':
                return {
                    onUpdateLinting: (annotations) => {
                        const warnings = [];
                        const errors = [];
                        if (this.overviewRuler) {
                            if (Array.isArray(annotations)) {
                                annotations.forEach((a) => {
                                    if (a.to && a.from && a.from.line >= 0 && a.from.ch >= 0 && a.to.line >= a.from.line && a.from.ch >= 0) {
                                        if (a.severity === 'error') {
                                            errors.push(a);
                                        }
                                        else if (a.severity === 'warning') {
                                            warnings.push(a);
                                        }
                                    }
                                });
                            }
                        }
                        this.warningRuler.update(warnings);
                        this.errorRuler.update(errors);
                    }
                };
        }
        return false;
    }
}
CodeEditorComponent.decorators = [
    { type: Component, args: [{
                selector: 'code-editor',
                template: "<div class=\"code-editor-host\">\n  <textarea id=\"code-editor-host\"></textarea>\n</div>\n",
                encapsulation: ViewEncapsulation.None,
                providers: [
                    {
                        provide: NG_VALUE_ACCESSOR,
                        useExisting: forwardRef(() => CodeEditorComponent),
                        multi: true
                    }
                ],
                styles: [".CodeMirror{-moz-user-select:none;-ms-user-select:none;-o-user-select:none;-webkit-user-select:none;height:100%;user-select:none}.CodeMirror-hint{max-width:38em}.CodeMirror-vertical-ruler-error{background-color:rgba(188,0,0,.5)}.CodeMirror-vertical-ruler-warning{background-color:rgba(255,188,0,.5)}.CodeMirror-lint-tooltip{z-index:2000}"]
            },] }
];
CodeEditorComponent.ctorParameters = () => [
    { type: ElementRef }
];
CodeEditorComponent.propDecorators = {
    lineNumbers: [{ type: Input, args: ['line-numbers',] }],
    lineWrapping: [{ type: Input, args: ['line-wrapping',] }],
    scrollbarStyle: [{ type: Input, args: ['scrollbar-style',] }],
    placeholder: [{ type: Input }],
    overviewRuler: [{ type: Input, args: ['overview-ruler',] }],
    dslChange: [{ type: Output }],
    focus: [{ type: Output }],
    blur: [{ type: Output }],
    editor: [{ type: Output }],
    dsl: [{ type: Input }],
    language: [{ type: Input }]
};
//# sourceMappingURL=data:application/json;base64,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