import { Component, Input, Output, ElementRef, EventEmitter, ViewEncapsulation, } from '@angular/core';
import { debounceTime } from 'rxjs/operators';
import { Flo } from '../shared/flo-common';
import { Shapes, Constants } from '../shared/shapes';
import { Utils } from './editor-utils';
import { CompositeDisposable, Disposable } from 'ts-disposables';
import * as _$ from 'jquery';
import * as _ from 'lodash';
import { Subject, BehaviorSubject } from 'rxjs';
import { Logger } from '../shared/logger';
const joint = Flo.joint;
const $ = _$;
const SCROLLBAR_SIZE = 17;
export class EditorComponent {
    constructor(element) {
        this.element = element;
        /**
         * Flag specifying whether the Flo-Editor is in read-only mode.
         */
        this._readOnlyCanvas = false;
        /**
         * Grid size
         */
        this._gridSize = 1;
        this._hiddenPalette = false;
        this.paletteSizeValue = 170;
        this.textToGraphEventEmitter = new EventEmitter();
        this.graphToTextEventEmitter = new EventEmitter();
        this._graphToTextSyncEnabled = true;
        this.validationEventEmitter = new EventEmitter();
        this._disposables = new CompositeDisposable();
        this._dslText = '';
        this.textToGraphConversionCompleted = new Subject();
        this.graphToTextConversionCompleted = new Subject();
        this.paletteReady = new BehaviorSubject(false);
        this.paletteSizeChange = new EventEmitter();
        this.searchFilterPlaceHolder = 'Search...';
        /**
         * Min zoom percent value
         */
        this.minZoom = 5;
        /**
         * Max zoom percent value
         */
        this.maxZoom = 400;
        /**
         * Zoom percent increment/decrement step
         */
        this.zoomStep = 5;
        this.paperPadding = 0;
        this.floApi = new EventEmitter();
        this.validationMarkers = new EventEmitter();
        this.contentValidated = new EventEmitter();
        this.dslChange = new EventEmitter();
        this.onProperties = new EventEmitter();
        this._resizeHandler = () => this.autosizePaper();
        let self = this;
        this.editorContext = new (class DefaultRunnableContext {
            set zoomPercent(percent) {
                self.zoomPercent = percent;
            }
            get zoomPercent() {
                return self.zoomPercent;
            }
            set noPalette(noPalette) {
                self.noPalette = noPalette;
            }
            get noPalette() {
                return self.noPalette;
            }
            set gridSize(gridSize) {
                self.gridSize = gridSize;
            }
            get gridSize() {
                return self.gridSize;
            }
            set readOnlyCanvas(readOnly) {
                self.readOnlyCanvas = readOnly;
            }
            get readOnlyCanvas() {
                return self.readOnlyCanvas;
            }
            setDsl(dsl) {
                self.dsl = dsl;
            }
            updateGraph() {
                return self.updateGraphRepresentation();
            }
            updateText() {
                return self.updateTextRepresentation();
            }
            performLayout() {
                return self.doLayout();
            }
            clearGraph() {
                self.selection = undefined;
                self.graph.clear();
                if (self.metamodel && self.metamodel.load && self.editor && self.editor.setDefaultContent) {
                    return self.metamodel.load().then(data => {
                        self.editor.setDefaultContent(this, data);
                        if (!self.graphToTextSync) {
                            return self.updateTextRepresentation();
                        }
                    });
                }
                else {
                    if (!self.graphToTextSync) {
                        return self.updateTextRepresentation();
                    }
                }
            }
            getGraph() {
                return self.graph;
            }
            getPaper() {
                return self.paper;
            }
            get graphToTextSync() {
                return self.graphToTextSync;
            }
            set graphToTextSync(sync) {
                self.graphToTextSync = sync;
            }
            getMinZoom() {
                return self.minZoom;
            }
            getMaxZoom() {
                return self.maxZoom;
            }
            getZoomStep() {
                return self.zoomStep;
            }
            fitToPage() {
                self.fitToPage();
            }
            createNode(metadata, props, position) {
                return self.createNode(metadata, props, position);
            }
            createLink(source, target, metadata, props) {
                return self.createLink(source, target, metadata, props);
            }
            get selection() {
                return self.selection;
            }
            set selection(newSelection) {
                self.selection = newSelection;
            }
            deleteSelectedNode() {
                self.deleteSelected();
            }
            delete(cell) {
                self.delete(cell);
            }
            get textToGraphConversionObservable() {
                return self.textToGraphConversionCompleted;
            }
            get graphToTextConversionObservable() {
                return self.graphToTextConversionCompleted;
            }
            get paletteReady() {
                return self.paletteReady;
            }
        })();
    }
    /**
     * Size (Width) of the palette
     */
    get paletteSize() {
        return this.paletteSizeValue;
    }
    set paletteSize(newSize) {
        this.paletteSizeValue = newSize;
        this.paletteSizeChange.emit(newSize);
    }
    onPropertiesHandle() {
        if (this.editorContext.selection) {
            this.onProperties.emit(this.editorContext.selection.model);
        }
    }
    ngOnInit() {
        this.initGraph();
        this.initPaper();
        this.initGraphListeners();
        this.initPaperListeners();
        this.initMetamodel();
        $(window).on('resize', this._resizeHandler);
        this._disposables.add(Disposable.create(() => $(window).off('resize', this._resizeHandler)));
        /*
         * Execute resize to get the right size for the SVG element on the editor canvas.
         * Executed via timeout to let angular render the DOM first and elements to have the right width and height
         */
        window.setTimeout(this._resizeHandler);
        this.floApi.emit(this.editorContext);
    }
    ngOnDestroy() {
        this._disposables.dispose();
    }
    deleteSelected() {
        if (this.selection) {
            this.delete(this.selection.model);
        }
    }
    delete(cell) {
        this.graph.trigger('startDeletion', cell);
    }
    get noPalette() {
        return this._hiddenPalette;
    }
    set noPalette(hidden) {
        this._hiddenPalette = hidden;
        // If palette is not shown ensure that canvas starts from the left==0!
        if (hidden) {
            $('#paper-container', this.element.nativeElement).css('left', 0);
        }
    }
    get graphToTextSync() {
        return this._graphToTextSyncEnabled;
    }
    set graphToTextSync(sync) {
        this._graphToTextSyncEnabled = sync;
        // Try commenting the sync out. Just set the flag but don't kick off graph->text conversion
        // this.performGraphToTextSyncing();
    }
    performGraphToTextSyncing() {
        if (this._graphToTextSyncEnabled) {
            this.graphToTextEventEmitter.emit();
        }
    }
    createHandle(element, kind, action, location) {
        if (!location) {
            let bbox = element.model.getBBox();
            location = bbox.origin().offset(bbox.width / 2, bbox.height / 2);
        }
        let handle = Shapes.Factory.createHandle({
            renderer: this.renderer,
            paper: this.paper,
            parent: element.model,
            kind: kind,
            position: location
        });
        const view = this.paper.findViewByModel(handle);
        view.on('cell:pointerdown', () => {
            if (action) {
                action();
            }
        });
        view.on('cell:mouseover', () => {
            handle.attr('image/filter', {
                name: 'dropShadow',
                args: { dx: 1, dy: 1, blur: 1, color: 'black' }
            });
        });
        view.on('cell:mouseout', () => {
            handle.removeAttr('image/filter');
        });
        view.setInteractivity(false);
        return handle;
    }
    removeEmbeddedChildrenOfType(element, types) {
        let embeds = element.getEmbeddedCells();
        for (let i = 0; i < embeds.length; i++) {
            if (types.indexOf(embeds[i].get('type')) >= 0) {
                embeds[i].remove();
            }
        }
    }
    get selection() {
        return this._selection;
    }
    set selection(newSelection) {
        var _a, _b;
        if (newSelection && (newSelection.model.get('type') === joint.shapes.flo.DECORATION_TYPE || newSelection.model.get('type') === joint.shapes.flo.HANDLE_TYPE)) {
            newSelection = this.paper.findViewByModel(this.graph.getCell(newSelection.model.get('parent')));
        }
        if (newSelection && (!newSelection.model.get('metadata') || ((_b = (_a = newSelection.model.get('metadata')) === null || _a === void 0 ? void 0 : _a.metadata) === null || _b === void 0 ? void 0 : _b.unselectable))) {
            newSelection = undefined;
        }
        if (newSelection !== this._selection) {
            if (this._selection) {
                const elementview = this.paper.findViewByModel(this._selection.model);
                if (elementview) { // May have been removed from the graph
                    this.removeEmbeddedChildrenOfType(elementview.model, joint.shapes.flo.HANDLE_TYPE);
                    elementview.unhighlight();
                }
            }
            if (newSelection) {
                newSelection.highlight();
                if (this.editor && this.editor.createHandles) {
                    this.editor.createHandles(this.editorContext, (owner, kind, action, location) => this.createHandle(owner, kind, action, location), newSelection);
                }
            }
            this._selection = newSelection;
        }
    }
    get readOnlyCanvas() {
        return this._readOnlyCanvas;
    }
    set readOnlyCanvas(value) {
        if (this._readOnlyCanvas === value) {
            // Nothing to do
            return;
        }
        if (value) {
            this.selection = undefined;
        }
        if (this.graph) {
            this.graph.getLinks().forEach((link) => {
                if (value) {
                    link.attr('.link-tools/display', 'none');
                    link.attr('.marker-vertices/display', 'none');
                    link.attr('.connection-wrap/display', 'none');
                }
                else {
                    link.removeAttr('.link-tools/display');
                    if (this.editor && this.editor.allowLinkVertexEdit) {
                        link.removeAttr('.marker-vertices/display');
                    }
                    link.removeAttr('.connection-wrap/display');
                }
            });
        }
        this._readOnlyCanvas = value;
    }
    /**
     * Displays graphical feedback for the drag and drop in progress based on current drag and drop descriptor object
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    showDragFeedback(dragDescriptor) {
        if (this.editor && this.editor.showDragFeedback) {
            this.editor.showDragFeedback(this.editorContext, dragDescriptor);
        }
        else {
            let magnet;
            if (dragDescriptor.source && dragDescriptor.source.view) {
                joint.V(dragDescriptor.source.view.el).addClass('dnd-source-feedback');
                if (dragDescriptor.source.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.source.view, dragDescriptor.source.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).addClass('dnd-source-feedback');
                    }
                }
            }
            if (dragDescriptor.target && dragDescriptor.target.view) {
                joint.V(dragDescriptor.target.view.el).addClass('dnd-target-feedback');
                if (dragDescriptor.target.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.target.view, dragDescriptor.target.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).addClass('dnd-target-feedback');
                    }
                }
            }
        }
    }
    /**
     * Hides graphical feedback for the drag and drop in progress based on current drag and drop descriptor object
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    hideDragFeedback(dragDescriptor) {
        if (this.editor && this.editor.hideDragFeedback) {
            this.editor.hideDragFeedback(this.editorContext, dragDescriptor);
        }
        else {
            let magnet;
            if (dragDescriptor.source && dragDescriptor.source.view) {
                joint.V(dragDescriptor.source.view.el).removeClass('dnd-source-feedback');
                if (dragDescriptor.source.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.source.view, dragDescriptor.source.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).removeClass('dnd-source-feedback');
                    }
                }
            }
            if (dragDescriptor.target && dragDescriptor.target.view) {
                joint.V(dragDescriptor.target.view.el).removeClass('dnd-target-feedback');
                if (dragDescriptor.target.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.target.view, dragDescriptor.target.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).removeClass('dnd-target-feedback');
                    }
                }
            }
        }
    }
    /**
     * Sets the new DnD info object - the descriptor for DnD
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    setDragDescriptor(dragDescriptor) {
        if (this.highlighted === dragDescriptor) {
            return;
        }
        if (this.highlighted && dragDescriptor && _.isEqual(this.highlighted.sourceComponent, dragDescriptor.sourceComponent)) {
            if (this.highlighted.source === dragDescriptor.source && this.highlighted.target === dragDescriptor.target) {
                return;
            }
            if (this.highlighted.source &&
                dragDescriptor.source &&
                this.highlighted.target &&
                dragDescriptor.target &&
                this.highlighted.source.view.model === dragDescriptor.source.view.model &&
                this.highlighted.source.cssClassSelector === dragDescriptor.source.cssClassSelector &&
                this.highlighted.target.view.model === dragDescriptor.target.view.model &&
                this.highlighted.target.cssClassSelector === dragDescriptor.target.cssClassSelector) {
                return;
            }
        }
        if (this.highlighted) {
            this.hideDragFeedback(this.highlighted);
        }
        this.highlighted = dragDescriptor;
        if (this.highlighted) {
            this.showDragFeedback(this.highlighted);
        }
    }
    /**
     * Handles DnD events when a node is being dragged over canvas
     *
     * @param draggedView The Joint JS view object being dragged
     * @param targetUnderMouse The Joint JS view under mouse cursor
     * @param x X coordinate of the mouse on the canvas
     * @param y Y coordinate of the mosue on the canvas
     * @param context DnD context (palette or canvas)
     */
    handleNodeDragging(draggedView, targetUnderMouse, x, y, sourceComponent) {
        if (this.editor && this.editor.calculateDragDescriptor) {
            this.setDragDescriptor(this.editor.calculateDragDescriptor(this.editorContext, draggedView, targetUnderMouse, joint.g.point(x, y), sourceComponent));
        }
    }
    /**
     * Handles DnD drop event when a node is being dragged and dropped on the main canvas
     */
    handleNodeDropping() {
        if (this.highlighted && this.editor && this.editor.handleNodeDropping) {
            this.editor.handleNodeDropping(this.editorContext, this.highlighted);
        }
        this.setDragDescriptor(undefined);
    }
    /**
     * Hides DOM Node (used to determine drop target DOM element)
     * @param domNode DOM node to hide
     * @returns
     */
    _hideNode(domNode) {
        let oldVisibility = {
            visibility: domNode.style ? domNode.style.display : undefined,
            children: []
        };
        for (let i = 0; i < domNode.children.length; i++) {
            let node = domNode.children.item(i);
            if (node instanceof HTMLElement) {
                oldVisibility.children.push(this._hideNode(node));
            }
        }
        domNode.style.display = 'none';
        return oldVisibility;
    }
    /**
     * Restored DOM node original visibility (used to determine drop target DOM element)
     * @param domNode DOM node to restore visibility of
     * @param oldVisibility original visibility parameter
     */
    _restoreNodeVisibility(domNode, oldVisibility) {
        if (domNode.style) {
            domNode.style.display = oldVisibility.visibility;
        }
        let j = 0;
        for (let i = 0; i < domNode.childNodes.length; i++) {
            if (j < oldVisibility.children.length) {
                let node = domNode.children.item(i);
                if (node instanceof HTMLElement) {
                    this._restoreNodeVisibility(node, oldVisibility.children[j++]);
                }
            }
        }
    }
    /**
     * Unfortunately we can't just use event.target because often draggable shape on the canvas overlaps the target.
     * We can easily find the element(s) at location, but only nodes :-( Unclear how to find links at location
     * (bounding box of a link for testing is bad).
     * The result of that is that links can only be the drop target when dragging from the palette currently.
     * When DnDing shapes on the canvas drop target cannot be a link.
     *
     * Excluded views enables you to choose to filter some possible answers (useful in the case where elements are stacked
     * - e.g. Drag-n-Drop)
     */
    getTargetViewFromEvent(event, x, y, excludeViews = []) {
        if (!x && !y) {
            let l = this.paper.snapToGrid({ x: event.clientX, y: event.clientY });
            x = l.x;
            y = l.y;
        }
        // TODO: See if next code paragraph is needed. Most likely it's just code executed for nothing
        // let elements = this.graph.findModelsFromPoint(joint.g.point(x, y));
        // let underMouse = elements.find(e => !_.isUndefined(excludeViews.find(x => x === this.paper.findViewByModel(e))));
        // if (underMouse) {
        //   return underMouse;
        // }
        let oldVisibility = excludeViews.map(_x => this._hideNode(_x.el));
        let targetElement = document.elementFromPoint(event.clientX, event.clientY);
        excludeViews.forEach((excluded, i) => {
            this._restoreNodeVisibility(excluded.el, oldVisibility[i]);
        });
        return this.paper.findView($(targetElement));
    }
    handleDnDFromPalette(dndEvent) {
        switch (dndEvent.type) {
            case Flo.DnDEventType.DRAG:
                this.handleDragFromPalette(dndEvent);
                break;
            case Flo.DnDEventType.DROP:
                this.handleDropFromPalette(dndEvent);
                break;
            default:
                break;
        }
    }
    handleDragFromPalette(dnDEvent) {
        if (dnDEvent.view && !this.readOnlyCanvas) {
            let location = this.paper.snapToGrid({ x: dnDEvent.event.clientX, y: dnDEvent.event.clientY });
            this.handleNodeDragging(dnDEvent.view, this.getTargetViewFromEvent(dnDEvent.event, location.x, location.y, [dnDEvent.view]), location.x, location.y, Constants.PALETTE_CONTEXT);
        }
    }
    createNode(metadata, props, position) {
        return Shapes.Factory.createNode({
            renderer: this.renderer,
            paper: this.paper,
            metadata: metadata,
            props: props,
            position: position
        });
    }
    createLink(source, target, metadata, props) {
        return Shapes.Factory.createLink({
            renderer: this.renderer,
            paper: this.paper,
            source: source,
            target: target,
            metadata: metadata,
            props: props
        });
    }
    handleDropFromPalette(event) {
        let cellview = event.view;
        let evt = event.event;
        if (this.paper.el === evt.target || $.contains(this.paper.el, evt.target)) {
            if (this.readOnlyCanvas) {
                this.setDragDescriptor(undefined);
            }
            else {
                let metadata = cellview.model.get('metadata');
                let props = cellview.model.attr('props');
                let position = this.paper.snapToGrid({ x: evt.clientX, y: evt.clientY });
                /* Calculate target element before creating the new
                 * element under mouse location. Otherwise target
                 * element would be the newly created element because
                 * it's under the mouse pointer
                 */
                let targetElement = this.getTargetViewFromEvent(evt, position.x, position.y, [event.view]);
                let newNode = this.createNode(metadata, props, position);
                let newView = this.paper.findViewByModel(newNode);
                this.handleNodeDragging(newView, targetElement, position.x, position.y, Constants.PALETTE_CONTEXT);
                this.handleNodeDropping();
            }
        }
    }
    fitToContent(gridWidth, gridHeight, padding, opt) {
        const paper = this.paper;
        if (joint.util.isObject(gridWidth)) {
            // first parameter is an option object
            opt = gridWidth;
            gridWidth = opt.gridWidth || 1;
            gridHeight = opt.gridHeight || 1;
            padding = opt.padding || 0;
        }
        else {
            opt = opt || {};
            gridWidth = gridWidth || 1;
            gridHeight = gridHeight || 1;
            padding = padding || 0;
        }
        const paddingJson = joint.util.normalizeSides(padding);
        // Calculate the paper size to accomodate all the graph's elements.
        const bbox = joint.V(paper.viewport).getBBox();
        const currentScale = paper.scale();
        const currentTranslate = paper.translate();
        bbox.x *= currentScale.sx;
        bbox.y *= currentScale.sy;
        bbox.width *= currentScale.sx;
        bbox.height *= currentScale.sy;
        let calcWidth = Math.max((bbox.width + bbox.x) / gridWidth, 1) * gridWidth;
        let calcHeight = Math.max((bbox.height + bbox.y) / gridHeight, 1) * gridHeight;
        let tx = 0;
        let ty = 0;
        if ((opt.allowNewOrigin === 'negative' && bbox.x < 0) || (opt.allowNewOrigin === 'positive' && bbox.x >= 0) || opt.allowNewOrigin === 'any') {
            tx = (-bbox.x / gridWidth) * gridWidth;
            tx += paddingJson.left;
        }
        else if (opt.allowNewOrigin === 'same') {
            tx = currentTranslate.tx;
        }
        calcWidth += tx;
        if ((opt.allowNewOrigin === 'negative' && bbox.y < 0) || (opt.allowNewOrigin === 'positive' && bbox.y >= 0) || opt.allowNewOrigin === 'any') {
            ty = (-bbox.y / gridHeight) * gridHeight;
            ty += paddingJson.top;
        }
        else if (opt.allowNewOrigin === 'same') {
            ty = currentTranslate.ty;
        }
        calcHeight += ty;
        calcWidth += paddingJson.right;
        calcHeight += paddingJson.bottom;
        // Make sure the resulting width and height are greater than minimum.
        calcWidth = Math.max(calcWidth, opt.minWidth || 0);
        calcHeight = Math.max(calcHeight, opt.minHeight || 0);
        // Make sure the resulting width and height are lesser than maximum.
        calcWidth = Math.min(calcWidth, opt.maxWidth || Number.MAX_VALUE);
        calcHeight = Math.min(calcHeight, opt.maxHeight || Number.MAX_VALUE);
        const dimensionChange = calcWidth !== paper.options.width || calcHeight !== paper.options.height;
        const originChange = tx !== currentTranslate.tx || ty !== currentTranslate.ty;
        // Change the dimensions only if there is a size discrepency or an origin change
        if (originChange) {
            paper.translate(tx, ty);
        }
        if (dimensionChange) {
            paper.setDimensions(calcWidth, calcHeight);
        }
    }
    autosizePaper() {
        let parent = $('#paper-container', this.element.nativeElement);
        const parentWidth = parent.innerWidth();
        const parentHeight = parent.innerHeight();
        this.fitToContent(this.gridSize, this.gridSize, this.paperPadding, {
            minWidth: parentWidth - Flo.SCROLLBAR_WIDTH,
            minHeight: parentHeight - Flo.SCROLLBAR_WIDTH,
            allowNewOrigin: 'same'
        });
    }
    fitToPage() {
        let parent = $('#paper-container', this.element.nativeElement);
        let minScale = this.minZoom / 100;
        let maxScale = 2;
        const parentWidth = parent.innerWidth();
        const parentHeight = parent.innerHeight();
        this.paper.scaleContentToFit({
            padding: this.paperPadding,
            minScaleX: minScale,
            minScaleY: minScale,
            maxScaleX: maxScale,
            maxScaleY: maxScale,
            fittingBBox: { x: 0, y: 0, width: parentWidth - Flo.SCROLLBAR_WIDTH, height: parentHeight - Flo.SCROLLBAR_WIDTH }
        });
        /**
         * Size the canvas appropriately and allow origin movement
         */
        this.fitToContent(this.gridSize, this.gridSize, this.paperPadding, {
            minWidth: parentWidth,
            minHeight: parentHeight,
            maxWidth: parentWidth,
            maxHeight: parentHeight,
            allowNewOrigin: 'any'
        });
    }
    get zoomPercent() {
        return Math.round(this.paper.scale().sx * 100);
    }
    set zoomPercent(percent) {
        if (!isNaN(percent)) {
            if (percent < this.minZoom) {
                percent = this.minZoom;
            }
            else if (percent >= this.maxZoom) {
                percent = this.maxZoom;
            }
            else {
                if (percent <= 0) {
                    percent = 0.00001;
                }
            }
            this.paper.scale(percent / 100, percent / 100);
        }
    }
    get gridSize() {
        return this._gridSize;
    }
    set gridSize(size) {
        if (!isNaN(size) && size >= 1) {
            this._gridSize = size;
            if (this.paper) {
                this.paper.setGridSize(size);
            }
        }
    }
    validateContent() {
        return new Promise(resolve => {
            if (this.editor && this.editor.validate) {
                return this.editor
                    .validate(this.graph, this.dsl, this.editorContext)
                    .then(allMarkers => {
                    this.graph.getCells()
                        .forEach((cell) => this.markElement(cell, allMarkers.has(cell.id) ? allMarkers.get(cell.id) : []));
                    this.validationMarkers.emit(allMarkers);
                    this.contentValidated.emit(true);
                    resolve();
                });
            }
            else {
                resolve();
            }
        });
    }
    markElement(cell, markers) {
        cell.set('markers', markers);
        // Old legacy code below consider removing
        let errorMessages = markers.map(m => m.message);
        let errorCell = cell.getEmbeddedCells().find((e) => e.attr('./kind') === Constants.ERROR_DECORATION_KIND);
        if (errorCell) {
            if (errorMessages.length === 0) {
                errorCell.remove();
            }
            else {
                // Without rewrite we merge this list with existing errors
                errorCell.attr('messages', errorMessages, { rewrite: true });
            }
        }
        else if (errorMessages.length > 0) {
            let error = Shapes.Factory.createDecoration({
                renderer: this.renderer,
                paper: this.paper,
                parent: cell,
                kind: Constants.ERROR_DECORATION_KIND,
                messages: errorMessages
            });
            if (error) {
                const view = this.paper.findViewByModel(error);
                view.setInteractivity(false);
            }
        }
    }
    doLayout() {
        if (this.renderer && this.renderer.layout) {
            return this.renderer.layout(this.paper);
        }
    }
    set dsl(dslText) {
        if (this._dslText !== dslText) {
            this._dslText = dslText;
            this.textToGraphEventEmitter.emit();
        }
    }
    get dsl() {
        return this._dslText;
    }
    /**
     * Ask the server to parse the supplied text into a JSON graph of nodes and links,
     * then update the view based on that new information.
     */
    updateGraphRepresentation() {
        Logger.debug(`Updating graph to represent '${this._dslText}'`);
        if (this.metamodel && this.metamodel.textToGraph) {
            return this.metamodel.textToGraph(this.editorContext, this._dslText).then(() => {
                this.textToGraphConversionCompleted.next();
                return this.validateContent();
            });
        }
        else {
            this.textToGraphConversionCompleted.next();
            return this.validateContent();
        }
    }
    updateTextRepresentation() {
        if (this.metamodel && this.metamodel.graphToText) {
            return this.metamodel.graphToText(this.editorContext).then(text => {
                if (this._dslText !== text) {
                    this._dslText = text;
                    this.dslChange.emit(text);
                }
                this.graphToTextConversionCompleted.next();
                return this.validateContent();
            })
                .catch(error => {
                // Validation may reveal why the graph couldn't be
                // converted so let it run
                this.graphToTextConversionCompleted.next();
                return this.validateContent();
            });
        }
        else {
            this.graphToTextConversionCompleted.next();
            return this.validateContent();
        }
    }
    initMetamodel() {
        this.metamodel.load().then(data => {
            this.updateGraphRepresentation();
            let textSyncSubscription = this.graphToTextEventEmitter.pipe(debounceTime(100)).subscribe(() => {
                if (this._graphToTextSyncEnabled) {
                    this.updateTextRepresentation();
                }
            });
            this._disposables.add(Disposable.create(() => textSyncSubscription.unsubscribe()));
            // Setup content validated event emitter. Emit not validated when graph to text conversion required
            let graphValidatedSubscription1 = this.graphToTextEventEmitter.subscribe(() => this.contentValidated.emit(false));
            this._disposables.add(Disposable.create(() => graphValidatedSubscription1.unsubscribe));
            // let validationSubscription = this.validationEventEmitter.pipe(debounceTime(100)).subscribe(() => this.validateGraph());
            // this._disposables.add(Disposable.create(() => validationSubscription.unsubscribe()));
            let graphSyncSubscription = this.textToGraphEventEmitter.pipe(debounceTime(300)).subscribe(() => this.updateGraphRepresentation());
            this._disposables.add(Disposable.create(() => graphSyncSubscription.unsubscribe()));
            // Setup content validated event emitter. Emit not validated when text to graph conversion required
            let graphValidatedSubscription2 = this.textToGraphEventEmitter.subscribe(() => this.contentValidated.emit(false));
            this._disposables.add(Disposable.create(() => graphValidatedSubscription2.unsubscribe));
            if (this.editor && this.editor.setDefaultContent) {
                this.editor.setDefaultContent(this.editorContext, data);
            }
        });
    }
    initGraph() {
        this.graph = new joint.dia.Graph();
        this.graph.set('type', Constants.CANVAS_CONTEXT);
        this.graph.set('paperPadding', this.paperPadding);
    }
    handleNodeCreation(node) {
        node.on('change:size', this._resizeHandler);
        node.on('change:position', this._resizeHandler);
        if (node.get('metadata')) {
            node.on('change:attrs', (cell, attrs, changeData) => {
                let propertyPath = changeData ? changeData.propertyPath : undefined;
                if (propertyPath) {
                    let propAttr = propertyPath.substr(propertyPath.indexOf('/') + 1);
                    if (propAttr.indexOf('props') === 0 ||
                        (this.renderer && this.renderer.isSemanticProperty && this.renderer.isSemanticProperty(propAttr, node))) {
                        this.performGraphToTextSyncing();
                    }
                    if (this.renderer && this.renderer.refreshVisuals) {
                        this.renderer.refreshVisuals(node, propAttr, this.paper);
                    }
                }
            });
            node.on('change:metadata', (cell, attrs, changeData) => {
                let propertyPath = changeData ? changeData.propertyPath : undefined;
                if (propertyPath && this.renderer && this.renderer.refreshVisuals) {
                    this.renderer.refreshVisuals(node, propertyPath, this.paper);
                }
            });
        }
        node.on('change:markers', () => {
            if (this.renderer && this.renderer.markersChanged) {
                this.renderer.markersChanged(node, this.paper);
            }
        });
    }
    /**
     * Forwards a link event occurrence to any handlers in the editor service, if they are defined. Event examples
     * are 'change:source', 'change:target'.
     */
    handleLinkEvent(event, link) {
        if (this.renderer && this.renderer.handleLinkEvent) {
            this.renderer.handleLinkEvent(this.editorContext, event, link);
        }
    }
    handleLinkCreation(link) {
        link.on('change:source', (l) => {
            this.autosizePaper();
            let newSourceId = l.get('source').id;
            let oldSourceId = l.previous('source').id;
            if (newSourceId !== oldSourceId) {
                this.performGraphToTextSyncing();
            }
            this.handleLinkEvent('change:source', l);
        });
        link.on('change:target', (l) => {
            this.autosizePaper();
            let newTargetId = l.get('target').id;
            let oldTargetId = l.previous('target').id;
            if (newTargetId !== oldTargetId) {
                this.performGraphToTextSyncing();
            }
            this.handleLinkEvent('change:target', l);
        });
        link.on('change:vertices', this._resizeHandler);
        link.on('change:attrs', (cell, attrs, changeData) => {
            let propertyPath = changeData ? changeData.propertyPath : undefined;
            if (propertyPath) {
                let propAttr = propertyPath.substr(propertyPath.indexOf('/') + 1);
                if (propAttr.indexOf('props') === 0 ||
                    (this.renderer && this.renderer.isSemanticProperty && this.renderer.isSemanticProperty(propAttr, link))) {
                    let sourceId = link.get('source').id;
                    let targetId = link.get('target').id;
                    this.performGraphToTextSyncing();
                }
                if (this.renderer && this.renderer.refreshVisuals) {
                    this.renderer.refreshVisuals(link, propAttr, this.paper);
                }
            }
        });
        link.on('change:metadata', (cell, attrs, changeData) => {
            let propertyPath = changeData ? changeData.propertyPath : undefined;
            if (propertyPath && this.renderer && this.renderer.refreshVisuals) {
                this.renderer.refreshVisuals(link, propertyPath, this.paper);
            }
        });
        this.paper.findViewByModel(link).on('link:options', () => this.handleLinkEvent('options', link));
        if (this.readOnlyCanvas) {
            link.attr('.link-tools/display', 'none');
        }
        this.handleLinkEvent('add', link);
    }
    initGraphListeners() {
        this.graph.on('add', (element) => {
            if (element instanceof joint.dia.Link) {
                this.handleLinkCreation(element);
            }
            else if (element instanceof joint.dia.Element) {
                this.handleNodeCreation(element);
            }
            if (element.get('type') === joint.shapes.flo.NODE_TYPE || element.get('type') === joint.shapes.flo.LINK_TYPE) {
                this.performGraphToTextSyncing();
            }
            this.autosizePaper();
        });
        this.graph.on('remove', (element) => {
            if (element instanceof joint.dia.Link) {
                this.handleLinkEvent('remove', element);
            }
            if (this.selection && this.selection.model === element) {
                this.selection = undefined;
            }
            if (element.isLink()) {
                window.setTimeout(() => this.performGraphToTextSyncing(), 100);
            }
            else if (element.get('type') === joint.shapes.flo.NODE_TYPE) {
                this.performGraphToTextSyncing();
            }
            this.autosizePaper();
        });
        // Set if link is fan-routed. Should be called before routing call
        this.graph.on('change:vertices', (link, changed, opt) => {
            if (opt.fanRouted) {
                link.set('fanRouted', true);
            }
            else {
                link.unset('fanRouted');
            }
        });
        // adjust vertices when a cell is removed or its source/target was changed
        this.graph.on('add remove change:source change:target change:vertices change:position', _.partial(Utils.fanRoute, this.graph));
        this.graph.on('startDeletion', (cell) => {
            if (this.editor && this.editor.preDelete) {
                if (this.editor.preDelete(this.editorContext, this.selection.model)) {
                    cell.remove();
                }
            }
            else {
                cell.remove();
            }
        });
    }
    initPaperListeners() {
        // https://stackoverflow.com/questions/20463533/how-to-add-an-onclick-event-to-a-joint-js-element
        this.paper.on('cell:pointerup', (cellView) => {
            if (!this.readOnlyCanvas) {
                this.selection = cellView;
            }
        });
        this.paper.on('blank:pointerdown', () => {
            this.selection = undefined;
        });
        this.paper.on('scale', this._resizeHandler);
        this.paper.on('all', function () {
            if (Utils.isCustomPaperEvent(arguments)) {
                arguments[2].trigger.apply(arguments[2], [arguments[0], arguments[1], arguments[3], arguments[4]]);
            }
        });
        this.paper.on('dragging-node-over-canvas', (dndEvent) => {
            let location = this.paper.snapToGrid({ x: dndEvent.event.clientX, y: dndEvent.event.clientY });
            switch (dndEvent.type) {
                case Flo.DnDEventType.DRAG:
                    this.handleNodeDragging(dndEvent.view, this.getTargetViewFromEvent(dndEvent.event, location.x, location.y, [dndEvent.view]), location.x, location.y, Constants.CANVAS_CONTEXT);
                    break;
                case Flo.DnDEventType.DROP:
                    this.handleNodeDropping();
                    break;
                default:
                    break;
            }
        });
        // JointJS now no longer grabs focus if working in a paper element - crude...
        // $('#flow-view', this.element.nativeElement).on('mousedown', () => {
        // $('#palette-filter-textfield', this.element.nativeElement).focus();
        // });
    }
    initPaper() {
        let options = {
            el: $('#paper', this.element.nativeElement),
            gridSize: this._gridSize,
            drawGrid: true,
            model: this.graph,
            elementView: this.renderer && this.renderer.getNodeView ? this.renderer.getNodeView() : joint.shapes.flo.ElementView /*joint.dia.ElementView*/,
            linkView: this.renderer && this.renderer.getLinkView ? this.renderer.getLinkView() : joint.shapes.flo.LinkView,
            // Enable link snapping within 25px lookup radius
            snapLinks: { radius: 25 },
            defaultLink: /*this.renderer && this.renderer.createDefaultLink ? this.renderer.createDefaultLink: new joint.shapes.flo.Link*/ (cellView, magnet) => {
                if (this.renderer && this.renderer.createLink) {
                    let linkEnd = {
                        id: cellView.model.id
                    };
                    if (magnet) {
                        linkEnd.selector = cellView.getSelector(magnet, undefined);
                    }
                    if (magnet.getAttribute('port')) {
                        linkEnd.port = magnet.getAttribute('port');
                    }
                    if (magnet.getAttribute('port') === 'input') {
                        return this.renderer.createLink(undefined, linkEnd);
                    }
                    else {
                        return this.renderer.createLink(linkEnd, undefined);
                    }
                }
                else {
                    return new joint.shapes.flo.Link();
                }
            },
            // decide whether to create a link if the user clicks a magnet
            validateMagnet: (cellView, magnet) => {
                if (this.readOnlyCanvas) {
                    return false;
                }
                else {
                    if (this.editor && this.editor.validatePort) {
                        return this.editor.validatePort(this.editorContext, cellView, magnet);
                    }
                    else {
                        return true;
                    }
                }
            },
            interactive: (cellView, event) => {
                if (this.readOnlyCanvas) {
                    return false;
                }
                else {
                    if (this.editor && this.editor.interactive) {
                        if (typeof this.editor.interactive === 'function') {
                            // Type for interactive is wrong in JointJS have to cast to <any>
                            return this.editor.interactive(cellView, event);
                        }
                        else {
                            return this.editor.interactive;
                        }
                    }
                    return true;
                }
            },
            highlighting: this.editor && this.editor.highlighting ? this.editor.highlighting : {
                'default': {
                    name: 'addClass',
                    options: {
                        className: 'highlighted'
                    }
                }
            },
            markAvailable: true
        };
        if (this.renderer && this.renderer.getLinkAnchorPoint) {
            options.linkConnectionPoint = this.renderer.getLinkAnchorPoint;
        }
        if (this.editor && this.editor.validateLink) {
            const self = this;
            options.validateConnection = (cellViewS, magnetS, cellViewT, magnetT, end, linkView) => self.editor.validateLink(this.editorContext, cellViewS, magnetS, cellViewT, magnetT, end === 'source', linkView);
        }
        // The paper is what will represent the graph on the screen
        this.paper = new joint.dia.Paper(options);
        this._disposables.add(Disposable.create(() => this.paper.remove()));
        this.paper.options.highlighterNamespace['addParentClass'] = {
            /**
             * @param {joint.dia.CellView} cellView
             * @param {Element} magnetEl
             * @param {object=} opt
             */
            highlight(cellView, magnetEl, opt) {
                opt = opt || {};
                const className = opt.className || this.className;
                joint.V(magnetEl.parentElement).addClass(className);
            },
            /**
             * @param {joint.dia.CellView} cellView
             * @param {Element} magnetEl
             * @param {object=} opt
             */
            unhighlight(cellView, magnetEl, opt) {
                opt = opt || {};
                const className = opt.className || this.className;
                joint.V(magnetEl.parentElement).removeClass(className);
            }
        };
    }
    updatePaletteReadyState(ready) {
        this.paletteReady.next(ready);
    }
}
EditorComponent.decorators = [
    { type: Component, args: [{
                selector: 'flo-editor',
                template: "<ng-content select=\"[header]\"></ng-content>\n<div id=\"flow-view\" class=\"flow-view\" style=\"position:relative\">\n  <div id=\"canvas\" class=\"canvas\" style=\"position:relative; display: block; width: 100%; height: 100%;\">\n    <div *ngIf=\"!noPalette\" id=\"palette-container\" class=\"palette-container\" style=\"overflow:hidden;\">\n      <flo-palette [metamodel]=\"metamodel\" [renderer]=\"renderer\" [paletteSize]=\"paletteSize\"\n                   [paletteEntryPadding]=\"paletteEntryPadding || {width: 12, height: 12}\"\n                   (onPaletteEntryDrop)=\"handleDnDFromPalette($event)\"\n                    (paletteReady)=\"updatePaletteReadyState($event)\"\n                    (paletteFocus)=\"graphToTextSync=true\"\n                    [searchFilterPlaceHolder]=\"searchFilterPlaceHolder\">\n      </flo-palette>\n    </div>\n\n    <div id=\"sidebar-resizer\" *ngIf=\"!noPalette\"\n      resizer\n      [splitSize]=\"paletteSize\"\n      (sizeChange)=\"paletteSize = $event\"\n      [resizerWidth]=\"6\"\n      [resizerLeft]=\"'#palette-container'\"\n      [resizerRight]=\"'#paper-container'\">\n    </div>\n\n      <flo-editor-paper tabindex=\"0\" (onDelete)=\"deleteSelected()\" (onBlur)=\"selection = null\"\n                        (onProperties)=\"onPropertiesHandle()\">\n        <div id=\"paper\" class=\"paper\" tabindex=\"0\" style=\"overflow: hidden; position: absolute; display: block; height:100%; width:100%; overflow:auto;\"></div>\n        <ng-content select=\"[canvas]\"></ng-content>\n      </flo-editor-paper>\n\n      <span class=\"canvas-controls-container\" ng-if=\"canvasControls\">\n        <table ng-if=\"canvasControls.zoom\" class=\"canvas-control zoom-canvas-control\">\n          <tbody>\n            <tr>\n              <td>\n                <input class=\"zoom-canvas-input canvas-control zoom-canvas-control\" type=\"text\"\n                       data-inline=\"true\" [(ngModel)]=\"zoomPercent\"\n                       size=\"3\">\n              </td>\n              <td>\n                <label class=\"canvas-control zoom-canvas-label\">%</label>\n              </td>\n              <td>\n                <input type=\"range\" data-inline=\"true\" [(ngModel)]=\"zoomPercent\"\n                       [step]=\"zoomStep\"\n                       [max]=\"maxZoom\" [min]=\"minZoom\" data-type=\"range\"\n                       name=\"range\" class=\"canvas-control zoom-canvas-control\">\n              </td>\n            </tr>\n          </tbody>\n        </table>\n      </span>\n\n  </div>\n</div>\n<ng-content select=\"[footer]\"></ng-content>\n\n",
                encapsulation: ViewEncapsulation.None,
                styles: ["flo-view{-moz-user-select:none;-ms-user-select:none;-o-user-select:none;-webkit-user-select:none;background-color:#eee;font-family:Varela Round,sans-serif;height:100%;margin:0;user-select:none;width:100%}.canvas{border:1px solid #6db33f;border-radius:2px;margin-top:3px}.paper{background-color:#fff;padding:0}#sidebar-resizer{background-color:#34302d;bottom:0;cursor:e-resize;position:absolute;top:0;width:6px}#palette-container{background-color:#eee;left:0}#palette-container,#paper-container{bottom:0;overflow:auto;position:absolute;top:0}#paper-container{background-color:#fff;color:#fff;outline:none;right:0}.node-tooltip .tooltip-description{margin-bottom:5px;margin-left:0;margin-top:5px}.node-tooltip{background-color:#34302d;border:1px solid #333;border-radius:5px;color:#fff;display:none;font-family:Varela Round,sans-serif;font-size:19px;padding:5px;position:absolute;z-index:100}.tooltip-title-type{font-size:24px;font-weight:700}.tooltip-title-group{font-size:20px;font-style:italic;padding-left:5px}.node-tooltip-option-name{font-family:monospace;font-size:17px;font-weight:700;padding-right:20px}.node-tooltip-option-description{font-family:Varela Round,sans-serif;font-size:18px}.error-tooltip p{color:#fff;margin-bottom:5px;margin-left:0;margin-top:5px}.error-tooltip{background-color:red;border:1px solid #333;border-radius:5px;color:#fff;display:none;font-family:Varela Round,sans-serif;font-size:20px;padding:5px;position:absolute;z-index:100}.canvas-controls-container{position:absolute;right:15px;top:5px}.canvas-control{background:transparent;font-family:Varela Round,sans-serif;font-size:11px;margin:0;vertical-align:middle}.zoom-canvas-control{border:0;margin:0;outline:none;padding:0}.zoom-canvas-input{background-color:transparent;color:#000;font-weight:700;text-align:right}.zoom-canvas-label{color:#000;padding-right:4px}.highlighted{outline:none}.joint-type-handle{cursor:pointer}.available-magnet{stroke-width:3}.link{fill:none;stroke:#ccc;stroke-width:1.5px}.link-tools .tool-options{display:none}.link-tools .tool-options circle{fill:transparent;stroke:transparent}.link-tools .tool-options path{fill:#000;stroke:#000}.link-tools .tool-remove circle{fill:red;stroke:red}.link-tools .tool-remove path{fill:#fff;stroke:#fff}.link-tools-container{fill:transparent;stroke-width:0}"]
            },] }
];
EditorComponent.ctorParameters = () => [
    { type: ElementRef }
];
EditorComponent.propDecorators = {
    metamodel: [{ type: Input }],
    renderer: [{ type: Input }],
    editor: [{ type: Input }],
    paletteSize: [{ type: Input }],
    paletteSizeChange: [{ type: Output }],
    searchFilterPlaceHolder: [{ type: Input }],
    paletteEntryPadding: [{ type: Input }],
    minZoom: [{ type: Input }],
    maxZoom: [{ type: Input }],
    zoomStep: [{ type: Input }],
    paperPadding: [{ type: Input }],
    floApi: [{ type: Output }],
    validationMarkers: [{ type: Output }],
    contentValidated: [{ type: Output }],
    dslChange: [{ type: Output }],
    onProperties: [{ type: Output }],
    dsl: [{ type: Input }]
};
//# sourceMappingURL=data:application/json;base64,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