import { Component, ElementRef, Input, Output, EventEmitter, Inject, ViewEncapsulation } from '@angular/core';
import { Subject } from 'rxjs';
import { debounceTime } from 'rxjs/operators';
import { Flo } from '../shared/flo-common';
import '../shared/shapes';
import { Shapes, Constants } from '../shared/shapes';
import { DOCUMENT } from '@angular/common';
import * as _$ from 'jquery';
import { Logger } from '../shared/logger';
const joint = Flo.joint;
const $ = _$;
const DEBOUNCE_TIME = 300;
export class Palette {
    constructor(element, document) {
        this.element = element;
        this.document = document;
        this._metamodelListener = {
            metadataError: (data) => { },
            metadataAboutToChange: () => { },
            metadataChanged: () => {
                if (this.initialized && this.metamodel) {
                    this.metamodel.load().then(metamodel => this.buildPalette(metamodel));
                }
            }
        };
        this.initialized = false;
        this._filterText = '';
        this.filterTextModel = new Subject();
        this.paletteEntryPadding = { width: 12, height: 12 };
        this.searchFilterPlaceHolder = 'Search...';
        this.onPaletteEntryDrop = new EventEmitter();
        this.paletteReady = new EventEmitter();
        this.paletteFocus = new EventEmitter();
        this.mouseMoveHanlder = (e) => this.handleDrag(e);
        this.mouseUpHanlder = (e) => this.handleMouseUp(e);
        this.paletteGraph = new joint.dia.Graph();
        this.paletteGraph.set('type', Constants.PALETTE_CONTEXT);
        this._filterText = '';
        this.closedGroups = new Set();
    }
    set paletteSize(size) {
        Logger.debug('Palette Size: ' + size);
        if (this._paletteSize !== size) {
            this._paletteSize = size;
            if (this.palette) {
                this.layout();
            }
        }
    }
    onFocus() {
        this.paletteFocus.emit();
    }
    ngOnInit() {
        let element = $('#palette-paper', this.element.nativeElement);
        // Create the paper for the palette using the specified element view
        this.palette = new joint.dia.Paper({
            el: element,
            gridSize: 1,
            model: this.paletteGraph,
            height: $(this.element.nativeElement.parentNode).height(),
            width: $(this.element.nativeElement.parentNode).width(),
            elementView: this.getPaletteView(this.renderer && this.renderer.getNodeView ? this.renderer.getNodeView() : joint.dia.ElementView),
            interactive: false
        });
        this.palette.on('cell:pointerup', (cellview, evt) => {
            if (this.viewBeingDragged) {
                this.trigger({
                    type: Flo.DnDEventType.DROP,
                    view: this.viewBeingDragged,
                    event: evt
                });
                this.viewBeingDragged = undefined;
            }
            this.clickedElement = undefined;
            $('#palette-floater').remove();
            if (this.floaterpaper) {
                this.floaterpaper.remove();
            }
        });
        // Toggle the header open/closed on a click
        this.palette.on('cell:pointerclick', (cellview, event) => {
            // TODO [design][palette] should the user need to click on the arrow rather than anywhere on the header?
            // Click position within the element would be: evt.offsetX, evt.offsetY
            const cell = cellview.model;
            if (cell.attributes.header) {
                // Toggle the header open/closed
                // if (cell.get('isOpen')) {
                //   this.rotateClosed(cell);
                // } else {
                //   this.rotateOpen(cell);
                // }
            }
            // TODO [palette] ensure other mouse handling events do nothing for headers
            // TODO [palette] move 'metadata' field to the right place (not inside attrs I think)
        });
        $(this.document).on('mouseup', this.mouseUpHanlder);
        if (this.metamodel) {
            this.metamodel.load().then(data => {
                this.buildPalette(data);
                // Add listener to metamodel
                if (this.metamodel && this.metamodel.subscribe) {
                    this.metamodel.subscribe(this._metamodelListener);
                }
                // Add debounced listener to filter text changes
                this.filterTextModel
                    .pipe(debounceTime(DEBOUNCE_TIME))
                    .subscribe((value) => this.layout());
                this.initialized = true;
            });
        }
        else {
            Logger.error('No Metamodel service specified for palette!');
        }
        this._paletteSize = this._paletteSize || $(this.element.nativeElement.parentNode).width();
    }
    ngOnDestroy() {
        if (this.metamodel && this.metamodel.unsubscribe) {
            this.metamodel.unsubscribe(this._metamodelListener);
        }
        $(this.document).off('mouseup', this.mouseUpHanlder);
        this.palette.remove();
    }
    createPaletteGroup(title, isOpen) {
        const paletteRenderer = this.renderer && this.renderer.getPaletteRenderer ? this.renderer.getPaletteRenderer() : {
            createGroupHeader: (titleStr, isOpenParam) => {
                const header = new joint.shapes.flo.PaletteGroupHeader({ attrs: { text: { text: titleStr } } });
                if (!isOpenParam) {
                    header.attr({ 'path': { 'transform': 'rotate(-90,15,13)' } });
                }
                return header;
            },
            onClose: (groupView) => this.rotateClosed(groupView.model),
            onOpen: (groupView) => this.rotateOpen(groupView.model)
        };
        let newGroupHeader = paletteRenderer.createGroupHeader(title, isOpen);
        if (!isOpen) {
            newGroupHeader.set('isOpen', false);
        }
        newGroupHeader.set('header', title);
        this.paletteGraph.addCell(newGroupHeader);
        const view = this.palette.findViewByModel(newGroupHeader);
        view.on('cell:pointerclick', () => {
            if (newGroupHeader.get('isOpen')) {
                if (typeof paletteRenderer.onClose === 'function') {
                    paletteRenderer.onClose(view).then(() => {
                        newGroupHeader.set('isOpen', false);
                        this.closedGroups.add(newGroupHeader.get('header'));
                        this.layout();
                    });
                }
                else {
                    newGroupHeader.set('isOpen', false);
                    this.closedGroups.add(newGroupHeader.get('header'));
                    this.layout();
                }
            }
            else {
                if (typeof paletteRenderer.onOpen === 'function') {
                    paletteRenderer.onOpen(view).then(() => {
                        newGroupHeader.set('isOpen', true);
                        this.closedGroups.delete(newGroupHeader.get('header'));
                        this.layout();
                    });
                }
                else {
                    newGroupHeader.set('isOpen', true);
                    this.closedGroups.delete(newGroupHeader.get('header'));
                    this.layout();
                }
            }
        });
        return newGroupHeader;
    }
    createPaletteEntry(title, metadata) {
        return Shapes.Factory.createNode({
            renderer: this.renderer,
            paper: this.palette,
            metadata: metadata
        });
    }
    buildPalette(metamodel) {
        let startTime = new Date().getTime();
        this.paletteReady.emit(false);
        this.paletteGraph.clear();
        let groupAdded = new Set();
        let parentWidth = this._paletteSize - Flo.SCROLLBAR_WIDTH;
        Logger.debug(`Parent Width: ${parentWidth}`);
        // The field closedGroups tells us which should not be shown
        // Work out the list of active groups/nodes based on the filter text
        this.metamodel.groups().forEach(group => {
            if (metamodel && metamodel.has(group)) {
                Array.from(metamodel.get(group).keys()).sort().forEach(name => {
                    let node = metamodel.get(group).get(name);
                    if (node) {
                        if (!groupAdded.has(group)) {
                            this.createPaletteGroup(group, !this.closedGroups.has(group));
                            groupAdded.add(group);
                        }
                        if (!(node.metadata && node.metadata.noPaletteEntry)) {
                            this.createPaletteEntry(name, node);
                        }
                    }
                });
            }
        });
        this.noMacthesFoundNode = new joint.shapes.flo.NoMatchesFound();
        this.palette.model.addCell(this.noMacthesFoundNode);
        this.layout();
        this.paletteReady.emit(true);
        Logger.debug('buildPalette took ' + (new Date().getTime() - startTime) + 'ms');
    }
    layout() {
        let startTime = new Date().getTime();
        let filterText = this.filterText;
        if (filterText) {
            filterText = filterText.toLowerCase();
        }
        let paletteNodes = [];
        let parentWidth = this._paletteSize - Flo.SCROLLBAR_WIDTH;
        Logger.debug(`Parent Width: ${parentWidth}`);
        const presentGroups = new Set();
        this.palette.model.getCells().forEach((cell) => {
            const metadata = cell.get('metadata');
            if (cell.get('header')) {
                paletteNodes.push(cell);
            }
            else if (metadata && metadata.group && metadata.name
                && (!filterText
                    || metadata.group.toLocaleLowerCase().indexOf(filterText) >= 0
                    || metadata.name.toLocaleLowerCase().indexOf(filterText) >= 0)) {
                if (!this.closedGroups.has(metadata.group)) {
                    cell.attr('./display', 'block');
                    cell.removeAttr('./display');
                    paletteNodes.push(cell);
                }
                else {
                    cell.attr('./display', 'none');
                }
                presentGroups.add(metadata.group);
            }
            else {
                if (cell === this.noMacthesFoundNode) {
                }
                else {
                    cell.attr('./display', 'none');
                }
            }
        });
        // Clean group headers
        const filteredGroupHeaders = [];
        paletteNodes.forEach(cell => {
            if (cell.get('header')) {
                if (presentGroups.has(cell.get('header'))) {
                    cell.attr('./display', 'block');
                    cell.removeAttr('./display');
                    filteredGroupHeaders.push(cell);
                }
                else {
                    cell.attr('./display', 'none');
                }
            }
            else {
                filteredGroupHeaders.push(cell);
            }
        });
        paletteNodes = filteredGroupHeaders;
        // Check if last group is empty
        const previous = paletteNodes.length > 0 ? paletteNodes[paletteNodes.length - 1] : undefined;
        // If previous is a paletter header node as well then the previous header had no nodes under it and we can hide it and remove from paletteNodes aeeay
        if (previous && previous.get('header') && !this.closedGroups.has(previous.get('header'))) {
            paletteNodes.pop().attr('./display', 'none');
        }
        let cellWidth = 0, cellHeight = 0;
        // Determine the size of the palette entry cell (width and height)
        paletteNodes.forEach(pnode => {
            var _a;
            if ((_a = pnode.get('metadata')) === null || _a === void 0 ? void 0 : _a.name) {
                const elementSize = this.palette.findViewByModel(pnode).getBBox();
                let dimension = {
                    width: elementSize.width,
                    height: elementSize.height
                };
                if (cellWidth < dimension.width) {
                    cellWidth = dimension.width;
                }
                if (cellHeight < dimension.height) {
                    cellHeight = dimension.height;
                }
            }
        });
        // Adjust the palette entry cell size with paddings.
        cellWidth += 2 * this.paletteEntryPadding.width;
        cellHeight += 2 * this.paletteEntryPadding.height;
        // Align palette entries row to be at the center
        let startX = parentWidth >= cellWidth ? (parentWidth - Math.floor(parentWidth / cellWidth) * cellWidth) / 2 : 0;
        let xpos = startX;
        let ypos = 0;
        let prevNode;
        // Layout palette entry nodes
        paletteNodes.forEach(pnode => {
            var _a;
            const elementSize = this.palette.findViewByModel(pnode).getBBox();
            let dimension = {
                width: elementSize.width,
                height: elementSize.height
            };
            if (pnode.get('header')) { //attributes.attrs.header) {
                // Palette entry header
                xpos = startX;
                if (ypos) {
                    ypos += this.paletteEntryPadding.height;
                }
                pnode.set('size', { width: parentWidth, height: pnode.get('size').height || 30 });
                pnode.set('position', { x: 0, y: ypos });
                ypos += dimension.height + this.paletteEntryPadding.height;
            }
            else {
                // Palette entry element
                if (xpos + cellWidth > parentWidth) {
                    // Not enough real estate to place entry in a row - reset x position and leave the y pos which is next line
                    xpos = startX;
                    pnode.set('position', { x: xpos + (cellWidth - dimension.width) / 2, y: ypos + (cellHeight - dimension.height) / 2 });
                }
                else {
                    // Enough real estate to place entry in a row - adjust y position
                    if (prevNode && ((_a = prevNode.get('metadata')) === null || _a === void 0 ? void 0 : _a.name)) {
                        ypos -= cellHeight;
                    }
                    pnode.set('position', { x: xpos + (cellWidth - dimension.width) / 2, y: ypos + (cellHeight - dimension.height) / 2 });
                }
                // increment x position and y position (can be reorganized)
                xpos += cellWidth;
                ypos += cellHeight;
            }
            prevNode = pnode;
        });
        this.noMacthesFoundNode.set('size', { width: parentWidth, height: this.noMacthesFoundNode.get('size').height || 30 });
        this.noMacthesFoundNode.set('position', { x: 0, y: 0 });
        if (paletteNodes.length === 0 && filterText) {
            // There is a filter present but everything is filtered out
            // Show no matches found node
            this.noMacthesFoundNode.attr('./display', 'block');
            this.noMacthesFoundNode.removeAttr('./display');
            ypos = this.noMacthesFoundNode.get('size').height;
        }
        else {
            // Hide no matches node in all other cases
            this.noMacthesFoundNode.attr('./display', 'none');
        }
        this.palette.setDimensions(parentWidth, ypos);
        Logger.debug('buildPalette layout ' + (new Date().getTime() - startTime) + 'ms');
    }
    set filterText(text) {
        if (this._filterText !== text) {
            this._filterText = text;
            this.filterTextModel.next(text);
        }
    }
    get filterText() {
        return this._filterText;
    }
    getPaletteView(view) {
        let self = this;
        return view.extend({
            pointerdown: function ( /*evt, x, y*/) {
                // Remove the tooltip
                // $('.node-tooltip').remove();
                // TODO move metadata to the right place (not inside attrs I think)
                self.clickedElement = this.model;
                if (self.clickedElement && self.clickedElement.get('metadata')) {
                    $(self.document).on('mousemove', self.mouseMoveHanlder);
                }
            },
            pointermove: function ( /*evt, x, y*/) {
                // Nothing to prevent move within the palette canvas
            },
        });
    }
    handleMouseUp(event) {
        $(this.document).off('mousemove', this.mouseMoveHanlder);
    }
    trigger(event) {
        this.onPaletteEntryDrop.emit(event);
    }
    handleDrag(event) {
        // TODO offsetX/Y not on firefox
        // Logger.debug("tracking move: x="+event.pageX+",y="+event.pageY);
        // Logger.debug('Element = ' + (this.clickedElement ? this.clickedElement.attr('metadata/name'): 'null'));
        if (this.clickedElement && this.clickedElement.get('metadata')) {
            if (!this.viewBeingDragged) {
                let dataOfClickedElement = this.clickedElement.get('metadata');
                // custom div if not already built.
                $('<div>', {
                    id: 'palette-floater'
                }).appendTo($('body'));
                let floatergraph = new joint.dia.Graph();
                floatergraph.set('type', Constants.FEEDBACK_CONTEXT);
                const parent = $('#palette-floater');
                this.floaterpaper = new joint.dia.Paper({
                    el: $('#palette-floater'),
                    elementView: this.renderer && this.renderer.getNodeView ? this.renderer.getNodeView() : joint.dia.ElementView,
                    gridSize: 10,
                    model: floatergraph,
                    height: parent.height(),
                    width: parent.width(),
                    validateMagnet: () => false,
                    validateConnection: () => false
                });
                // TODO float thing needs to be bigger otherwise icon label is missing
                // Initiative drag and drop - create draggable element
                let floaternode = Shapes.Factory.createNode({
                    'renderer': this.renderer,
                    'paper': this.floaterpaper,
                    'graph': floatergraph,
                    'metadata': dataOfClickedElement
                });
                // Only node view expected
                this.viewBeingDragged = this.floaterpaper.findViewByModel(floaternode);
                const resizeObserver = new ResizeObserver(() => {
                    if (this.viewBeingDragged) {
                        const box = this.viewBeingDragged.getBBox();
                        const size = this.viewBeingDragged.model.size();
                        // Account for ports. Especially on the left side. Box includes ports, size does not
                        parent.css('width', box.width + box.width - size.width);
                        parent.css('height', box.height + box.height - size.height);
                        floaternode.position(box.width - size.width, box.height - size.height);
                    }
                });
                resizeObserver.observe(this.viewBeingDragged.el);
                let box = this.viewBeingDragged.getBBox();
                let size = floaternode.size();
                // Account for node real size including ports
                parent.css('width', box.width + box.width - size.width);
                parent.css('height', box.height + box.height - size.height);
                floaternode.position(box.width - size.width, box.height - size.height);
                parent.offset({ left: event.pageX + 5, top: event.pageY + 5 });
            }
            else {
                $('#palette-floater').offset({ left: event.pageX + 5, top: event.pageY + 5 });
                this.trigger({
                    type: Flo.DnDEventType.DRAG,
                    view: this.viewBeingDragged,
                    event: event
                });
            }
        }
    }
    /*
     * Modify the rotation of the arrow in the header from horizontal(closed) to vertical(open)
     */
    rotateOpen(element) {
        return new Promise(resolve => {
            setTimeout(() => this.doRotateOpen(element, 90).then(() => {
                resolve();
            }));
        });
    }
    doRotateOpen(element, angle) {
        return new Promise(resolve => {
            angle -= 10;
            element.attr({ 'path': { 'transform': 'rotate(-' + angle + ',15,13)' } });
            if (angle <= 0) {
                resolve();
            }
            else {
                setTimeout(() => this.doRotateOpen(element, angle).then(() => resolve()), 10);
            }
        });
    }
    doRotateClose(element, angle) {
        return new Promise(resolve => {
            angle += 10;
            element.attr({ 'path': { 'transform': 'rotate(-' + angle + ',15,13)' } });
            if (angle >= 90) {
                resolve();
            }
            else {
                setTimeout(() => this.doRotateClose(element, angle).then(() => resolve()), 10);
            }
        });
    }
    // TODO better name for this function as this does the animation *and* updates the palette
    /*
     * Modify the rotation of the arrow in the header from vertical(open) to horizontal(closed)
     */
    rotateClosed(element) {
        return new Promise(resolve => {
            setTimeout(() => {
                this.doRotateClose(element, 0).then(() => {
                    resolve();
                });
            });
        });
    }
}
Palette.decorators = [
    { type: Component, args: [{
                selector: 'flo-palette',
                template: "<div id=\"palette-filter\" class=\"palette-filter\">\n  <label class=\"palette-filter-label\" for=\"palette-filter-textfield\">\n    <input type=\"search\" id=\"palette-filter-textfield\" class=\"palette-filter-textfield\" [(ngModel)]=\"filterText\" (focus)=\"onFocus()\" [placeholder]=\"searchFilterPlaceHolder\"/>\n  </label>\n</div>\n<div id=\"palette-paper-container\">\n  <div id=\"palette-paper\" class=\"palette-paper\" style=\"overflow:hidden;\"></div>\n</div>\n\n",
                encapsulation: ViewEncapsulation.None,
                styles: ["#palette-floater{-moz-user-select:none;-ms-user-select:none;-o-user-select:none;-webkit-user-select:none;float:left;opacity:.75;position:absolute;user-select:none}#palette-floater,#palette-floater.joint-paper>svg{background-color:transparent}#palette-paper-container{background-color:#fff;color:#fff;height:calc(100% - 46px);overflow-x:hidden;overflow-y:auto;width:100%}.palette-filter{background-color:#fff;border:3px solid #6db33f;font-size:24px}.palette-filter-textfield{border:none;font-family:Varela Round,sans-serif;width:100%}.palette-paper{background-color:#eee;border-color:#6db33f}.no-matches-label{fill:#777;font-size:14px}"]
            },] }
];
Palette.ctorParameters = () => [
    { type: ElementRef },
    { type: undefined, decorators: [{ type: Inject, args: [DOCUMENT,] }] }
];
Palette.propDecorators = {
    metamodel: [{ type: Input }],
    renderer: [{ type: Input }],
    paletteEntryPadding: [{ type: Input }],
    searchFilterPlaceHolder: [{ type: Input }],
    onPaletteEntryDrop: [{ type: Output }],
    paletteReady: [{ type: Output }],
    paletteFocus: [{ type: Output }],
    paletteSize: [{ type: Input }]
};
//# sourceMappingURL=data:application/json;base64,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