import { Subject, Observable } from 'rxjs';
import { debounceTime, switchMap } from 'rxjs/operators';
export var Properties;
(function (Properties) {
    let InputType;
    (function (InputType) {
        InputType[InputType["TEXT"] = 0] = "TEXT";
        InputType[InputType["NUMBER"] = 1] = "NUMBER";
        InputType[InputType["SELECT"] = 2] = "SELECT";
        InputType[InputType["CHECKBOX"] = 3] = "CHECKBOX";
        InputType[InputType["PASSWORD"] = 4] = "PASSWORD";
        InputType[InputType["EMAIL"] = 5] = "EMAIL";
        InputType[InputType["URL"] = 6] = "URL";
        InputType[InputType["CODE"] = 7] = "CODE";
    })(InputType = Properties.InputType || (Properties.InputType = {}));
    class GenericControlModel {
        constructor(_property, type, validation) {
            this._property = _property;
            this.type = type;
            this.validation = validation;
        }
        get id() {
            return this.property.id;
        }
        get name() {
            return this.property.name;
        }
        get description() {
            return this.property.description;
        }
        get defaultValue() {
            return this.property.defaultValue;
        }
        get value() {
            return this.getValue();
        }
        set value(value) {
            this.setValue(value);
        }
        get property() {
            return this._property;
        }
        setValue(value) {
            this.property.value = value;
        }
        getValue() {
            return this.property.value;
        }
    }
    Properties.GenericControlModel = GenericControlModel;
    class CheckBoxControlModel extends GenericControlModel {
        constructor(_property, validation) {
            super(_property, InputType.CHECKBOX, validation);
        }
        getValue() {
            const res = super.getValue();
            const type = typeof res;
            switch (type) {
                case 'boolean':
                    return res;
                case 'string':
                    switch (res.trim().toLowerCase()) {
                        case 'true':
                        case '1':
                            return true;
                        case 'false':
                        case '0':
                            return false;
                        default:
                            return this.property.defaultValue;
                    }
                case 'number':
                    const num = res;
                    if (num === 0) {
                        return false;
                    }
                    else if (num === 1) {
                        return true;
                    }
                    else {
                        return this.property.defaultValue;
                    }
            }
            return this.property.defaultValue;
        }
    }
    Properties.CheckBoxControlModel = CheckBoxControlModel;
    class AbstractCodeControlModel extends GenericControlModel {
        constructor(_property, encode, decode, validation) {
            super(_property, InputType.CODE, validation);
            this.encode = encode;
            this.decode = decode;
        }
        set value(value) {
            if (value && this.encode) {
                super.setValue(this.encode(value));
            }
            else {
                super.setValue(value);
            }
        }
        get value() {
            let dsl = super.getValue();
            if (dsl && this.decode) {
                return this.decode(dsl);
            }
            else {
                return dsl;
            }
        }
    }
    Properties.AbstractCodeControlModel = AbstractCodeControlModel;
    class GenericCodeControlModel extends AbstractCodeControlModel {
        constructor(_property, language, encode, decode, validation) {
            super(_property, encode, decode, validation);
            this.language = language;
        }
    }
    Properties.GenericCodeControlModel = GenericCodeControlModel;
    class CodeControlModelWithDynamicLanguageProperty extends AbstractCodeControlModel {
        constructor(_property, _languagePropertyName, _groupModel, encode, decode, validation) {
            super(_property, encode, decode, validation);
            this._languagePropertyName = _languagePropertyName;
            this._groupModel = _groupModel;
        }
        get language() {
            const value = this.languageControlModel.value;
            return value ? value : this.languageControlModel.defaultValue;
        }
        get languageControlModel() {
            if (!this._langControlModel) {
                // Cast to Properties.ControlModel<any> from Properties.ControlModel<any> | undefined
                // Should not be undefined!
                this._langControlModel = this._groupModel.getControlsModels().find(c => c.id === this._languagePropertyName);
            }
            return this._langControlModel;
        }
    }
    Properties.CodeControlModelWithDynamicLanguageProperty = CodeControlModelWithDynamicLanguageProperty;
    class GenericListControlModel extends GenericControlModel {
        constructor(property, validation) {
            super(property, InputType.TEXT, validation);
        }
        get value() {
            return this.property.value ? this.property.value.toString().trim().split(/\s*,\s*/).join(', ') : '';
        }
        set value(value) {
            this.property.value = value && value.trim() ? value.split(/\s*,\s*/).join(',') : undefined;
        }
    }
    Properties.GenericListControlModel = GenericListControlModel;
    class SelectControlModel extends GenericControlModel {
        constructor(_property, type, options) {
            super(_property, type);
            this.options = options;
            if (_property.defaultValue === undefined) {
                options.unshift({
                    name: 'SELECT',
                    value: _property.defaultValue
                });
            }
        }
    }
    Properties.SelectControlModel = SelectControlModel;
    class DefaultCellPropertiesSource {
        constructor(cell) {
            this.cell = cell;
        }
        getProperties() {
            let metadata = this.cell.get('metadata');
            return Promise.resolve(metadata.properties().then(propsMetadata => Array.from(propsMetadata.values()).map(m => this.createProperty(m))));
        }
        createProperty(metadata) {
            return {
                id: metadata.id,
                name: metadata.name,
                type: metadata.type,
                defaultValue: metadata.defaultValue,
                attr: `props/${metadata.name}`,
                value: this.cell.attr(`props/${metadata.name}`),
                description: metadata.description,
                valueOptions: metadata.options
            };
        }
        applyChanges(properties) {
            this.cell.trigger('batch:start', { batchName: 'update properties' });
            properties.forEach(property => {
                if ((typeof property.value === 'boolean' && !property.defaultValue && !property.value) ||
                    (property.value === property.defaultValue || property.value === '' || property.value === undefined || property.value === null)) {
                    let currentValue = this.cell.attr(property.attr);
                    if (currentValue !== undefined && currentValue !== null) {
                        // Remove attr doesn't fire appropriate event. Set default value first as a workaround to schedule DSL resync
                        // this.cell.attr(property.attr, property.defaultValue === undefined ? null : property.defaultValue);
                        this.cell.attr(property.attr, null);
                        this.cell.removeAttr(property.attr);
                    }
                }
                else {
                    this.cell.attr(property.attr, property.value);
                }
            });
            this.cell.trigger('batch:stop', { batchName: 'update properties' });
        }
    }
    Properties.DefaultCellPropertiesSource = DefaultCellPropertiesSource;
    class PropertiesGroupModel {
        constructor(propertiesSource) {
            this.loading = true;
            this.propertiesSource = propertiesSource;
        }
        load() {
            this.loading = true;
            this._loadedSubject = new Subject();
            this.propertiesSource.getProperties().then(properties => {
                this.controlModels = properties.map(p => this.createControlModel(p));
                this.loading = false;
                this._loadedSubject.next(true);
                this._loadedSubject.complete();
            });
        }
        get isLoading() {
            return this.loading;
        }
        get loadedSubject() {
            return this._loadedSubject;
        }
        getControlsModels() {
            return this.controlModels;
        }
        createControlModel(property) {
            return new GenericControlModel(property, InputType.TEXT);
        }
        applyChanges() {
            if (this.loading) {
                return;
            }
            let properties = this.controlModels.map(cm => cm.property);
            this.propertiesSource.applyChanges(properties);
        }
    }
    Properties.PropertiesGroupModel = PropertiesGroupModel;
    const UNIQUE_RESOURCE_ERROR = { uniqueResource: true };
    let Validators;
    (function (Validators) {
        function uniqueResource(service, debounceDuration) {
            return (control) => {
                return new Observable(obs => {
                    if (control.valueChanges) {
                        return control.valueChanges.pipe(debounceTime(debounceDuration), switchMap(() => service(control.value))).subscribe((res) => {
                            if (res) {
                                obs.next(undefined);
                            }
                            else {
                                obs.next(UNIQUE_RESOURCE_ERROR);
                            }
                            obs.complete();
                        }, () => {
                            obs.next(UNIQUE_RESOURCE_ERROR);
                            obs.complete();
                        });
                    }
                    else {
                        service(control.value).subscribe((res) => {
                            if (res) {
                                obs.next(undefined);
                            }
                            else {
                                obs.next(UNIQUE_RESOURCE_ERROR);
                            }
                            obs.complete();
                        }, () => {
                            obs.next(UNIQUE_RESOURCE_ERROR);
                            obs.complete();
                        });
                    }
                });
            };
        }
        Validators.uniqueResource = uniqueResource;
        function noneOf(excluded) {
            return (control) => {
                return excluded.find(e => e === control.value) ? { 'noneOf': { value: control.value } } : {};
            };
        }
        Validators.noneOf = noneOf;
    })(Validators = Properties.Validators || (Properties.Validators = {}));
})(Properties || (Properties = {}));
//# sourceMappingURL=data:application/json;base64,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