import { Flo } from './flo-common';
import * as _ from 'lodash';
import * as _$ from 'jquery';
const joint = Flo.joint;
const $ = _$;
const isChrome = /Chrome/.test(navigator.userAgent) && /Google Inc/.test(navigator.vendor);
const isFF = navigator.userAgent.indexOf('Firefox') > 0;
const IMAGE_W = 120;
const IMAGE_H = 35;
const ERROR_MARKER_SIZE = { width: 16, height: 16 };
const HANDLE_SIZE = { width: 10, height: 10 };
const HANDLE_ICON_MAP = new Map();
const REMOVE = 'remove';
HANDLE_ICON_MAP.set(REMOVE, 'icons/delete.svg');
const DECORATION_ICON_MAP = new Map();
const ERROR = 'error';
DECORATION_ICON_MAP.set(ERROR, 'icons/error.svg');
export function loadShapes() {
    joint.shapes.flo = {};
    joint.shapes.flo.NODE_TYPE = 'sinspctr.IntNode';
    joint.shapes.flo.LINK_TYPE = 'sinspctr.Link';
    joint.shapes.flo.DECORATION_TYPE = 'decoration';
    joint.shapes.flo.HANDLE_TYPE = 'handle';
    joint.util.cloneDeep = (obj) => {
        return _.cloneDeepWith(obj, (o) => {
            if (_.isObject(o) && !_.isPlainObject(o)) {
                return o;
            }
        });
    };
    joint.util.filter.redscale = (args) => {
        let amount = Number.isFinite(args.amount) ? args.amount : 1;
        return _.template('<filter><feColorMatrix type="matrix" values="${a} ${b} ${c} 0 ${d} ${e} ${f} ${g} 0 0 ${h} ${i} ${k} 0 0 0 0 0 1 0"/></filter>', {
            a: 1 - 0.96 * amount,
            b: 0.95 * amount,
            c: 0.01 * amount,
            d: 0.3 * amount,
            e: 0.2 * amount,
            f: 1 - 0.9 * amount,
            g: 0.7 * amount,
            h: 0.05 * amount,
            i: 0.05 * amount,
            k: 1 - 0.1 * amount
        });
    };
    joint.util.filter.orangescale = (args) => {
        let amount = Number.isFinite(args.amount) ? args.amount : 1;
        return _.template('<filter><feColorMatrix type="matrix" values="${a} ${b} ${c} 0 ${d} ${e} ${f} ${g} 0 ${h} ${i} ${k} ${l} 0 0 0 0 0 1 0"/></filter>', {
            a: 1.0 + 0.5 * amount,
            b: 1.4 * amount,
            c: 0.2 * amount,
            d: 0.3 * amount,
            e: 0.3 * amount,
            f: 1 + 0.05 * amount,
            g: 0.2 * amount,
            h: 0.15 * amount,
            i: 0.3 * amount,
            k: 0.3 * amount,
            l: 1 - 0.6 * amount
        });
    };
    joint.shapes.flo.Node = joint.shapes.basic.Generic.extend({
        markup: '<g class="shape"><image class="image" /></g>' +
            '<rect class="border-white"/>' +
            '<rect class="border"/>' +
            '<rect class="box"/>' +
            '<text class="label"/>' +
            '<text class="label2"></text>' +
            '<rect class="input-port" />' +
            '<rect class="output-port"/>' +
            '<rect class="output-port-cover"/>',
        defaults: joint.util.deepSupplement({
            type: joint.shapes.flo.NODE_TYPE,
            position: { x: 0, y: 0 },
            size: { width: IMAGE_W, height: IMAGE_H },
            attrs: {
                '.': { magnet: false },
                // rounded edges around image
                '.border': {
                    refWidth: 1,
                    refHeight: 1,
                    rx: 3,
                    ry: 3,
                    'fill-opacity': 0,
                    stroke: '#eeeeee',
                    'stroke-width': 0
                },
                '.box': {
                    refWidth: 1,
                    refHeight: 1,
                    rx: 3,
                    ry: 3,
                    //'fill-opacity': 0, // see through
                    stroke: '#6db33f',
                    fill: '#eeeeee',
                    'stroke-width': 1
                },
                '.input-port': {
                    idp: 'input',
                    port: 'input',
                    height: 8, width: 8,
                    magnet: true,
                    fill: '#eeeeee',
                    transform: 'translate(' + -4 + ',' + ((IMAGE_H / 2) - 4) + ')',
                    stroke: '#34302d',
                    'stroke-width': 1
                },
                '.output-port': {
                    id: 'output',
                    port: 'output',
                    height: 8, width: 8,
                    magnet: true,
                    fill: '#eeeeee',
                    transform: 'translate(' + (IMAGE_W - 4) + ',' + ((IMAGE_H / 2) - 4) + ')',
                    stroke: '#34302d',
                    'stroke-width': 1
                },
                '.label': {
                    'text-anchor': 'middle',
                    'ref-x': 0.5,
                    // 'ref-y': -12, // jointjs specific: relative position to ref'd element
                    'ref-y': 0.3,
                    ref: '.border',
                    fill: 'black',
                    'font-size': 14
                },
                '.label2': {
                    'text': '\u21d2',
                    'text-anchor': 'middle',
                    'ref-x': 0.15,
                    'ref-y': 0.2,
                    ref: '.border',
                    // transform: 'translate(' + (IMAGE_W/2) + ',' + (IMAGE_H/2) + ')',
                    fill: 'black',
                    'font-size': 24
                },
                '.shape': {},
                '.image': {
                    refWidth: 1,
                    refHeight: 1,
                }
            }
        }, joint.shapes.basic.Generic.prototype.defaults)
    });
    joint.shapes.flo.Link = joint.dia.Link.extend({
        defaults: joint.util.deepSupplement({
            type: joint.shapes.flo.LINK_TYPE,
            attrs: {
                '.connection': { stroke: '#34302d', 'stroke-width': 2 },
                // Lots of alternatives that have been played with:
                //	            '.smoooth': true
                //	            '.marker-source': { stroke: '#9B59B6', fill: '#9B59B6', d: 'M24.316,5.318,9.833,13.682,9.833,5.5,5.5,5.5,5.5,25.5,9.833,25.5,9.833,17.318,24.316,25.682z' },
                //	            '.marker-target': { stroke: '#F39C12', fill: '#F39C12', d: 'M14.615,4.928c0.487-0.986,1.284-0.986,1.771,0l2.249,4.554c0.486,0.986,1.775,1.923,2.864,2.081l5.024,0.73c1.089,0.158,1.335,0.916,0.547,1.684l-3.636,3.544c-0.788,0.769-1.28,2.283-1.095,3.368l0.859,5.004c0.186,1.085-0.459,1.553-1.433,1.041l-4.495-2.363c-0.974-0.512-2.567-0.512-3.541,0l-4.495,2.363c-0.974,0.512-1.618,0.044-1.432-1.041l0.858-5.004c0.186-1.085-0.307-2.6-1.094-3.368L3.93,13.977c-0.788-0.768-0.542-1.525,0.547-1.684l5.026-0.73c1.088-0.158,2.377-1.095,2.864-2.081L14.615,4.928z' },
                //	        	'.connection': { 'stroke':'black'},
                //	        	'.': { filter: { name: 'dropShadow', args: { dx: 1, dy: 1, blur: 2 } } },
                //	        	'.connection': { 'stroke-width': 10, 'stroke-linecap': 'round' },
                // This means: moveto 10 0, lineto 0 5, lineto, 10 10 closepath(z)
                //	        	'.marker-target': { d: 'M 5 0 L 0 7 L 5 14 z', stroke: '#34302d','stroke-width': 1},
                //	        	'.marker-target': { d: 'M 14 2 L 9,2 L9,0 L 0,7 L 9,14 L 9,12 L 14,12 z', 'stroke-width': 1, fill: '#34302d', stroke: '#34302d'},
                //	        	'.marker-source': {d: 'M 5 0 L 5,10 L 0,10 L 0,0 z', 'stroke-width': 0, fill: '#34302d', stroke: '#34302d'},
                //	            '.marker-target': { stroke: '#E74C3C', fill: '#E74C3C', d: 'M 10 0 L 0 5 L 10 10 z' },
                '.marker-arrowheads': { display: 'none' },
                '.tool-options': { display: 'none' }
            },
        }, joint.dia.Link.prototype.defaults)
    });
    joint.shapes.flo.LinkView = joint.dia.LinkView.extend({
        options: joint.util.deepSupplement({
            linkToolsOffset: 0.5,
            shortLinkLength: 40
        }, joint.dia.LinkView.prototype.options),
        updateToolsPosition: function () {
            // Overriden to support relative offset for tools placement.
            // If offset is between 0 and 1 then percentage of the connection length will be used to offset the tools group
            if (this.options.linkToolsOffset < 1 && this.options.linkToolsOffset > 0) {
                let connectionLength = this.getConnectionLength();
                const relativeOffset = this.options.linkToolsOffset;
                this.options.linkToolsOffset = connectionLength * relativeOffset;
                const returnValue = joint.dia.LinkView.prototype.updateToolsPosition.apply(this, arguments);
                this.options.linkToolsOffset = relativeOffset;
                return returnValue;
            }
            else {
                return joint.dia.LinkView.prototype.updateToolsPosition.apply(this, arguments);
            }
        },
        _beforeArrowheadMove: function () {
            if (this.model.get('source').id) {
                this._oldSource = this.model.get('source');
            }
            if (this.model.get('target').id) {
                this._oldTarget = this.model.get('target');
            }
            joint.dia.LinkView.prototype._beforeArrowheadMove.apply(this, arguments);
        },
        _afterArrowheadMove: function () {
            joint.dia.LinkView.prototype._afterArrowheadMove.apply(this, arguments);
            if (!this.model.get('source').id) {
                if (this._oldSource) {
                    this.model.set('source', this._oldSource);
                }
                else {
                    this.model.remove();
                }
            }
            if (!this.model.get('target').id) {
                if (this._oldTarget) {
                    this.model.set('target', this._oldTarget);
                }
                else {
                    this.model.remove();
                }
            }
            delete this._oldSource;
            delete this._oldTarget;
        }
    });
    // TODO: must do cleanup for the `mainElementView'
    joint.shapes.flo.ElementView = joint.dia.ElementView.extend({
        // canShowTooltip: true,
        beingDragged: false,
        // _tempZorder: 0,
        _tempOpacity: 1.0,
        _hovering: false,
        dragLinkStart: function (evt, magnet, x, y) {
            this.model.startBatch('add-link');
            const linkView = this.addLinkFromMagnet(magnet, x, y);
            // backwards compatiblity events
            joint.dia.CellView.prototype.pointerdown.apply(linkView, [evt, x, y]);
            linkView.notify('link:pointerdown', evt, x, y);
            /*** START MAIN DIFF ***/
            const sourceOrTarget = $(magnet).attr('port') === 'input' ? 'source' : 'target';
            linkView.eventData(evt, linkView.startArrowheadMove(sourceOrTarget, { whenNotAllowed: 'remove' }));
            /*** END MAIN DIFF ***/
            this.eventData(evt, { linkView: linkView });
        },
        addLinkFromMagnet: function (magnet, x, y) {
            const paper = this.paper;
            const graph = paper.model;
            const link = paper.getDefaultLink(this, magnet);
            let sourceEnd, targetEnd;
            /*** START MAIN DIFF ***/
            if ($(magnet).attr('port') === 'input') {
                sourceEnd = { x: x, y: y };
                targetEnd = this.getLinkEnd(magnet, x, y, link, 'target');
            }
            else {
                sourceEnd = this.getLinkEnd(magnet, x, y, link, 'source');
                targetEnd = { x: x, y: y };
            }
            /*** END MAIN DIFF ***/
            link.set({
                source: sourceEnd,
                target: targetEnd
            }).addTo(graph, {
                async: false,
                ui: true
            });
            return link.findView(paper);
        },
        // pointerdown: function(evt: any, x: number, y: number) {
        //   // this.canShowTooltip = false;
        //   // this.hideTooltip();
        //   this.beingDragged = false;
        //   this._tempOpacity = this.model.attr('./opacity');
        //
        //   this.model.trigger('batch:start');
        //
        //   if ( // target is a valid magnet start linking
        //     evt.target.getAttribute('magnet') &&
        //     this.paper.options.validateMagnet.call(this.paper, this, evt.target)
        //   ) {
        //     let link = this.paper.getDefaultLink(this, evt.target);
        //     if ($(evt.target).attr('port') === 'input') {
        //       link.set({
        //         source: { x: x, y: y },
        //         target: {
        //           id: this.model.id,
        //           selector: this.getSelector(evt.target),
        //           port: evt.target.getAttribute('port')
        //         }
        //       });
        //     } else {
        //       link.set({
        //         source: {
        //           id: this.model.id,
        //           selector: this.getSelector(evt.target),
        //           port: evt.target.getAttribute('port')
        //         },
        //         target: { x: x, y: y }
        //       });
        //     }
        //     this.paper.model.addCell(link);
        //     this._linkView = this.paper.findViewByModel(link);
        //     if ($(evt.target).attr('port') === 'input') {
        //       this._linkView.startArrowheadMove('source');
        //     } else {
        //       this._linkView.startArrowheadMove('target');
        //     }
        //     this.paper.__creatingLinkFromPort = true;
        //   } else {
        //     this._dx = x;
        //     this._dy = y;
        //     joint.dia.CellView.prototype.pointerdown.apply(this, arguments);
        //   }
        // },
        drag: function (evt, x, y) {
            let interactive = _.isFunction(this.options.interactive) ? this.options.interactive(this, 'pointermove') :
                this.options.interactive;
            if (interactive !== false) {
                this.paper.trigger('dragging-node-over-canvas', { type: Flo.DnDEventType.DRAG, view: this, event: evt });
            }
            joint.dia.ElementView.prototype.drag.apply(this, arguments);
        },
        dragEnd: function (evt, x, y) {
            this.paper.trigger('dragging-node-over-canvas', { type: Flo.DnDEventType.DROP, view: this, event: evt });
            joint.dia.ElementView.prototype.dragEnd.apply(this, arguments);
        },
    });
    joint.shapes.flo.ErrorDecoration = joint.shapes.basic.Generic.extend({
        markup: '<g class="rotatable"><g class="scalable"><image/></g></g>',
        defaults: joint.util.deepSupplement({
            type: joint.shapes.flo.DECORATION_TYPE,
            size: ERROR_MARKER_SIZE,
            attrs: {
                'image': ERROR_MARKER_SIZE
            }
        }, joint.shapes.basic.Generic.prototype.defaults)
    });
    joint.shapes.flo.PaletteGroupHeader = joint.shapes.basic.Generic.extend({
        // The path is the open/close arrow, defaults to vertical (open)
        markup: '<g class="scalable"><rect/></g><text/><g class="rotatable"><path d="m 10 10 l 5 8.7 l 5 -8.7 z"/></g>',
        defaults: joint.util.deepSupplement({
            type: 'palette.groupheader',
            size: { width: 170, height: 30 },
            position: { x: 0, y: 0 },
            attrs: {
                'rect': { fill: '#34302d', 'stroke-width': 1, stroke: '#6db33f', 'follow-scale': true, width: 80, height: 40 },
                'text': {
                    text: '',
                    fill: '#eeeeee',
                    'ref-x': 0.5,
                    'ref-y': 7,
                    'x-alignment': 'middle',
                    'font-size': 18 /*, 'font-weight': 'bold', 'font-variant': 'small-caps', 'text-transform': 'capitalize'*/
                },
                'path': { fill: 'white', 'stroke-width': 2, stroke: 'white' /*,transform:'rotate(90,15,15)'*/ }
            },
            // custom properties
            isOpen: true
        }, joint.shapes.basic.Generic.prototype.defaults)
    });
    joint.shapes.flo.NoMatchesFound = joint.shapes.basic.Generic.extend({
        // The path is the open/close arrow, defaults to vertical (open)
        markup: '<g class="scalable"><rect class="no-matches-label-border"/></g><rect class="no-mathes-label-bg"/><text class="no-matches-label"/>',
        defaults: joint.util.deepSupplement({
            size: { width: 170, height: 30 },
            position: { x: 0, y: 0 },
            attrs: {
                '.no-matches-label-border': {
                    refWidth: 1,
                    refHeight: 1,
                    refX: 0,
                    refY: 0,
                },
                '.no-macthes-label-bg': {
                    ref: '.no-matches-label',
                    refWidth: 10,
                    refHeight: 2,
                    'follow-scale': true
                },
                '.no-matches-label': {
                    text: 'No results found.',
                    ref: '.no-matches-label-border',
                    refY: 0.5,
                    refY2: 5,
                    yAlignment: 'middle',
                },
            },
        }, joint.shapes.basic.Generic.prototype.defaults)
    });
}
export var Constants;
(function (Constants) {
    Constants.REMOVE_HANDLE_TYPE = REMOVE;
    Constants.PROPERTIES_HANDLE_TYPE = 'properties';
    Constants.ERROR_DECORATION_KIND = ERROR;
    Constants.PALETTE_CONTEXT = 'palette';
    Constants.CANVAS_CONTEXT = 'canvas';
    Constants.FEEDBACK_CONTEXT = 'feedback';
})(Constants || (Constants = {}));
export var Shapes;
(function (Shapes) {
    class Factory {
        /**
         * Create a JointJS node that embeds extra metadata (properties).
         */
        static createNode(params) {
            let renderer = params.renderer;
            let paper = params.paper;
            let metadata = params.metadata;
            let position = params.position;
            let props = params.props;
            let graph = params.graph || (params.paper ? params.paper.model : undefined);
            let node;
            if (!position) {
                position = { x: 0, y: 0 };
            }
            if (renderer && _.isFunction(renderer.createNode)) {
                node = renderer.createNode({ graph, paper }, metadata, props);
            }
            else {
                node = new joint.shapes.flo.Node();
                if (metadata) {
                    node.attr('.label/text', metadata.name);
                }
            }
            node.set('type', joint.shapes.flo.NODE_TYPE);
            if (position) {
                node.set('position', position);
            }
            if (props) {
                Array.from(props.keys()).forEach(key => node.attr(`props/${key}`, props.get(key)));
            }
            node.set('metadata', metadata);
            if (graph) {
                graph.addCell(node);
            }
            if (renderer && _.isFunction(renderer.initializeNewNode)) {
                let descriptor = {
                    paper: paper,
                    graph: graph
                };
                renderer.initializeNewNode(node, descriptor);
            }
            return node;
        }
        static createLink(params) {
            let renderer = params.renderer;
            let paper = params.paper;
            let metadata = params.metadata;
            let source = params.source;
            let target = params.target;
            let props = params.props;
            let graph = params.graph || (params.paper ? params.paper.model : undefined);
            let link;
            if (renderer && _.isFunction(renderer.createLink)) {
                link = renderer.createLink(source, target, metadata, props);
            }
            else {
                link = new joint.shapes.flo.Link();
            }
            if (source) {
                link.set('source', source);
            }
            if (target) {
                link.set('target', target);
            }
            link.set('type', joint.shapes.flo.LINK_TYPE);
            if (metadata) {
                link.set('metadata', metadata);
            }
            if (props) {
                Array.from(props.keys()).forEach(key => link.attr(`props/${key}`, props.get(key)));
            }
            if (graph) {
                graph.addCell(link);
            }
            if (renderer && _.isFunction(renderer.initializeNewLink)) {
                let descriptor = {
                    paper: paper,
                    graph: graph
                };
                renderer.initializeNewLink(link, descriptor);
            }
            // prevent creation of link breaks
            link.attr('.marker-vertices/display', 'none');
            return link;
        }
        static createDecoration(params) {
            let renderer = params.renderer;
            let paper = params.paper;
            let parent = params.parent;
            let kind = params.kind;
            let messages = params.messages;
            let location = params.position;
            let graph = params.graph || (params.paper ? params.paper.model : undefined);
            let decoration;
            if (renderer && _.isFunction(renderer.createDecoration)) {
                decoration = renderer.createDecoration(kind, parent);
            }
            if (decoration) {
                decoration.set('type', joint.shapes.flo.DECORATION_TYPE);
                if ((isChrome || isFF) && parent && typeof parent.get('z') === 'number') {
                    decoration.set('z', parent.get('z') + 1);
                }
                decoration.attr('./kind', kind);
                decoration.attr('messages', messages);
                if (graph) {
                    graph.addCell(decoration);
                }
                parent.embed(decoration);
                if (renderer && _.isFunction(renderer.initializeNewDecoration)) {
                    let descriptor = {
                        paper: paper,
                        graph: graph
                    };
                    renderer.initializeNewDecoration(decoration, descriptor);
                }
                return decoration;
            }
        }
        static createHandle(params) {
            let renderer = params.renderer;
            let paper = params.paper;
            let parent = params.parent;
            let kind = params.kind;
            let location = params.position;
            let graph = params.graph || (params.paper ? params.paper.model : undefined);
            let handle;
            if (!location) {
                location = { x: 0, y: 0 };
            }
            if (renderer && _.isFunction(renderer.createHandle)) {
                handle = renderer.createHandle(kind, parent);
            }
            else {
                handle = new joint.shapes.flo.ErrorDecoration({
                    size: HANDLE_SIZE,
                    attrs: {
                        'image': {
                            'xlink:href': HANDLE_ICON_MAP.get(kind)
                        }
                    }
                });
            }
            handle.set('type', joint.shapes.flo.HANDLE_TYPE);
            handle.set('position', location);
            if ((isChrome || isFF) && parent && typeof parent.get('z') === 'number') {
                handle.set('z', parent.get('z') + 1);
            }
            handle.attr('./kind', kind);
            if (graph) {
                graph.addCell(handle);
            }
            parent.embed(handle);
            if (renderer && _.isFunction(renderer.initializeNewHandle)) {
                let descriptor = {
                    paper: paper,
                    graph: graph
                };
                renderer.initializeNewHandle(handle, descriptor);
            }
            return handle;
        }
    }
    Shapes.Factory = Factory;
})(Shapes || (Shapes = {}));
loadShapes();
//# sourceMappingURL=data:application/json;base64,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