import * as tslib_1 from "tslib";
import { Component, Input, Output, ElementRef, EventEmitter, ViewEncapsulation, } from '@angular/core';
import { debounceTime } from 'rxjs/operators';
import { dia } from 'jointjs';
import { Flo } from '../shared/flo-common';
import { Shapes, Constants } from '../shared/shapes';
import { Utils } from './editor-utils';
import { CompositeDisposable, Disposable } from 'ts-disposables';
import * as _$ from 'jquery';
import * as _ from 'lodash';
import { Subject, BehaviorSubject } from 'rxjs';
const joint = Flo.joint;
const $ = _$;
const SCROLLBAR_SIZE = 17;
let EditorComponent = class EditorComponent {
    constructor(element) {
        this.element = element;
        /**
         * Flag specifying whether the Flo-Editor is in read-only mode.
         */
        this._readOnlyCanvas = false;
        /**
         * Grid size
         */
        this._gridSize = 1;
        this._hiddenPalette = false;
        this.paletteSizeValue = 170;
        this.textToGraphEventEmitter = new EventEmitter();
        this.graphToTextEventEmitter = new EventEmitter();
        this._graphToTextSyncEnabled = true;
        this.validationEventEmitter = new EventEmitter();
        this._disposables = new CompositeDisposable();
        this._dslText = '';
        this.textToGraphConversionCompleted = new Subject();
        this.graphToTextConversionCompleted = new Subject();
        this.paletteReady = new BehaviorSubject(false);
        this.paletteSizeChange = new EventEmitter();
        this.searchFilterPlaceHolder = 'Search...';
        /**
         * Min zoom percent value
         */
        this.minZoom = 5;
        /**
         * Max zoom percent value
         */
        this.maxZoom = 400;
        /**
         * Zoom percent increment/decrement step
         */
        this.zoomStep = 5;
        this.paperPadding = 0;
        this.floApi = new EventEmitter();
        this.validationMarkers = new EventEmitter();
        this.contentValidated = new EventEmitter();
        this.dslChange = new EventEmitter();
        this.onProperties = new EventEmitter();
        this._resizeHandler = () => this.autosizePaper();
        let self = this;
        this.editorContext = new (class DefaultRunnableContext {
            set zoomPercent(percent) {
                self.zoomPercent = percent;
            }
            get zoomPercent() {
                return self.zoomPercent;
            }
            set noPalette(noPalette) {
                self.noPalette = noPalette;
            }
            get noPalette() {
                return self.noPalette;
            }
            set gridSize(gridSize) {
                self.gridSize = gridSize;
            }
            get gridSize() {
                return self.gridSize;
            }
            set readOnlyCanvas(readOnly) {
                self.readOnlyCanvas = readOnly;
            }
            get readOnlyCanvas() {
                return self.readOnlyCanvas;
            }
            setDsl(dsl) {
                self.dsl = dsl;
            }
            updateGraph() {
                return self.updateGraphRepresentation();
            }
            updateText() {
                return self.updateTextRepresentation();
            }
            performLayout() {
                return self.doLayout();
            }
            clearGraph() {
                self.selection = undefined;
                self.graph.clear();
                if (self.metamodel && self.metamodel.load && self.editor && self.editor.setDefaultContent) {
                    return self.metamodel.load().then(data => {
                        self.editor.setDefaultContent(this, data);
                        if (!self.graphToTextSync) {
                            return self.updateTextRepresentation();
                        }
                    });
                }
                else {
                    if (!self.graphToTextSync) {
                        return self.updateTextRepresentation();
                    }
                }
            }
            getGraph() {
                return self.graph;
            }
            getPaper() {
                return self.paper;
            }
            get graphToTextSync() {
                return self.graphToTextSync;
            }
            set graphToTextSync(sync) {
                self.graphToTextSync = sync;
            }
            getMinZoom() {
                return self.minZoom;
            }
            getMaxZoom() {
                return self.maxZoom;
            }
            getZoomStep() {
                return self.zoomStep;
            }
            fitToPage() {
                self.fitToPage();
            }
            createNode(metadata, props, position) {
                return self.createNode(metadata, props, position);
            }
            createLink(source, target, metadata, props) {
                return self.createLink(source, target, metadata, props);
            }
            get selection() {
                return self.selection;
            }
            set selection(newSelection) {
                self.selection = newSelection;
            }
            deleteSelectedNode() {
                self.deleteSelected();
            }
            delete(cell) {
                self.delete(cell);
            }
            get textToGraphConversionObservable() {
                return self.textToGraphConversionCompleted;
            }
            get graphToTextConversionObservable() {
                return self.graphToTextConversionCompleted;
            }
            get paletteReady() {
                return self.paletteReady;
            }
        })();
    }
    /**
     * Size (Width) of the palette
     */
    get paletteSize() {
        return this.paletteSizeValue;
    }
    set paletteSize(newSize) {
        this.paletteSizeValue = newSize;
        this.paletteSizeChange.emit(newSize);
    }
    onPropertiesHandle() {
        if (this.editorContext.selection) {
            this.onProperties.emit(this.editorContext.selection.model);
        }
    }
    ngOnInit() {
        this.initGraph();
        this.initPaper();
        this.initGraphListeners();
        this.initPaperListeners();
        this.initMetamodel();
        $(window).on('resize', this._resizeHandler);
        this._disposables.add(Disposable.create(() => $(window).off('resize', this._resizeHandler)));
        /*
         * Execute resize to get the right size for the SVG element on the editor canvas.
         * Executed via timeout to let angular render the DOM first and elements to have the right width and height
         */
        window.setTimeout(this._resizeHandler);
        this.floApi.emit(this.editorContext);
    }
    ngOnDestroy() {
        this._disposables.dispose();
    }
    deleteSelected() {
        if (this.selection) {
            this.delete(this.selection.model);
        }
    }
    delete(cell) {
        this.graph.trigger('startDeletion', cell);
    }
    get noPalette() {
        return this._hiddenPalette;
    }
    set noPalette(hidden) {
        this._hiddenPalette = hidden;
        // If palette is not shown ensure that canvas starts from the left==0!
        if (hidden) {
            $('#paper-container', this.element.nativeElement).css('left', 0);
        }
    }
    get graphToTextSync() {
        return this._graphToTextSyncEnabled;
    }
    set graphToTextSync(sync) {
        this._graphToTextSyncEnabled = sync;
        // Try commenting the sync out. Just set the flag but don't kick off graph->text conversion
        // this.performGraphToTextSyncing();
    }
    performGraphToTextSyncing() {
        if (this._graphToTextSyncEnabled) {
            this.graphToTextEventEmitter.emit();
        }
    }
    createHandle(element, kind, action, location) {
        if (!location) {
            let bbox = element.model.getBBox();
            location = bbox.origin().offset(bbox.width / 2, bbox.height / 2);
        }
        let handle = Shapes.Factory.createHandle({
            renderer: this.renderer,
            paper: this.paper,
            parent: element.model,
            kind: kind,
            position: location
        });
        const view = this.paper.findViewByModel(handle);
        view.on('cell:pointerdown', () => {
            if (action) {
                action();
            }
        });
        view.on('cell:mouseover', () => {
            handle.attr('image/filter', {
                name: 'dropShadow',
                args: { dx: 1, dy: 1, blur: 1, color: 'black' }
            });
        });
        view.on('cell:mouseout', () => {
            handle.removeAttr('image/filter');
        });
        view.setInteractivity(false);
        return handle;
    }
    removeEmbeddedChildrenOfType(element, types) {
        let embeds = element.getEmbeddedCells();
        for (let i = 0; i < embeds.length; i++) {
            if (types.indexOf(embeds[i].get('type')) >= 0) {
                embeds[i].remove();
            }
        }
    }
    get selection() {
        return this._selection;
    }
    set selection(newSelection) {
        if (newSelection && (newSelection.model.get('type') === joint.shapes.flo.DECORATION_TYPE || newSelection.model.get('type') === joint.shapes.flo.HANDLE_TYPE)) {
            newSelection = this.paper.findViewByModel(this.graph.getCell(newSelection.model.get('parent')));
        }
        if (newSelection && (!newSelection.model.attr('metadata') || newSelection.model.attr('metadata/metadata/unselectable'))) {
            newSelection = undefined;
        }
        if (newSelection !== this._selection) {
            if (this._selection) {
                const elementview = this.paper.findViewByModel(this._selection.model);
                if (elementview) { // May have been removed from the graph
                    this.removeEmbeddedChildrenOfType(elementview.model, joint.shapes.flo.HANDLE_TYPE);
                    elementview.unhighlight();
                }
            }
            if (newSelection) {
                newSelection.highlight();
                if (this.editor && this.editor.createHandles) {
                    this.editor.createHandles(this.editorContext, (owner, kind, action, location) => this.createHandle(owner, kind, action, location), newSelection);
                }
            }
            this._selection = newSelection;
        }
    }
    get readOnlyCanvas() {
        return this._readOnlyCanvas;
    }
    set readOnlyCanvas(value) {
        if (this._readOnlyCanvas === value) {
            // Nothing to do
            return;
        }
        if (value) {
            this.selection = undefined;
        }
        if (this.graph) {
            this.graph.getLinks().forEach((link) => {
                if (value) {
                    link.attr('.link-tools/display', 'none');
                    link.attr('.marker-vertices/display', 'none');
                    link.attr('.connection-wrap/display', 'none');
                }
                else {
                    link.removeAttr('.link-tools/display');
                    if (this.editor && this.editor.allowLinkVertexEdit) {
                        link.removeAttr('.marker-vertices/display');
                    }
                    link.removeAttr('.connection-wrap/display');
                }
            });
        }
        this._readOnlyCanvas = value;
    }
    /**
     * Displays graphical feedback for the drag and drop in progress based on current drag and drop descriptor object
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    showDragFeedback(dragDescriptor) {
        if (this.editor && this.editor.showDragFeedback) {
            this.editor.showDragFeedback(this.editorContext, dragDescriptor);
        }
        else {
            let magnet;
            if (dragDescriptor.source && dragDescriptor.source.view) {
                joint.V(dragDescriptor.source.view.el).addClass('dnd-source-feedback');
                if (dragDescriptor.source.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.source.view, dragDescriptor.source.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).addClass('dnd-source-feedback');
                    }
                }
            }
            if (dragDescriptor.target && dragDescriptor.target.view) {
                joint.V(dragDescriptor.target.view.el).addClass('dnd-target-feedback');
                if (dragDescriptor.target.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.target.view, dragDescriptor.target.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).addClass('dnd-target-feedback');
                    }
                }
            }
        }
    }
    /**
     * Hides graphical feedback for the drag and drop in progress based on current drag and drop descriptor object
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    hideDragFeedback(dragDescriptor) {
        if (this.editor && this.editor.hideDragFeedback) {
            this.editor.hideDragFeedback(this.editorContext, dragDescriptor);
        }
        else {
            let magnet;
            if (dragDescriptor.source && dragDescriptor.source.view) {
                joint.V(dragDescriptor.source.view.el).removeClass('dnd-source-feedback');
                if (dragDescriptor.source.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.source.view, dragDescriptor.source.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).removeClass('dnd-source-feedback');
                    }
                }
            }
            if (dragDescriptor.target && dragDescriptor.target.view) {
                joint.V(dragDescriptor.target.view.el).removeClass('dnd-target-feedback');
                if (dragDescriptor.target.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.target.view, dragDescriptor.target.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).removeClass('dnd-target-feedback');
                    }
                }
            }
        }
    }
    /**
     * Sets the new DnD info object - the descriptor for DnD
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    setDragDescriptor(dragDescriptor) {
        if (this.highlighted === dragDescriptor) {
            return;
        }
        if (this.highlighted && dragDescriptor && _.isEqual(this.highlighted.sourceComponent, dragDescriptor.sourceComponent)) {
            if (this.highlighted.source === dragDescriptor.source && this.highlighted.target === dragDescriptor.target) {
                return;
            }
            if (this.highlighted.source &&
                dragDescriptor.source &&
                this.highlighted.target &&
                dragDescriptor.target &&
                this.highlighted.source.view.model === dragDescriptor.source.view.model &&
                this.highlighted.source.cssClassSelector === dragDescriptor.source.cssClassSelector &&
                this.highlighted.target.view.model === dragDescriptor.target.view.model &&
                this.highlighted.target.cssClassSelector === dragDescriptor.target.cssClassSelector) {
                return;
            }
        }
        if (this.highlighted) {
            this.hideDragFeedback(this.highlighted);
        }
        this.highlighted = dragDescriptor;
        if (this.highlighted) {
            this.showDragFeedback(this.highlighted);
        }
    }
    /**
     * Handles DnD events when a node is being dragged over canvas
     *
     * @param draggedView The Joint JS view object being dragged
     * @param targetUnderMouse The Joint JS view under mouse cursor
     * @param x X coordinate of the mouse on the canvas
     * @param y Y coordinate of the mosue on the canvas
     * @param context DnD context (palette or canvas)
     */
    handleNodeDragging(draggedView, targetUnderMouse, x, y, sourceComponent) {
        if (this.editor && this.editor.calculateDragDescriptor) {
            this.setDragDescriptor(this.editor.calculateDragDescriptor(this.editorContext, draggedView, targetUnderMouse, joint.g.point(x, y), sourceComponent));
        }
    }
    /**
     * Handles DnD drop event when a node is being dragged and dropped on the main canvas
     */
    handleNodeDropping() {
        if (this.highlighted && this.editor && this.editor.handleNodeDropping) {
            this.editor.handleNodeDropping(this.editorContext, this.highlighted);
        }
        this.setDragDescriptor(undefined);
    }
    /**
     * Hides DOM Node (used to determine drop target DOM element)
     * @param domNode DOM node to hide
     * @returns
     */
    _hideNode(domNode) {
        let oldVisibility = {
            visibility: domNode.style ? domNode.style.display : undefined,
            children: []
        };
        for (let i = 0; i < domNode.children.length; i++) {
            let node = domNode.children.item(i);
            if (node instanceof HTMLElement) {
                oldVisibility.children.push(this._hideNode(node));
            }
        }
        domNode.style.display = 'none';
        return oldVisibility;
    }
    /**
     * Restored DOM node original visibility (used to determine drop target DOM element)
     * @param domNode DOM node to restore visibility of
     * @param oldVisibility original visibility parameter
     */
    _restoreNodeVisibility(domNode, oldVisibility) {
        if (domNode.style) {
            domNode.style.display = oldVisibility.visibility;
        }
        let j = 0;
        for (let i = 0; i < domNode.childNodes.length; i++) {
            if (j < oldVisibility.children.length) {
                let node = domNode.children.item(i);
                if (node instanceof HTMLElement) {
                    this._restoreNodeVisibility(node, oldVisibility.children[j++]);
                }
            }
        }
    }
    /**
     * Unfortunately we can't just use event.target because often draggable shape on the canvas overlaps the target.
     * We can easily find the element(s) at location, but only nodes :-( Unclear how to find links at location
     * (bounding box of a link for testing is bad).
     * The result of that is that links can only be the drop target when dragging from the palette currently.
     * When DnDing shapes on the canvas drop target cannot be a link.
     *
     * Excluded views enables you to choose to filter some possible answers (useful in the case where elements are stacked
     * - e.g. Drag-n-Drop)
     */
    getTargetViewFromEvent(event, x, y, excludeViews = []) {
        if (!x && !y) {
            let l = this.paper.snapToGrid({ x: event.clientX, y: event.clientY });
            x = l.x;
            y = l.y;
        }
        // TODO: See if next code paragraph is needed. Most likely it's just code executed for nothing
        // let elements = this.graph.findModelsFromPoint(joint.g.point(x, y));
        // let underMouse = elements.find(e => !_.isUndefined(excludeViews.find(x => x === this.paper.findViewByModel(e))));
        // if (underMouse) {
        //   return underMouse;
        // }
        let oldVisibility = excludeViews.map(_x => this._hideNode(_x.el));
        let targetElement = document.elementFromPoint(event.clientX, event.clientY);
        excludeViews.forEach((excluded, i) => {
            this._restoreNodeVisibility(excluded.el, oldVisibility[i]);
        });
        return this.paper.findView($(targetElement));
    }
    handleDnDFromPalette(dndEvent) {
        switch (dndEvent.type) {
            case Flo.DnDEventType.DRAG:
                this.handleDragFromPalette(dndEvent);
                break;
            case Flo.DnDEventType.DROP:
                this.handleDropFromPalette(dndEvent);
                break;
            default:
                break;
        }
    }
    handleDragFromPalette(dnDEvent) {
        console.debug('Dragging from palette');
        if (dnDEvent.view && !this.readOnlyCanvas) {
            let location = this.paper.snapToGrid({ x: dnDEvent.event.clientX, y: dnDEvent.event.clientY });
            this.handleNodeDragging(dnDEvent.view, this.getTargetViewFromEvent(dnDEvent.event, location.x, location.y, [dnDEvent.view]), location.x, location.y, Constants.PALETTE_CONTEXT);
        }
    }
    createNode(metadata, props, position) {
        return Shapes.Factory.createNode({
            renderer: this.renderer,
            paper: this.paper,
            metadata: metadata,
            props: props,
            position: position
        });
    }
    createLink(source, target, metadata, props) {
        return Shapes.Factory.createLink({
            renderer: this.renderer,
            paper: this.paper,
            source: source,
            target: target,
            metadata: metadata,
            props: props
        });
    }
    handleDropFromPalette(event) {
        let cellview = event.view;
        let evt = event.event;
        if (this.paper.el === evt.target || $.contains(this.paper.el, evt.target)) {
            if (this.readOnlyCanvas) {
                this.setDragDescriptor(undefined);
            }
            else {
                let metadata = cellview.model.attr('metadata');
                let props = cellview.model.attr('props');
                let position = this.paper.snapToGrid({ x: evt.clientX, y: evt.clientY });
                /* Calculate target element before creating the new
                 * element under mouse location. Otherwise target
                 * element would be the newly created element because
                 * it's under the mouse pointer
                 */
                let targetElement = this.getTargetViewFromEvent(evt, position.x, position.y, [event.view]);
                let newNode = this.createNode(metadata, props, position);
                let newView = this.paper.findViewByModel(newNode);
                this.handleNodeDragging(newView, targetElement, position.x, position.y, Constants.PALETTE_CONTEXT);
                this.handleNodeDropping();
            }
        }
    }
    fitToContent(gridWidth, gridHeight, padding, opt) {
        const paper = this.paper;
        if (joint.util.isObject(gridWidth)) {
            // first parameter is an option object
            opt = gridWidth;
            gridWidth = opt.gridWidth || 1;
            gridHeight = opt.gridHeight || 1;
            padding = opt.padding || 0;
        }
        else {
            opt = opt || {};
            gridWidth = gridWidth || 1;
            gridHeight = gridHeight || 1;
            padding = padding || 0;
        }
        const paddingJson = joint.util.normalizeSides(padding);
        // Calculate the paper size to accomodate all the graph's elements.
        const bbox = joint.V(paper.viewport).getBBox();
        const currentScale = paper.scale();
        const currentTranslate = paper.translate();
        bbox.x *= currentScale.sx;
        bbox.y *= currentScale.sy;
        bbox.width *= currentScale.sx;
        bbox.height *= currentScale.sy;
        let calcWidth = Math.max((bbox.width + bbox.x) / gridWidth, 1) * gridWidth;
        let calcHeight = Math.max((bbox.height + bbox.y) / gridHeight, 1) * gridHeight;
        let tx = 0;
        let ty = 0;
        if ((opt.allowNewOrigin === 'negative' && bbox.x < 0) || (opt.allowNewOrigin === 'positive' && bbox.x >= 0) || opt.allowNewOrigin === 'any') {
            tx = (-bbox.x / gridWidth) * gridWidth;
            tx += paddingJson.left;
        }
        else if (opt.allowNewOrigin === 'same') {
            tx = currentTranslate.tx;
        }
        calcWidth += tx;
        if ((opt.allowNewOrigin === 'negative' && bbox.y < 0) || (opt.allowNewOrigin === 'positive' && bbox.y >= 0) || opt.allowNewOrigin === 'any') {
            ty = (-bbox.y / gridHeight) * gridHeight;
            ty += paddingJson.top;
        }
        else if (opt.allowNewOrigin === 'same') {
            ty = currentTranslate.ty;
        }
        calcHeight += ty;
        calcWidth += paddingJson.right;
        calcHeight += paddingJson.bottom;
        // Make sure the resulting width and height are greater than minimum.
        calcWidth = Math.max(calcWidth, opt.minWidth || 0);
        calcHeight = Math.max(calcHeight, opt.minHeight || 0);
        // Make sure the resulting width and height are lesser than maximum.
        calcWidth = Math.min(calcWidth, opt.maxWidth || Number.MAX_VALUE);
        calcHeight = Math.min(calcHeight, opt.maxHeight || Number.MAX_VALUE);
        const dimensionChange = calcWidth !== paper.options.width || calcHeight !== paper.options.height;
        const originChange = tx !== currentTranslate.tx || ty !== currentTranslate.ty;
        // Change the dimensions only if there is a size discrepency or an origin change
        if (originChange) {
            paper.translate(tx, ty);
        }
        if (dimensionChange) {
            paper.setDimensions(calcWidth, calcHeight);
        }
    }
    autosizePaper() {
        let parent = $('#paper-container', this.element.nativeElement);
        const parentWidth = parent.innerWidth();
        const parentHeight = parent.innerHeight();
        this.fitToContent(this.gridSize, this.gridSize, this.paperPadding, {
            minWidth: parentWidth - Flo.SCROLLBAR_WIDTH,
            minHeight: parentHeight - Flo.SCROLLBAR_WIDTH,
            allowNewOrigin: 'same'
        });
    }
    fitToPage() {
        let parent = $('#paper-container', this.element.nativeElement);
        let minScale = this.minZoom / 100;
        let maxScale = 2;
        const parentWidth = parent.innerWidth();
        const parentHeight = parent.innerHeight();
        this.paper.scaleContentToFit({
            padding: this.paperPadding,
            minScaleX: minScale,
            minScaleY: minScale,
            maxScaleX: maxScale,
            maxScaleY: maxScale,
            fittingBBox: { x: 0, y: 0, width: parentWidth - Flo.SCROLLBAR_WIDTH, height: parentHeight - Flo.SCROLLBAR_WIDTH }
        });
        /**
         * Size the canvas appropriately and allow origin movement
         */
        this.fitToContent(this.gridSize, this.gridSize, this.paperPadding, {
            minWidth: parentWidth,
            minHeight: parentHeight,
            maxWidth: parentWidth,
            maxHeight: parentHeight,
            allowNewOrigin: 'any'
        });
    }
    get zoomPercent() {
        return Math.round(this.paper.scale().sx * 100);
    }
    set zoomPercent(percent) {
        if (!isNaN(percent)) {
            if (percent < this.minZoom) {
                percent = this.minZoom;
            }
            else if (percent >= this.maxZoom) {
                percent = this.maxZoom;
            }
            else {
                if (percent <= 0) {
                    percent = 0.00001;
                }
            }
            this.paper.scale(percent / 100, percent / 100);
        }
    }
    get gridSize() {
        return this._gridSize;
    }
    set gridSize(size) {
        if (!isNaN(size) && size >= 1) {
            this._gridSize = size;
            if (this.paper) {
                this.paper.setGridSize(size);
            }
        }
    }
    validateContent() {
        return new Promise(resolve => {
            if (this.editor && this.editor.validate) {
                return this.editor
                    .validate(this.graph, this.dsl, this.editorContext)
                    .then(allMarkers => {
                    this.graph.getCells()
                        .forEach((cell) => this.markElement(cell, allMarkers.has(cell.id) ? allMarkers.get(cell.id) : []));
                    this.validationMarkers.emit(allMarkers);
                    this.contentValidated.emit(true);
                    resolve();
                });
            }
            else {
                resolve();
            }
        });
    }
    markElement(cell, markers) {
        cell.set('markers', markers);
        // Old legacy code below consider removing
        let errorMessages = markers.map(m => m.message);
        let errorCell = cell.getEmbeddedCells().find((e) => e.attr('./kind') === Constants.ERROR_DECORATION_KIND);
        if (errorCell) {
            if (errorMessages.length === 0) {
                errorCell.remove();
            }
            else {
                // Without rewrite we merge this list with existing errors
                errorCell.attr('messages', errorMessages, { rewrite: true });
            }
        }
        else if (errorMessages.length > 0) {
            let error = Shapes.Factory.createDecoration({
                renderer: this.renderer,
                paper: this.paper,
                parent: cell,
                kind: Constants.ERROR_DECORATION_KIND,
                messages: errorMessages
            });
            if (error) {
                const view = this.paper.findViewByModel(error);
                view.setInteractivity(false);
            }
        }
    }
    doLayout() {
        if (this.renderer && this.renderer.layout) {
            return this.renderer.layout(this.paper);
        }
    }
    set dsl(dslText) {
        if (this._dslText !== dslText) {
            this._dslText = dslText;
            this.textToGraphEventEmitter.emit();
        }
    }
    get dsl() {
        return this._dslText;
    }
    /**
     * Ask the server to parse the supplied text into a JSON graph of nodes and links,
     * then update the view based on that new information.
     */
    updateGraphRepresentation() {
        console.debug(`Updating graph to represent '${this._dslText}'`);
        if (this.metamodel && this.metamodel.textToGraph) {
            return this.metamodel.textToGraph(this.editorContext, this._dslText).then(() => {
                this.textToGraphConversionCompleted.next();
                return this.validateContent();
            });
        }
        else {
            this.textToGraphConversionCompleted.next();
            return this.validateContent();
        }
    }
    updateTextRepresentation() {
        if (this.metamodel && this.metamodel.graphToText) {
            return this.metamodel.graphToText(this.editorContext).then(text => {
                if (this._dslText !== text) {
                    this._dslText = text;
                    this.dslChange.emit(text);
                }
                this.graphToTextConversionCompleted.next();
                return this.validateContent();
            })
                .catch(error => {
                // Validation may reveal why the graph couldn't be
                // converted so let it run
                this.graphToTextConversionCompleted.next();
                return this.validateContent();
            });
        }
        else {
            this.graphToTextConversionCompleted.next();
            return this.validateContent();
        }
    }
    initMetamodel() {
        this.metamodel.load().then(data => {
            this.updateGraphRepresentation();
            let textSyncSubscription = this.graphToTextEventEmitter.pipe(debounceTime(100)).subscribe(() => {
                if (this._graphToTextSyncEnabled) {
                    this.updateTextRepresentation();
                }
            });
            this._disposables.add(Disposable.create(() => textSyncSubscription.unsubscribe()));
            // Setup content validated event emitter. Emit not validated when graph to text conversion required
            let graphValidatedSubscription1 = this.graphToTextEventEmitter.subscribe(() => this.contentValidated.emit(false));
            this._disposables.add(Disposable.create(() => graphValidatedSubscription1.unsubscribe));
            // let validationSubscription = this.validationEventEmitter.pipe(debounceTime(100)).subscribe(() => this.validateGraph());
            // this._disposables.add(Disposable.create(() => validationSubscription.unsubscribe()));
            let graphSyncSubscription = this.textToGraphEventEmitter.pipe(debounceTime(300)).subscribe(() => this.updateGraphRepresentation());
            this._disposables.add(Disposable.create(() => graphSyncSubscription.unsubscribe()));
            // Setup content validated event emitter. Emit not validated when text to graph conversion required
            let graphValidatedSubscription2 = this.textToGraphEventEmitter.subscribe(() => this.contentValidated.emit(false));
            this._disposables.add(Disposable.create(() => graphValidatedSubscription2.unsubscribe));
            if (this.editor && this.editor.setDefaultContent) {
                this.editor.setDefaultContent(this.editorContext, data);
            }
        });
    }
    initGraph() {
        this.graph = new joint.dia.Graph();
        this.graph.set('type', Constants.CANVAS_CONTEXT);
        this.graph.set('paperPadding', this.paperPadding);
    }
    handleNodeCreation(node) {
        node.on('change:size', this._resizeHandler);
        node.on('change:position', this._resizeHandler);
        if (node.attr('metadata')) {
            node.on('change:attrs', (cell, attrs, changeData) => {
                let propertyPath = changeData ? changeData.propertyPath : undefined;
                if (propertyPath) {
                    let propAttr = propertyPath.substr(propertyPath.indexOf('/') + 1);
                    if (propAttr.indexOf('metadata') === 0 ||
                        propAttr.indexOf('props') === 0 ||
                        (this.renderer && this.renderer.isSemanticProperty && this.renderer.isSemanticProperty(propAttr, node))) {
                        this.performGraphToTextSyncing();
                    }
                    if (this.renderer && this.renderer.refreshVisuals) {
                        this.renderer.refreshVisuals(node, propAttr, this.paper);
                    }
                }
            });
        }
        node.on('change:markers', () => {
            if (this.renderer && this.renderer.markersChanged) {
                this.renderer.markersChanged(node, this.paper);
            }
        });
    }
    /**
     * Forwards a link event occurrence to any handlers in the editor service, if they are defined. Event examples
     * are 'change:source', 'change:target'.
     */
    handleLinkEvent(event, link) {
        if (this.renderer && this.renderer.handleLinkEvent) {
            this.renderer.handleLinkEvent(this.editorContext, event, link);
        }
    }
    handleLinkCreation(link) {
        link.on('change:source', (l) => {
            this.autosizePaper();
            let newSourceId = l.get('source').id;
            let oldSourceId = l.previous('source').id;
            if (newSourceId !== oldSourceId) {
                this.performGraphToTextSyncing();
            }
            this.handleLinkEvent('change:source', l);
        });
        link.on('change:target', (l) => {
            this.autosizePaper();
            let newTargetId = l.get('target').id;
            let oldTargetId = l.previous('target').id;
            if (newTargetId !== oldTargetId) {
                this.performGraphToTextSyncing();
            }
            this.handleLinkEvent('change:target', l);
        });
        link.on('change:vertices', this._resizeHandler);
        link.on('change:attrs', (cell, attrs, changeData) => {
            let propertyPath = changeData ? changeData.propertyPath : undefined;
            if (propertyPath) {
                let propAttr = propertyPath.substr(propertyPath.indexOf('/') + 1);
                if (propAttr.indexOf('metadata') === 0 ||
                    propAttr.indexOf('props') === 0 ||
                    (this.renderer && this.renderer.isSemanticProperty && this.renderer.isSemanticProperty(propAttr, link))) {
                    let sourceId = link.get('source').id;
                    let targetId = link.get('target').id;
                    this.performGraphToTextSyncing();
                }
                if (this.renderer && this.renderer.refreshVisuals) {
                    this.renderer.refreshVisuals(link, propAttr, this.paper);
                }
            }
        });
        this.paper.findViewByModel(link).on('link:options', () => this.handleLinkEvent('options', link));
        if (this.readOnlyCanvas) {
            link.attr('.link-tools/display', 'none');
        }
        this.handleLinkEvent('add', link);
    }
    initGraphListeners() {
        this.graph.on('add', (element) => {
            if (element instanceof joint.dia.Link) {
                this.handleLinkCreation(element);
            }
            else if (element instanceof joint.dia.Element) {
                this.handleNodeCreation(element);
            }
            if (element.get('type') === joint.shapes.flo.NODE_TYPE || element.get('type') === joint.shapes.flo.LINK_TYPE) {
                this.performGraphToTextSyncing();
            }
            this.autosizePaper();
        });
        this.graph.on('remove', (element) => {
            if (element instanceof joint.dia.Link) {
                this.handleLinkEvent('remove', element);
            }
            if (this.selection && this.selection.model === element) {
                this.selection = undefined;
            }
            if (element.isLink()) {
                window.setTimeout(() => this.performGraphToTextSyncing(), 100);
            }
            else if (element.get('type') === joint.shapes.flo.NODE_TYPE) {
                this.performGraphToTextSyncing();
            }
            this.autosizePaper();
        });
        // Set if link is fan-routed. Should be called before routing call
        this.graph.on('change:vertices', (link, changed, opt) => {
            if (opt.fanRouted) {
                link.set('fanRouted', true);
            }
            else {
                link.unset('fanRouted');
            }
        });
        // adjust vertices when a cell is removed or its source/target was changed
        this.graph.on('add remove change:source change:target change:vertices change:position', _.partial(Utils.fanRoute, this.graph));
        this.graph.on('startDeletion', (cell) => {
            if (this.editor && this.editor.preDelete) {
                if (this.editor.preDelete(this.editorContext, this.selection.model)) {
                    cell.remove();
                }
            }
            else {
                cell.remove();
            }
        });
    }
    initPaperListeners() {
        // https://stackoverflow.com/questions/20463533/how-to-add-an-onclick-event-to-a-joint-js-element
        this.paper.on('cell:pointerup', (cellView) => {
            if (!this.readOnlyCanvas) {
                this.selection = cellView;
            }
        });
        this.paper.on('blank:pointerdown', () => {
            this.selection = undefined;
        });
        this.paper.on('scale', this._resizeHandler);
        this.paper.on('all', function () {
            if (Utils.isCustomPaperEvent(arguments)) {
                arguments[2].trigger.apply(arguments[2], [arguments[0], arguments[1], arguments[3], arguments[4]]);
            }
        });
        this.paper.on('dragging-node-over-canvas', (dndEvent) => {
            console.debug(`Canvas DnD type = ${dndEvent.type}`);
            let location = this.paper.snapToGrid({ x: dndEvent.event.clientX, y: dndEvent.event.clientY });
            switch (dndEvent.type) {
                case Flo.DnDEventType.DRAG:
                    this.handleNodeDragging(dndEvent.view, this.getTargetViewFromEvent(dndEvent.event, location.x, location.y, [dndEvent.view]), location.x, location.y, Constants.CANVAS_CONTEXT);
                    break;
                case Flo.DnDEventType.DROP:
                    this.handleNodeDropping();
                    break;
                default:
                    break;
            }
        });
        // JointJS now no longer grabs focus if working in a paper element - crude...
        // $('#flow-view', this.element.nativeElement).on('mousedown', () => {
        // $('#palette-filter-textfield', this.element.nativeElement).focus();
        // });
    }
    initPaper() {
        let options = {
            el: $('#paper', this.element.nativeElement),
            gridSize: this._gridSize,
            drawGrid: true,
            model: this.graph,
            elementView: this.renderer && this.renderer.getNodeView ? this.renderer.getNodeView() : joint.shapes.flo.ElementView /*joint.dia.ElementView*/,
            linkView: this.renderer && this.renderer.getLinkView ? this.renderer.getLinkView() : joint.shapes.flo.LinkView,
            // Enable link snapping within 25px lookup radius
            snapLinks: { radius: 25 },
            defaultLink: /*this.renderer && this.renderer.createDefaultLink ? this.renderer.createDefaultLink: new joint.shapes.flo.Link*/ (cellView, magnet) => {
                if (this.renderer && this.renderer.createLink) {
                    let linkEnd = {
                        id: cellView.model.id
                    };
                    if (magnet) {
                        linkEnd.selector = cellView.getSelector(magnet, undefined);
                    }
                    if (magnet.getAttribute('port')) {
                        linkEnd.port = magnet.getAttribute('port');
                    }
                    if (magnet.getAttribute('port') === 'input') {
                        return this.renderer.createLink(undefined, linkEnd);
                    }
                    else {
                        return this.renderer.createLink(linkEnd, undefined);
                    }
                }
                else {
                    return new joint.shapes.flo.Link();
                }
            },
            // decide whether to create a link if the user clicks a magnet
            validateMagnet: (cellView, magnet) => {
                if (this.readOnlyCanvas) {
                    return false;
                }
                else {
                    if (this.editor && this.editor.validatePort) {
                        return this.editor.validatePort(this.editorContext, cellView, magnet);
                    }
                    else {
                        return true;
                    }
                }
            },
            interactive: (cellView, event) => {
                if (this.readOnlyCanvas) {
                    return false;
                }
                else {
                    if (this.editor && this.editor.interactive) {
                        if (typeof this.editor.interactive === 'function') {
                            // Type for interactive is wrong in JointJS have to cast to <any>
                            return this.editor.interactive(cellView, event);
                        }
                        else {
                            return this.editor.interactive;
                        }
                    }
                    return true;
                }
            },
            highlighting: this.editor && this.editor.highlighting ? this.editor.highlighting : {
                'default': {
                    name: 'addClass',
                    options: {
                        className: 'highlighted'
                    }
                }
            },
            markAvailable: true
        };
        if (this.renderer && this.renderer.getLinkAnchorPoint) {
            options.linkConnectionPoint = this.renderer.getLinkAnchorPoint;
        }
        if (this.editor && this.editor.validateLink) {
            const self = this;
            options.validateConnection = (cellViewS, magnetS, cellViewT, magnetT, end, linkView) => self.editor.validateLink(this.editorContext, cellViewS, magnetS, cellViewT, magnetT, end === 'source', linkView);
        }
        // The paper is what will represent the graph on the screen
        this.paper = new joint.dia.Paper(options);
        this._disposables.add(Disposable.create(() => this.paper.remove()));
        this.paper.options.highlighterNamespace['addParentClass'] = {
            /**
             * @param {joint.dia.CellView} cellView
             * @param {Element} magnetEl
             * @param {object=} opt
             */
            highlight(cellView, magnetEl, opt) {
                opt = opt || {};
                const className = opt.className || this.className;
                joint.V(magnetEl.parentElement).addClass(className);
            },
            /**
             * @param {joint.dia.CellView} cellView
             * @param {Element} magnetEl
             * @param {object=} opt
             */
            unhighlight(cellView, magnetEl, opt) {
                opt = opt || {};
                const className = opt.className || this.className;
                joint.V(magnetEl.parentElement).removeClass(className);
            }
        };
    }
    updatePaletteReadyState(ready) {
        this.paletteReady.next(ready);
    }
};
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "metamodel", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "renderer", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "editor", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Number),
    tslib_1.__metadata("design:paramtypes", [Number])
], EditorComponent.prototype, "paletteSize", null);
tslib_1.__decorate([
    Output(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "paletteSizeChange", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "searchFilterPlaceHolder", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "paletteEntryPadding", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "minZoom", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "maxZoom", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "zoomStep", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "paperPadding", void 0);
tslib_1.__decorate([
    Output(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "floApi", void 0);
tslib_1.__decorate([
    Output(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "validationMarkers", void 0);
tslib_1.__decorate([
    Output(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "contentValidated", void 0);
tslib_1.__decorate([
    Output(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "dslChange", void 0);
tslib_1.__decorate([
    Output(),
    tslib_1.__metadata("design:type", Object)
], EditorComponent.prototype, "onProperties", void 0);
tslib_1.__decorate([
    Input(),
    tslib_1.__metadata("design:type", String),
    tslib_1.__metadata("design:paramtypes", [String])
], EditorComponent.prototype, "dsl", null);
EditorComponent = tslib_1.__decorate([
    Component({
        selector: 'flo-editor',
        template: `
    <ng-content select="[header]"></ng-content>
    <div id="flow-view" class="flow-view" style="position:relative">
      <div id="canvas" class="canvas" style="position:relative; display: block; width: 100%; height: 100%;">
        <div *ngIf="!noPalette" id="palette-container" class="palette-container" style="overflow:hidden;">
          <flo-palette [metamodel]="metamodel" [renderer]="renderer" [paletteSize]="paletteSize"
                       [paletteEntryPadding]="paletteEntryPadding || {width: 12, height: 12}"
                       (onPaletteEntryDrop)="handleDnDFromPalette($event)"
                        (paletteReady)="updatePaletteReadyState($event)"
                        (paletteFocus)="graphToTextSync=true"
                        [searchFilterPlaceHolder]="searchFilterPlaceHolder">
          </flo-palette>
        </div>

        <div id="sidebar-resizer" *ngIf="!noPalette"
          resizer
          [splitSize]="paletteSize"
          (sizeChange)="paletteSize = $event"
          [resizerWidth]="6"
          [resizerLeft]="'#palette-container'"
          [resizerRight]="'#paper-container'">
        </div>

          <flo-editor-paper tabindex="0" (onDelete)="deleteSelected()" (onBlur)="selection = null"
                            (onProperties)="onPropertiesHandle()">
            <div id="paper" class="paper" tabindex="0" style="overflow: hidden; position: absolute; display: block; height:100%; width:100%; overflow:auto;"></div>
            <ng-content select="[canvas]"></ng-content>
          </flo-editor-paper>

          <span class="canvas-controls-container" ng-if="canvasControls">
            <table ng-if="canvasControls.zoom" class="canvas-control zoom-canvas-control">
              <tbody>
                <tr>
                  <td>
                    <input class="zoom-canvas-input canvas-control zoom-canvas-control" type="text"
                           data-inline="true" [(ngModel)]="zoomPercent"
                           size="3">
                  </td>
                  <td>
                    <label class="canvas-control zoom-canvas-label">%</label>
                  </td>
                  <td>
                    <input type="range" data-inline="true" [(ngModel)]="zoomPercent"
                           [step]="zoomStep"
                           [max]="maxZoom" [min]="minZoom" data-type="range"
                           name="range" class="canvas-control zoom-canvas-control">
                  </td>
                </tr>
              </tbody>
            </table>
          </span>

      </div>
    </div>
    <ng-content select="[footer]"></ng-content>
  `,
        styles: [`
    flo-view {
      width:100%;
      height:100%;
      margin: 0;
      background-color: #eeeeee;
      font-family: "Varela Round",sans-serif;
      -webkit-user-select: none;
      -khtml-user-select: none;
      -moz-user-select: none;
      -o-user-select: none;
      user-select: none;
    }

    .canvas {
      border: 1px solid;
      border-color: #6db33f;
      border-radius: 2px;
      margin-top: 3px;
    }

    /* Canvas contains the palette on the left and the paper on the right */

    .paper {
      padding: 0px;
      background-color: #ffffff;
      /* 	height: 100%;
          width: 100%;
          position: relative;
          overflow: hidden;
       *//* 	margin-left: 400px; */
    }

    #sidebar-resizer {
      background-color: #34302d;
      position: absolute;
      top: 0;
      bottom: 0;
      width: 6px;
      cursor: e-resize;
    }

    #palette-container {
      background-color: #EEE;
      position: absolute;
      top: 0;
      bottom: 0;
      left: 0;
      overflow: auto;
    }

    #paper-container {
      position: absolute;
      top: 0;
      bottom: 0;
      right: 0;
      overflow: auto;
      color: #FFF;
      background-color: #FFF;
      outline: none;
    }

    /* Tooltip START */

    .node-tooltip .tooltip-description {
      margin-top: 5px;
      margin-left: 0px;
      margin-bottom: 5px;
    }

    .node-tooltip {
      display:none;
      position:absolute;
      border:1px solid #333;
      background-color:#34302d;/*#161616;*/
      border-radius:5px;
      padding:5px;
      color:#fff;
      /*	font-size:12px Arial;*/
      font-family: "Varela Round",sans-serif;
      font-size: 19px;
      z-index: 100;
    }

    .tooltip-title-type {
      font-size: 24px;
      font-weight: bold;
    }

    .tooltip-title-group {
      padding-left: 5px;
      font-size: 20px;
      font-style: italic;
    }

    .node-tooltip-option-name {
      font-family: monospace;/*"Varela Round",sans-serif;*/
      font-size: 17px;
      font-weight: bold;
      padding-right: 20px;

    }

    .node-tooltip-option-description {
      font-family: "Varela Round",sans-serif;
      font-size: 18px;
    }

    /* Tooltip END */


    /* Validation Error Marker on Canvas START */

    .error-tooltip p {
      margin-top: 5px;
      margin-left: 0px;
      margin-bottom: 5px;
      color:#fff;
    }
    .error-tooltip {
      display:none;
      position:absolute;
      border:1px solid #333;
      background-color:red;/*#161616;*/
      border-radius:5px;
      padding:5px;
      color:#fff;
      /*	font-size:12px Arial;*/
      font-family: "Varela Round",sans-serif;
      font-size: 20px;
      z-index: 100;
    }

    /* Validation Error Marker on Canvas END */

    /* Controls on Canvas START */

    .canvas-controls-container {
      position: absolute;
      right: 15px;
      top: 5px;
    }

    .canvas-control {
      background: transparent;
      font-family: "Varela Round",sans-serif;
      font-size: 11px;
      vertical-align: middle;
      margin: 0px;
    }

    .zoom-canvas-control {
      border: 0px;
      padding: 0px;
      margin: 0px;
      outline: none;
    }

    .zoom-canvas-input {
      text-align: right;
      font-weight:bold;
      color: black;
      background-color: transparent;
    }

    .zoom-canvas-label {
      padding-right: 4px;
      color: black;
    }

    /* Controls on Canvas END */




    /* START - FLO CANVAS STYLES - override joint js styles */

    .highlighted {
      outline: none;
    }

    .joint-type-handle {
      cursor: pointer;
    }

    .available-magnet {
      stroke-width: 3;
    }

    .link {
      fill: none;
      stroke: #ccc;
      stroke-width: 1.5px;
    }

    .link-tools .tool-options {
      display: none;       /* by default, we don't display link options tool */
    }

    /* Make transparent the circle around the link-tools (cog) icon. It'll allow shape to have a circle clicking area */
    .link-tools .tool-options circle {
      fill: transparent;
      stroke: transparent;
    }

    .link-tools .tool-options path {
      fill: black;
      stroke: black;
    }

    .link-tools .tool-remove circle {
      fill: red;
      stroke: red;
    }

    .link-tools .tool-remove path {
      fill: white;
      stroke: white;
    }

    .link-tools-container {
      stroke-width: 0;
      fill: transparent;
    }

    /* END - FLO CANVAS STYLES */
  `],
        encapsulation: ViewEncapsulation.None
    }),
    tslib_1.__metadata("design:paramtypes", [ElementRef])
], EditorComponent);
export { EditorComponent };
//# sourceMappingURL=editor.component.js.map