import * as tslib_1 from "tslib";
import { Component, Input, Output, ElementRef, EventEmitter, ViewEncapsulation, } from '@angular/core';
import { debounceTime } from 'rxjs/operators';
import { dia } from 'jointjs';
import { Flo } from '../shared/flo-common';
import { Shapes, Constants } from '../shared/shapes';
import { Utils } from './editor-utils';
import { CompositeDisposable, Disposable } from 'ts-disposables';
import * as _$ from 'jquery';
import * as _ from 'lodash';
import { Subject, BehaviorSubject } from 'rxjs';
var joint = Flo.joint;
var $ = _$;
var SCROLLBAR_SIZE = 17;
var EditorComponent = /** @class */ (function () {
    function EditorComponent(element) {
        var _this = this;
        this.element = element;
        /**
         * Flag specifying whether the Flo-Editor is in read-only mode.
         */
        this._readOnlyCanvas = false;
        /**
         * Grid size
         */
        this._gridSize = 1;
        this._hiddenPalette = false;
        this.paletteSizeValue = 170;
        this.textToGraphEventEmitter = new EventEmitter();
        this.graphToTextEventEmitter = new EventEmitter();
        this._graphToTextSyncEnabled = true;
        this.validationEventEmitter = new EventEmitter();
        this._disposables = new CompositeDisposable();
        this._dslText = '';
        this.textToGraphConversionCompleted = new Subject();
        this.graphToTextConversionCompleted = new Subject();
        this.paletteReady = new BehaviorSubject(false);
        this.paletteSizeChange = new EventEmitter();
        this.searchFilterPlaceHolder = 'Search...';
        /**
         * Min zoom percent value
         */
        this.minZoom = 5;
        /**
         * Max zoom percent value
         */
        this.maxZoom = 400;
        /**
         * Zoom percent increment/decrement step
         */
        this.zoomStep = 5;
        this.paperPadding = 0;
        this.floApi = new EventEmitter();
        this.validationMarkers = new EventEmitter();
        this.contentValidated = new EventEmitter();
        this.dslChange = new EventEmitter();
        this.onProperties = new EventEmitter();
        this._resizeHandler = function () { return _this.autosizePaper(); };
        var self = this;
        this.editorContext = new (/** @class */ (function () {
            function DefaultRunnableContext() {
            }
            Object.defineProperty(DefaultRunnableContext.prototype, "zoomPercent", {
                get: function () {
                    return self.zoomPercent;
                },
                set: function (percent) {
                    self.zoomPercent = percent;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(DefaultRunnableContext.prototype, "noPalette", {
                get: function () {
                    return self.noPalette;
                },
                set: function (noPalette) {
                    self.noPalette = noPalette;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(DefaultRunnableContext.prototype, "gridSize", {
                get: function () {
                    return self.gridSize;
                },
                set: function (gridSize) {
                    self.gridSize = gridSize;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(DefaultRunnableContext.prototype, "readOnlyCanvas", {
                get: function () {
                    return self.readOnlyCanvas;
                },
                set: function (readOnly) {
                    self.readOnlyCanvas = readOnly;
                },
                enumerable: true,
                configurable: true
            });
            DefaultRunnableContext.prototype.setDsl = function (dsl) {
                self.dsl = dsl;
            };
            DefaultRunnableContext.prototype.updateGraph = function () {
                return self.updateGraphRepresentation();
            };
            DefaultRunnableContext.prototype.updateText = function () {
                return self.updateTextRepresentation();
            };
            DefaultRunnableContext.prototype.performLayout = function () {
                return self.doLayout();
            };
            DefaultRunnableContext.prototype.clearGraph = function () {
                var _this = this;
                self.selection = undefined;
                self.graph.clear();
                if (self.metamodel && self.metamodel.load && self.editor && self.editor.setDefaultContent) {
                    return self.metamodel.load().then(function (data) {
                        self.editor.setDefaultContent(_this, data);
                        if (!self.graphToTextSync) {
                            return self.updateTextRepresentation();
                        }
                    });
                }
                else {
                    if (!self.graphToTextSync) {
                        return self.updateTextRepresentation();
                    }
                }
            };
            DefaultRunnableContext.prototype.getGraph = function () {
                return self.graph;
            };
            DefaultRunnableContext.prototype.getPaper = function () {
                return self.paper;
            };
            Object.defineProperty(DefaultRunnableContext.prototype, "graphToTextSync", {
                get: function () {
                    return self.graphToTextSync;
                },
                set: function (sync) {
                    self.graphToTextSync = sync;
                },
                enumerable: true,
                configurable: true
            });
            DefaultRunnableContext.prototype.getMinZoom = function () {
                return self.minZoom;
            };
            DefaultRunnableContext.prototype.getMaxZoom = function () {
                return self.maxZoom;
            };
            DefaultRunnableContext.prototype.getZoomStep = function () {
                return self.zoomStep;
            };
            DefaultRunnableContext.prototype.fitToPage = function () {
                self.fitToPage();
            };
            DefaultRunnableContext.prototype.createNode = function (metadata, props, position) {
                return self.createNode(metadata, props, position);
            };
            DefaultRunnableContext.prototype.createLink = function (source, target, metadata, props) {
                return self.createLink(source, target, metadata, props);
            };
            Object.defineProperty(DefaultRunnableContext.prototype, "selection", {
                get: function () {
                    return self.selection;
                },
                set: function (newSelection) {
                    self.selection = newSelection;
                },
                enumerable: true,
                configurable: true
            });
            DefaultRunnableContext.prototype.deleteSelectedNode = function () {
                self.deleteSelected();
            };
            DefaultRunnableContext.prototype.delete = function (cell) {
                self.delete(cell);
            };
            Object.defineProperty(DefaultRunnableContext.prototype, "textToGraphConversionObservable", {
                get: function () {
                    return self.textToGraphConversionCompleted;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(DefaultRunnableContext.prototype, "graphToTextConversionObservable", {
                get: function () {
                    return self.graphToTextConversionCompleted;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(DefaultRunnableContext.prototype, "paletteReady", {
                get: function () {
                    return self.paletteReady;
                },
                enumerable: true,
                configurable: true
            });
            return DefaultRunnableContext;
        }()))();
    }
    Object.defineProperty(EditorComponent.prototype, "paletteSize", {
        /**
         * Size (Width) of the palette
         */
        get: function () {
            return this.paletteSizeValue;
        },
        set: function (newSize) {
            this.paletteSizeValue = newSize;
            this.paletteSizeChange.emit(newSize);
        },
        enumerable: true,
        configurable: true
    });
    EditorComponent.prototype.onPropertiesHandle = function () {
        if (this.editorContext.selection) {
            this.onProperties.emit(this.editorContext.selection.model);
        }
    };
    EditorComponent.prototype.ngOnInit = function () {
        var _this = this;
        this.initGraph();
        this.initPaper();
        this.initGraphListeners();
        this.initPaperListeners();
        this.initMetamodel();
        $(window).on('resize', this._resizeHandler);
        this._disposables.add(Disposable.create(function () { return $(window).off('resize', _this._resizeHandler); }));
        /*
         * Execute resize to get the right size for the SVG element on the editor canvas.
         * Executed via timeout to let angular render the DOM first and elements to have the right width and height
         */
        window.setTimeout(this._resizeHandler);
        this.floApi.emit(this.editorContext);
    };
    EditorComponent.prototype.ngOnDestroy = function () {
        this._disposables.dispose();
    };
    EditorComponent.prototype.deleteSelected = function () {
        if (this.selection) {
            this.delete(this.selection.model);
        }
    };
    EditorComponent.prototype.delete = function (cell) {
        this.graph.trigger('startDeletion', cell);
    };
    Object.defineProperty(EditorComponent.prototype, "noPalette", {
        get: function () {
            return this._hiddenPalette;
        },
        set: function (hidden) {
            this._hiddenPalette = hidden;
            // If palette is not shown ensure that canvas starts from the left==0!
            if (hidden) {
                $('#paper-container', this.element.nativeElement).css('left', 0);
            }
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(EditorComponent.prototype, "graphToTextSync", {
        get: function () {
            return this._graphToTextSyncEnabled;
        },
        set: function (sync) {
            this._graphToTextSyncEnabled = sync;
            // Try commenting the sync out. Just set the flag but don't kick off graph->text conversion
            // this.performGraphToTextSyncing();
        },
        enumerable: true,
        configurable: true
    });
    EditorComponent.prototype.performGraphToTextSyncing = function () {
        if (this._graphToTextSyncEnabled) {
            this.graphToTextEventEmitter.emit();
        }
    };
    EditorComponent.prototype.createHandle = function (element, kind, action, location) {
        if (!location) {
            var bbox = element.model.getBBox();
            location = bbox.origin().offset(bbox.width / 2, bbox.height / 2);
        }
        var handle = Shapes.Factory.createHandle({
            renderer: this.renderer,
            paper: this.paper,
            parent: element.model,
            kind: kind,
            position: location
        });
        var view = this.paper.findViewByModel(handle);
        view.on('cell:pointerdown', function () {
            if (action) {
                action();
            }
        });
        view.on('cell:mouseover', function () {
            handle.attr('image/filter', {
                name: 'dropShadow',
                args: { dx: 1, dy: 1, blur: 1, color: 'black' }
            });
        });
        view.on('cell:mouseout', function () {
            handle.removeAttr('image/filter');
        });
        view.setInteractivity(false);
        return handle;
    };
    EditorComponent.prototype.removeEmbeddedChildrenOfType = function (element, types) {
        var embeds = element.getEmbeddedCells();
        for (var i = 0; i < embeds.length; i++) {
            if (types.indexOf(embeds[i].get('type')) >= 0) {
                embeds[i].remove();
            }
        }
    };
    Object.defineProperty(EditorComponent.prototype, "selection", {
        get: function () {
            return this._selection;
        },
        set: function (newSelection) {
            var _this = this;
            if (newSelection && (newSelection.model.get('type') === joint.shapes.flo.DECORATION_TYPE || newSelection.model.get('type') === joint.shapes.flo.HANDLE_TYPE)) {
                newSelection = this.paper.findViewByModel(this.graph.getCell(newSelection.model.get('parent')));
            }
            if (newSelection && (!newSelection.model.attr('metadata') || newSelection.model.attr('metadata/metadata/unselectable'))) {
                newSelection = undefined;
            }
            if (newSelection !== this._selection) {
                if (this._selection) {
                    var elementview = this.paper.findViewByModel(this._selection.model);
                    if (elementview) { // May have been removed from the graph
                        this.removeEmbeddedChildrenOfType(elementview.model, joint.shapes.flo.HANDLE_TYPE);
                        elementview.unhighlight();
                    }
                }
                if (newSelection) {
                    newSelection.highlight();
                    if (this.editor && this.editor.createHandles) {
                        this.editor.createHandles(this.editorContext, function (owner, kind, action, location) { return _this.createHandle(owner, kind, action, location); }, newSelection);
                    }
                }
                this._selection = newSelection;
            }
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(EditorComponent.prototype, "readOnlyCanvas", {
        get: function () {
            return this._readOnlyCanvas;
        },
        set: function (value) {
            var _this = this;
            if (this._readOnlyCanvas === value) {
                // Nothing to do
                return;
            }
            if (value) {
                this.selection = undefined;
            }
            if (this.graph) {
                this.graph.getLinks().forEach(function (link) {
                    if (value) {
                        link.attr('.link-tools/display', 'none');
                        link.attr('.marker-vertices/display', 'none');
                        link.attr('.connection-wrap/display', 'none');
                    }
                    else {
                        link.removeAttr('.link-tools/display');
                        if (_this.editor && _this.editor.allowLinkVertexEdit) {
                            link.removeAttr('.marker-vertices/display');
                        }
                        link.removeAttr('.connection-wrap/display');
                    }
                });
            }
            this._readOnlyCanvas = value;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Displays graphical feedback for the drag and drop in progress based on current drag and drop descriptor object
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    EditorComponent.prototype.showDragFeedback = function (dragDescriptor) {
        if (this.editor && this.editor.showDragFeedback) {
            this.editor.showDragFeedback(this.editorContext, dragDescriptor);
        }
        else {
            var magnet = void 0;
            if (dragDescriptor.source && dragDescriptor.source.view) {
                joint.V(dragDescriptor.source.view.el).addClass('dnd-source-feedback');
                if (dragDescriptor.source.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.source.view, dragDescriptor.source.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).addClass('dnd-source-feedback');
                    }
                }
            }
            if (dragDescriptor.target && dragDescriptor.target.view) {
                joint.V(dragDescriptor.target.view.el).addClass('dnd-target-feedback');
                if (dragDescriptor.target.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.target.view, dragDescriptor.target.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).addClass('dnd-target-feedback');
                    }
                }
            }
        }
    };
    /**
     * Hides graphical feedback for the drag and drop in progress based on current drag and drop descriptor object
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    EditorComponent.prototype.hideDragFeedback = function (dragDescriptor) {
        if (this.editor && this.editor.hideDragFeedback) {
            this.editor.hideDragFeedback(this.editorContext, dragDescriptor);
        }
        else {
            var magnet = void 0;
            if (dragDescriptor.source && dragDescriptor.source.view) {
                joint.V(dragDescriptor.source.view.el).removeClass('dnd-source-feedback');
                if (dragDescriptor.source.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.source.view, dragDescriptor.source.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).removeClass('dnd-source-feedback');
                    }
                }
            }
            if (dragDescriptor.target && dragDescriptor.target.view) {
                joint.V(dragDescriptor.target.view.el).removeClass('dnd-target-feedback');
                if (dragDescriptor.target.cssClassSelector) {
                    magnet = Flo.findMagnetByClass(dragDescriptor.target.view, dragDescriptor.target.cssClassSelector);
                    if (magnet) {
                        joint.V(magnet).removeClass('dnd-target-feedback');
                    }
                }
            }
        }
    };
    /**
     * Sets the new DnD info object - the descriptor for DnD
     *
     * @param dragDescriptor DnD info object. Has on info on graph node being dragged (drag source) and what it is
     * being dragged over at the moment (drop target)
     */
    EditorComponent.prototype.setDragDescriptor = function (dragDescriptor) {
        if (this.highlighted === dragDescriptor) {
            return;
        }
        if (this.highlighted && dragDescriptor && _.isEqual(this.highlighted.sourceComponent, dragDescriptor.sourceComponent)) {
            if (this.highlighted.source === dragDescriptor.source && this.highlighted.target === dragDescriptor.target) {
                return;
            }
            if (this.highlighted.source &&
                dragDescriptor.source &&
                this.highlighted.target &&
                dragDescriptor.target &&
                this.highlighted.source.view.model === dragDescriptor.source.view.model &&
                this.highlighted.source.cssClassSelector === dragDescriptor.source.cssClassSelector &&
                this.highlighted.target.view.model === dragDescriptor.target.view.model &&
                this.highlighted.target.cssClassSelector === dragDescriptor.target.cssClassSelector) {
                return;
            }
        }
        if (this.highlighted) {
            this.hideDragFeedback(this.highlighted);
        }
        this.highlighted = dragDescriptor;
        if (this.highlighted) {
            this.showDragFeedback(this.highlighted);
        }
    };
    /**
     * Handles DnD events when a node is being dragged over canvas
     *
     * @param draggedView The Joint JS view object being dragged
     * @param targetUnderMouse The Joint JS view under mouse cursor
     * @param x X coordinate of the mouse on the canvas
     * @param y Y coordinate of the mosue on the canvas
     * @param context DnD context (palette or canvas)
     */
    EditorComponent.prototype.handleNodeDragging = function (draggedView, targetUnderMouse, x, y, sourceComponent) {
        if (this.editor && this.editor.calculateDragDescriptor) {
            this.setDragDescriptor(this.editor.calculateDragDescriptor(this.editorContext, draggedView, targetUnderMouse, joint.g.point(x, y), sourceComponent));
        }
    };
    /**
     * Handles DnD drop event when a node is being dragged and dropped on the main canvas
     */
    EditorComponent.prototype.handleNodeDropping = function () {
        if (this.highlighted && this.editor && this.editor.handleNodeDropping) {
            this.editor.handleNodeDropping(this.editorContext, this.highlighted);
        }
        this.setDragDescriptor(undefined);
    };
    /**
     * Hides DOM Node (used to determine drop target DOM element)
     * @param domNode DOM node to hide
     * @returns
     */
    EditorComponent.prototype._hideNode = function (domNode) {
        var oldVisibility = {
            visibility: domNode.style ? domNode.style.display : undefined,
            children: []
        };
        for (var i = 0; i < domNode.children.length; i++) {
            var node = domNode.children.item(i);
            if (node instanceof HTMLElement) {
                oldVisibility.children.push(this._hideNode(node));
            }
        }
        domNode.style.display = 'none';
        return oldVisibility;
    };
    /**
     * Restored DOM node original visibility (used to determine drop target DOM element)
     * @param domNode DOM node to restore visibility of
     * @param oldVisibility original visibility parameter
     */
    EditorComponent.prototype._restoreNodeVisibility = function (domNode, oldVisibility) {
        if (domNode.style) {
            domNode.style.display = oldVisibility.visibility;
        }
        var j = 0;
        for (var i = 0; i < domNode.childNodes.length; i++) {
            if (j < oldVisibility.children.length) {
                var node = domNode.children.item(i);
                if (node instanceof HTMLElement) {
                    this._restoreNodeVisibility(node, oldVisibility.children[j++]);
                }
            }
        }
    };
    /**
     * Unfortunately we can't just use event.target because often draggable shape on the canvas overlaps the target.
     * We can easily find the element(s) at location, but only nodes :-( Unclear how to find links at location
     * (bounding box of a link for testing is bad).
     * The result of that is that links can only be the drop target when dragging from the palette currently.
     * When DnDing shapes on the canvas drop target cannot be a link.
     *
     * Excluded views enables you to choose to filter some possible answers (useful in the case where elements are stacked
     * - e.g. Drag-n-Drop)
     */
    EditorComponent.prototype.getTargetViewFromEvent = function (event, x, y, excludeViews) {
        var _this = this;
        if (excludeViews === void 0) { excludeViews = []; }
        if (!x && !y) {
            var l = this.paper.snapToGrid({ x: event.clientX, y: event.clientY });
            x = l.x;
            y = l.y;
        }
        // TODO: See if next code paragraph is needed. Most likely it's just code executed for nothing
        // let elements = this.graph.findModelsFromPoint(joint.g.point(x, y));
        // let underMouse = elements.find(e => !_.isUndefined(excludeViews.find(x => x === this.paper.findViewByModel(e))));
        // if (underMouse) {
        //   return underMouse;
        // }
        var oldVisibility = excludeViews.map(function (_x) { return _this._hideNode(_x.el); });
        var targetElement = document.elementFromPoint(event.clientX, event.clientY);
        excludeViews.forEach(function (excluded, i) {
            _this._restoreNodeVisibility(excluded.el, oldVisibility[i]);
        });
        return this.paper.findView($(targetElement));
    };
    EditorComponent.prototype.handleDnDFromPalette = function (dndEvent) {
        switch (dndEvent.type) {
            case Flo.DnDEventType.DRAG:
                this.handleDragFromPalette(dndEvent);
                break;
            case Flo.DnDEventType.DROP:
                this.handleDropFromPalette(dndEvent);
                break;
            default:
                break;
        }
    };
    EditorComponent.prototype.handleDragFromPalette = function (dnDEvent) {
        console.debug('Dragging from palette');
        if (dnDEvent.view && !this.readOnlyCanvas) {
            var location_1 = this.paper.snapToGrid({ x: dnDEvent.event.clientX, y: dnDEvent.event.clientY });
            this.handleNodeDragging(dnDEvent.view, this.getTargetViewFromEvent(dnDEvent.event, location_1.x, location_1.y, [dnDEvent.view]), location_1.x, location_1.y, Constants.PALETTE_CONTEXT);
        }
    };
    EditorComponent.prototype.createNode = function (metadata, props, position) {
        return Shapes.Factory.createNode({
            renderer: this.renderer,
            paper: this.paper,
            metadata: metadata,
            props: props,
            position: position
        });
    };
    EditorComponent.prototype.createLink = function (source, target, metadata, props) {
        return Shapes.Factory.createLink({
            renderer: this.renderer,
            paper: this.paper,
            source: source,
            target: target,
            metadata: metadata,
            props: props
        });
    };
    EditorComponent.prototype.handleDropFromPalette = function (event) {
        var cellview = event.view;
        var evt = event.event;
        if (this.paper.el === evt.target || $.contains(this.paper.el, evt.target)) {
            if (this.readOnlyCanvas) {
                this.setDragDescriptor(undefined);
            }
            else {
                var metadata = cellview.model.attr('metadata');
                var props = cellview.model.attr('props');
                var position = this.paper.snapToGrid({ x: evt.clientX, y: evt.clientY });
                /* Calculate target element before creating the new
                 * element under mouse location. Otherwise target
                 * element would be the newly created element because
                 * it's under the mouse pointer
                 */
                var targetElement = this.getTargetViewFromEvent(evt, position.x, position.y, [event.view]);
                var newNode = this.createNode(metadata, props, position);
                var newView = this.paper.findViewByModel(newNode);
                this.handleNodeDragging(newView, targetElement, position.x, position.y, Constants.PALETTE_CONTEXT);
                this.handleNodeDropping();
            }
        }
    };
    EditorComponent.prototype.fitToContent = function (gridWidth, gridHeight, padding, opt) {
        var paper = this.paper;
        if (joint.util.isObject(gridWidth)) {
            // first parameter is an option object
            opt = gridWidth;
            gridWidth = opt.gridWidth || 1;
            gridHeight = opt.gridHeight || 1;
            padding = opt.padding || 0;
        }
        else {
            opt = opt || {};
            gridWidth = gridWidth || 1;
            gridHeight = gridHeight || 1;
            padding = padding || 0;
        }
        var paddingJson = joint.util.normalizeSides(padding);
        // Calculate the paper size to accomodate all the graph's elements.
        var bbox = joint.V(paper.viewport).getBBox();
        var currentScale = paper.scale();
        var currentTranslate = paper.translate();
        bbox.x *= currentScale.sx;
        bbox.y *= currentScale.sy;
        bbox.width *= currentScale.sx;
        bbox.height *= currentScale.sy;
        var calcWidth = Math.max((bbox.width + bbox.x) / gridWidth, 1) * gridWidth;
        var calcHeight = Math.max((bbox.height + bbox.y) / gridHeight, 1) * gridHeight;
        var tx = 0;
        var ty = 0;
        if ((opt.allowNewOrigin === 'negative' && bbox.x < 0) || (opt.allowNewOrigin === 'positive' && bbox.x >= 0) || opt.allowNewOrigin === 'any') {
            tx = (-bbox.x / gridWidth) * gridWidth;
            tx += paddingJson.left;
        }
        else if (opt.allowNewOrigin === 'same') {
            tx = currentTranslate.tx;
        }
        calcWidth += tx;
        if ((opt.allowNewOrigin === 'negative' && bbox.y < 0) || (opt.allowNewOrigin === 'positive' && bbox.y >= 0) || opt.allowNewOrigin === 'any') {
            ty = (-bbox.y / gridHeight) * gridHeight;
            ty += paddingJson.top;
        }
        else if (opt.allowNewOrigin === 'same') {
            ty = currentTranslate.ty;
        }
        calcHeight += ty;
        calcWidth += paddingJson.right;
        calcHeight += paddingJson.bottom;
        // Make sure the resulting width and height are greater than minimum.
        calcWidth = Math.max(calcWidth, opt.minWidth || 0);
        calcHeight = Math.max(calcHeight, opt.minHeight || 0);
        // Make sure the resulting width and height are lesser than maximum.
        calcWidth = Math.min(calcWidth, opt.maxWidth || Number.MAX_VALUE);
        calcHeight = Math.min(calcHeight, opt.maxHeight || Number.MAX_VALUE);
        var dimensionChange = calcWidth !== paper.options.width || calcHeight !== paper.options.height;
        var originChange = tx !== currentTranslate.tx || ty !== currentTranslate.ty;
        // Change the dimensions only if there is a size discrepency or an origin change
        if (originChange) {
            paper.translate(tx, ty);
        }
        if (dimensionChange) {
            paper.setDimensions(calcWidth, calcHeight);
        }
    };
    EditorComponent.prototype.autosizePaper = function () {
        var parent = $('#paper-container', this.element.nativeElement);
        var parentWidth = parent.innerWidth();
        var parentHeight = parent.innerHeight();
        this.fitToContent(this.gridSize, this.gridSize, this.paperPadding, {
            minWidth: parentWidth - Flo.SCROLLBAR_WIDTH,
            minHeight: parentHeight - Flo.SCROLLBAR_WIDTH,
            allowNewOrigin: 'same'
        });
    };
    EditorComponent.prototype.fitToPage = function () {
        var parent = $('#paper-container', this.element.nativeElement);
        var minScale = this.minZoom / 100;
        var maxScale = 2;
        var parentWidth = parent.innerWidth();
        var parentHeight = parent.innerHeight();
        this.paper.scaleContentToFit({
            padding: this.paperPadding,
            minScaleX: minScale,
            minScaleY: minScale,
            maxScaleX: maxScale,
            maxScaleY: maxScale,
            fittingBBox: { x: 0, y: 0, width: parentWidth - Flo.SCROLLBAR_WIDTH, height: parentHeight - Flo.SCROLLBAR_WIDTH }
        });
        /**
         * Size the canvas appropriately and allow origin movement
         */
        this.fitToContent(this.gridSize, this.gridSize, this.paperPadding, {
            minWidth: parentWidth,
            minHeight: parentHeight,
            maxWidth: parentWidth,
            maxHeight: parentHeight,
            allowNewOrigin: 'any'
        });
    };
    Object.defineProperty(EditorComponent.prototype, "zoomPercent", {
        get: function () {
            return Math.round(this.paper.scale().sx * 100);
        },
        set: function (percent) {
            if (!isNaN(percent)) {
                if (percent < this.minZoom) {
                    percent = this.minZoom;
                }
                else if (percent >= this.maxZoom) {
                    percent = this.maxZoom;
                }
                else {
                    if (percent <= 0) {
                        percent = 0.00001;
                    }
                }
                this.paper.scale(percent / 100, percent / 100);
            }
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(EditorComponent.prototype, "gridSize", {
        get: function () {
            return this._gridSize;
        },
        set: function (size) {
            if (!isNaN(size) && size >= 1) {
                this._gridSize = size;
                if (this.paper) {
                    this.paper.setGridSize(size);
                }
            }
        },
        enumerable: true,
        configurable: true
    });
    EditorComponent.prototype.validateContent = function () {
        var _this = this;
        return new Promise(function (resolve) {
            if (_this.editor && _this.editor.validate) {
                return _this.editor
                    .validate(_this.graph, _this.dsl, _this.editorContext)
                    .then(function (allMarkers) {
                    _this.graph.getCells()
                        .forEach(function (cell) { return _this.markElement(cell, allMarkers.has(cell.id) ? allMarkers.get(cell.id) : []); });
                    _this.validationMarkers.emit(allMarkers);
                    _this.contentValidated.emit(true);
                    resolve();
                });
            }
            else {
                resolve();
            }
        });
    };
    EditorComponent.prototype.markElement = function (cell, markers) {
        cell.set('markers', markers);
        // Old legacy code below consider removing
        var errorMessages = markers.map(function (m) { return m.message; });
        var errorCell = cell.getEmbeddedCells().find(function (e) { return e.attr('./kind') === Constants.ERROR_DECORATION_KIND; });
        if (errorCell) {
            if (errorMessages.length === 0) {
                errorCell.remove();
            }
            else {
                // Without rewrite we merge this list with existing errors
                errorCell.attr('messages', errorMessages, { rewrite: true });
            }
        }
        else if (errorMessages.length > 0) {
            var error = Shapes.Factory.createDecoration({
                renderer: this.renderer,
                paper: this.paper,
                parent: cell,
                kind: Constants.ERROR_DECORATION_KIND,
                messages: errorMessages
            });
            if (error) {
                var view = this.paper.findViewByModel(error);
                view.setInteractivity(false);
            }
        }
    };
    EditorComponent.prototype.doLayout = function () {
        if (this.renderer && this.renderer.layout) {
            return this.renderer.layout(this.paper);
        }
    };
    Object.defineProperty(EditorComponent.prototype, "dsl", {
        get: function () {
            return this._dslText;
        },
        set: function (dslText) {
            if (this._dslText !== dslText) {
                this._dslText = dslText;
                this.textToGraphEventEmitter.emit();
            }
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Ask the server to parse the supplied text into a JSON graph of nodes and links,
     * then update the view based on that new information.
     */
    EditorComponent.prototype.updateGraphRepresentation = function () {
        var _this = this;
        console.debug("Updating graph to represent '" + this._dslText + "'");
        if (this.metamodel && this.metamodel.textToGraph) {
            return this.metamodel.textToGraph(this.editorContext, this._dslText).then(function () {
                _this.textToGraphConversionCompleted.next();
                return _this.validateContent();
            });
        }
        else {
            this.textToGraphConversionCompleted.next();
            return this.validateContent();
        }
    };
    EditorComponent.prototype.updateTextRepresentation = function () {
        var _this = this;
        if (this.metamodel && this.metamodel.graphToText) {
            return this.metamodel.graphToText(this.editorContext).then(function (text) {
                if (_this._dslText !== text) {
                    _this._dslText = text;
                    _this.dslChange.emit(text);
                }
                _this.graphToTextConversionCompleted.next();
                return _this.validateContent();
            })
                .catch(function (error) {
                // Validation may reveal why the graph couldn't be
                // converted so let it run
                _this.graphToTextConversionCompleted.next();
                return _this.validateContent();
            });
        }
        else {
            this.graphToTextConversionCompleted.next();
            return this.validateContent();
        }
    };
    EditorComponent.prototype.initMetamodel = function () {
        var _this = this;
        this.metamodel.load().then(function (data) {
            _this.updateGraphRepresentation();
            var textSyncSubscription = _this.graphToTextEventEmitter.pipe(debounceTime(100)).subscribe(function () {
                if (_this._graphToTextSyncEnabled) {
                    _this.updateTextRepresentation();
                }
            });
            _this._disposables.add(Disposable.create(function () { return textSyncSubscription.unsubscribe(); }));
            // Setup content validated event emitter. Emit not validated when graph to text conversion required
            var graphValidatedSubscription1 = _this.graphToTextEventEmitter.subscribe(function () { return _this.contentValidated.emit(false); });
            _this._disposables.add(Disposable.create(function () { return graphValidatedSubscription1.unsubscribe; }));
            // let validationSubscription = this.validationEventEmitter.pipe(debounceTime(100)).subscribe(() => this.validateGraph());
            // this._disposables.add(Disposable.create(() => validationSubscription.unsubscribe()));
            var graphSyncSubscription = _this.textToGraphEventEmitter.pipe(debounceTime(300)).subscribe(function () { return _this.updateGraphRepresentation(); });
            _this._disposables.add(Disposable.create(function () { return graphSyncSubscription.unsubscribe(); }));
            // Setup content validated event emitter. Emit not validated when text to graph conversion required
            var graphValidatedSubscription2 = _this.textToGraphEventEmitter.subscribe(function () { return _this.contentValidated.emit(false); });
            _this._disposables.add(Disposable.create(function () { return graphValidatedSubscription2.unsubscribe; }));
            if (_this.editor && _this.editor.setDefaultContent) {
                _this.editor.setDefaultContent(_this.editorContext, data);
            }
        });
    };
    EditorComponent.prototype.initGraph = function () {
        this.graph = new joint.dia.Graph();
        this.graph.set('type', Constants.CANVAS_CONTEXT);
        this.graph.set('paperPadding', this.paperPadding);
    };
    EditorComponent.prototype.handleNodeCreation = function (node) {
        var _this = this;
        node.on('change:size', this._resizeHandler);
        node.on('change:position', this._resizeHandler);
        if (node.attr('metadata')) {
            node.on('change:attrs', function (cell, attrs, changeData) {
                var propertyPath = changeData ? changeData.propertyPath : undefined;
                if (propertyPath) {
                    var propAttr = propertyPath.substr(propertyPath.indexOf('/') + 1);
                    if (propAttr.indexOf('metadata') === 0 ||
                        propAttr.indexOf('props') === 0 ||
                        (_this.renderer && _this.renderer.isSemanticProperty && _this.renderer.isSemanticProperty(propAttr, node))) {
                        _this.performGraphToTextSyncing();
                    }
                    if (_this.renderer && _this.renderer.refreshVisuals) {
                        _this.renderer.refreshVisuals(node, propAttr, _this.paper);
                    }
                }
            });
        }
        node.on('change:markers', function () {
            if (_this.renderer && _this.renderer.markersChanged) {
                _this.renderer.markersChanged(node, _this.paper);
            }
        });
    };
    /**
     * Forwards a link event occurrence to any handlers in the editor service, if they are defined. Event examples
     * are 'change:source', 'change:target'.
     */
    EditorComponent.prototype.handleLinkEvent = function (event, link) {
        if (this.renderer && this.renderer.handleLinkEvent) {
            this.renderer.handleLinkEvent(this.editorContext, event, link);
        }
    };
    EditorComponent.prototype.handleLinkCreation = function (link) {
        var _this = this;
        link.on('change:source', function (l) {
            _this.autosizePaper();
            var newSourceId = l.get('source').id;
            var oldSourceId = l.previous('source').id;
            if (newSourceId !== oldSourceId) {
                _this.performGraphToTextSyncing();
            }
            _this.handleLinkEvent('change:source', l);
        });
        link.on('change:target', function (l) {
            _this.autosizePaper();
            var newTargetId = l.get('target').id;
            var oldTargetId = l.previous('target').id;
            if (newTargetId !== oldTargetId) {
                _this.performGraphToTextSyncing();
            }
            _this.handleLinkEvent('change:target', l);
        });
        link.on('change:vertices', this._resizeHandler);
        link.on('change:attrs', function (cell, attrs, changeData) {
            var propertyPath = changeData ? changeData.propertyPath : undefined;
            if (propertyPath) {
                var propAttr = propertyPath.substr(propertyPath.indexOf('/') + 1);
                if (propAttr.indexOf('metadata') === 0 ||
                    propAttr.indexOf('props') === 0 ||
                    (_this.renderer && _this.renderer.isSemanticProperty && _this.renderer.isSemanticProperty(propAttr, link))) {
                    var sourceId = link.get('source').id;
                    var targetId = link.get('target').id;
                    _this.performGraphToTextSyncing();
                }
                if (_this.renderer && _this.renderer.refreshVisuals) {
                    _this.renderer.refreshVisuals(link, propAttr, _this.paper);
                }
            }
        });
        this.paper.findViewByModel(link).on('link:options', function () { return _this.handleLinkEvent('options', link); });
        if (this.readOnlyCanvas) {
            link.attr('.link-tools/display', 'none');
        }
        this.handleLinkEvent('add', link);
    };
    EditorComponent.prototype.initGraphListeners = function () {
        var _this = this;
        this.graph.on('add', function (element) {
            if (element instanceof joint.dia.Link) {
                _this.handleLinkCreation(element);
            }
            else if (element instanceof joint.dia.Element) {
                _this.handleNodeCreation(element);
            }
            if (element.get('type') === joint.shapes.flo.NODE_TYPE || element.get('type') === joint.shapes.flo.LINK_TYPE) {
                _this.performGraphToTextSyncing();
            }
            _this.autosizePaper();
        });
        this.graph.on('remove', function (element) {
            if (element instanceof joint.dia.Link) {
                _this.handleLinkEvent('remove', element);
            }
            if (_this.selection && _this.selection.model === element) {
                _this.selection = undefined;
            }
            if (element.isLink()) {
                window.setTimeout(function () { return _this.performGraphToTextSyncing(); }, 100);
            }
            else if (element.get('type') === joint.shapes.flo.NODE_TYPE) {
                _this.performGraphToTextSyncing();
            }
            _this.autosizePaper();
        });
        // Set if link is fan-routed. Should be called before routing call
        this.graph.on('change:vertices', function (link, changed, opt) {
            if (opt.fanRouted) {
                link.set('fanRouted', true);
            }
            else {
                link.unset('fanRouted');
            }
        });
        // adjust vertices when a cell is removed or its source/target was changed
        this.graph.on('add remove change:source change:target change:vertices change:position', _.partial(Utils.fanRoute, this.graph));
        this.graph.on('startDeletion', function (cell) {
            if (_this.editor && _this.editor.preDelete) {
                if (_this.editor.preDelete(_this.editorContext, _this.selection.model)) {
                    cell.remove();
                }
            }
            else {
                cell.remove();
            }
        });
    };
    EditorComponent.prototype.initPaperListeners = function () {
        var _this = this;
        // https://stackoverflow.com/questions/20463533/how-to-add-an-onclick-event-to-a-joint-js-element
        this.paper.on('cell:pointerup', function (cellView) {
            if (!_this.readOnlyCanvas) {
                _this.selection = cellView;
            }
        });
        this.paper.on('blank:pointerdown', function () {
            _this.selection = undefined;
        });
        this.paper.on('scale', this._resizeHandler);
        this.paper.on('all', function () {
            if (Utils.isCustomPaperEvent(arguments)) {
                arguments[2].trigger.apply(arguments[2], [arguments[0], arguments[1], arguments[3], arguments[4]]);
            }
        });
        this.paper.on('dragging-node-over-canvas', function (dndEvent) {
            console.debug("Canvas DnD type = " + dndEvent.type);
            var location = _this.paper.snapToGrid({ x: dndEvent.event.clientX, y: dndEvent.event.clientY });
            switch (dndEvent.type) {
                case Flo.DnDEventType.DRAG:
                    _this.handleNodeDragging(dndEvent.view, _this.getTargetViewFromEvent(dndEvent.event, location.x, location.y, [dndEvent.view]), location.x, location.y, Constants.CANVAS_CONTEXT);
                    break;
                case Flo.DnDEventType.DROP:
                    _this.handleNodeDropping();
                    break;
                default:
                    break;
            }
        });
        // JointJS now no longer grabs focus if working in a paper element - crude...
        // $('#flow-view', this.element.nativeElement).on('mousedown', () => {
        // $('#palette-filter-textfield', this.element.nativeElement).focus();
        // });
    };
    EditorComponent.prototype.initPaper = function () {
        var _this = this;
        var options = {
            el: $('#paper', this.element.nativeElement),
            gridSize: this._gridSize,
            drawGrid: true,
            model: this.graph,
            elementView: this.renderer && this.renderer.getNodeView ? this.renderer.getNodeView() : joint.shapes.flo.ElementView /*joint.dia.ElementView*/,
            linkView: this.renderer && this.renderer.getLinkView ? this.renderer.getLinkView() : joint.shapes.flo.LinkView,
            // Enable link snapping within 25px lookup radius
            snapLinks: { radius: 25 },
            defaultLink: /*this.renderer && this.renderer.createDefaultLink ? this.renderer.createDefaultLink: new joint.shapes.flo.Link*/ function (cellView, magnet) {
                if (_this.renderer && _this.renderer.createLink) {
                    var linkEnd = {
                        id: cellView.model.id
                    };
                    if (magnet) {
                        linkEnd.selector = cellView.getSelector(magnet, undefined);
                    }
                    if (magnet.getAttribute('port')) {
                        linkEnd.port = magnet.getAttribute('port');
                    }
                    if (magnet.getAttribute('port') === 'input') {
                        return _this.renderer.createLink(undefined, linkEnd);
                    }
                    else {
                        return _this.renderer.createLink(linkEnd, undefined);
                    }
                }
                else {
                    return new joint.shapes.flo.Link();
                }
            },
            // decide whether to create a link if the user clicks a magnet
            validateMagnet: function (cellView, magnet) {
                if (_this.readOnlyCanvas) {
                    return false;
                }
                else {
                    if (_this.editor && _this.editor.validatePort) {
                        return _this.editor.validatePort(_this.editorContext, cellView, magnet);
                    }
                    else {
                        return true;
                    }
                }
            },
            interactive: function (cellView, event) {
                if (_this.readOnlyCanvas) {
                    return false;
                }
                else {
                    if (_this.editor && _this.editor.interactive) {
                        if (typeof _this.editor.interactive === 'function') {
                            // Type for interactive is wrong in JointJS have to cast to <any>
                            return _this.editor.interactive(cellView, event);
                        }
                        else {
                            return _this.editor.interactive;
                        }
                    }
                    return true;
                }
            },
            highlighting: this.editor && this.editor.highlighting ? this.editor.highlighting : {
                'default': {
                    name: 'addClass',
                    options: {
                        className: 'highlighted'
                    }
                }
            },
            markAvailable: true
        };
        if (this.renderer && this.renderer.getLinkAnchorPoint) {
            options.linkConnectionPoint = this.renderer.getLinkAnchorPoint;
        }
        if (this.editor && this.editor.validateLink) {
            var self_1 = this;
            options.validateConnection = function (cellViewS, magnetS, cellViewT, magnetT, end, linkView) {
                return self_1.editor.validateLink(_this.editorContext, cellViewS, magnetS, cellViewT, magnetT, end === 'source', linkView);
            };
        }
        // The paper is what will represent the graph on the screen
        this.paper = new joint.dia.Paper(options);
        this._disposables.add(Disposable.create(function () { return _this.paper.remove(); }));
        this.paper.options.highlighterNamespace['addParentClass'] = {
            /**
             * @param {joint.dia.CellView} cellView
             * @param {Element} magnetEl
             * @param {object=} opt
             */
            highlight: function (cellView, magnetEl, opt) {
                opt = opt || {};
                var className = opt.className || this.className;
                joint.V(magnetEl.parentElement).addClass(className);
            },
            /**
             * @param {joint.dia.CellView} cellView
             * @param {Element} magnetEl
             * @param {object=} opt
             */
            unhighlight: function (cellView, magnetEl, opt) {
                opt = opt || {};
                var className = opt.className || this.className;
                joint.V(magnetEl.parentElement).removeClass(className);
            }
        };
    };
    EditorComponent.prototype.updatePaletteReadyState = function (ready) {
        this.paletteReady.next(ready);
    };
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "metamodel", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "renderer", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "editor", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Number),
        tslib_1.__metadata("design:paramtypes", [Number])
    ], EditorComponent.prototype, "paletteSize", null);
    tslib_1.__decorate([
        Output(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "paletteSizeChange", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "searchFilterPlaceHolder", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "paletteEntryPadding", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "minZoom", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "maxZoom", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "zoomStep", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "paperPadding", void 0);
    tslib_1.__decorate([
        Output(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "floApi", void 0);
    tslib_1.__decorate([
        Output(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "validationMarkers", void 0);
    tslib_1.__decorate([
        Output(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "contentValidated", void 0);
    tslib_1.__decorate([
        Output(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "dslChange", void 0);
    tslib_1.__decorate([
        Output(),
        tslib_1.__metadata("design:type", Object)
    ], EditorComponent.prototype, "onProperties", void 0);
    tslib_1.__decorate([
        Input(),
        tslib_1.__metadata("design:type", String),
        tslib_1.__metadata("design:paramtypes", [String])
    ], EditorComponent.prototype, "dsl", null);
    EditorComponent = tslib_1.__decorate([
        Component({
            selector: 'flo-editor',
            template: "\n    <ng-content select=\"[header]\"></ng-content>\n    <div id=\"flow-view\" class=\"flow-view\" style=\"position:relative\">\n      <div id=\"canvas\" class=\"canvas\" style=\"position:relative; display: block; width: 100%; height: 100%;\">\n        <div *ngIf=\"!noPalette\" id=\"palette-container\" class=\"palette-container\" style=\"overflow:hidden;\">\n          <flo-palette [metamodel]=\"metamodel\" [renderer]=\"renderer\" [paletteSize]=\"paletteSize\"\n                       [paletteEntryPadding]=\"paletteEntryPadding || {width: 12, height: 12}\"\n                       (onPaletteEntryDrop)=\"handleDnDFromPalette($event)\"\n                        (paletteReady)=\"updatePaletteReadyState($event)\"\n                        (paletteFocus)=\"graphToTextSync=true\"\n                        [searchFilterPlaceHolder]=\"searchFilterPlaceHolder\">\n          </flo-palette>\n        </div>\n\n        <div id=\"sidebar-resizer\" *ngIf=\"!noPalette\"\n          resizer\n          [splitSize]=\"paletteSize\"\n          (sizeChange)=\"paletteSize = $event\"\n          [resizerWidth]=\"6\"\n          [resizerLeft]=\"'#palette-container'\"\n          [resizerRight]=\"'#paper-container'\">\n        </div>\n\n          <flo-editor-paper tabindex=\"0\" (onDelete)=\"deleteSelected()\" (onBlur)=\"selection = null\"\n                            (onProperties)=\"onPropertiesHandle()\">\n            <div id=\"paper\" class=\"paper\" tabindex=\"0\" style=\"overflow: hidden; position: absolute; display: block; height:100%; width:100%; overflow:auto;\"></div>\n            <ng-content select=\"[canvas]\"></ng-content>\n          </flo-editor-paper>\n\n          <span class=\"canvas-controls-container\" ng-if=\"canvasControls\">\n            <table ng-if=\"canvasControls.zoom\" class=\"canvas-control zoom-canvas-control\">\n              <tbody>\n                <tr>\n                  <td>\n                    <input class=\"zoom-canvas-input canvas-control zoom-canvas-control\" type=\"text\"\n                           data-inline=\"true\" [(ngModel)]=\"zoomPercent\"\n                           size=\"3\">\n                  </td>\n                  <td>\n                    <label class=\"canvas-control zoom-canvas-label\">%</label>\n                  </td>\n                  <td>\n                    <input type=\"range\" data-inline=\"true\" [(ngModel)]=\"zoomPercent\"\n                           [step]=\"zoomStep\"\n                           [max]=\"maxZoom\" [min]=\"minZoom\" data-type=\"range\"\n                           name=\"range\" class=\"canvas-control zoom-canvas-control\">\n                  </td>\n                </tr>\n              </tbody>\n            </table>\n          </span>\n\n      </div>\n    </div>\n    <ng-content select=\"[footer]\"></ng-content>\n  ",
            styles: ["\n    flo-view {\n      width:100%;\n      height:100%;\n      margin: 0;\n      background-color: #eeeeee;\n      font-family: \"Varela Round\",sans-serif;\n      -webkit-user-select: none;\n      -khtml-user-select: none;\n      -moz-user-select: none;\n      -o-user-select: none;\n      user-select: none;\n    }\n\n    .canvas {\n      border: 1px solid;\n      border-color: #6db33f;\n      border-radius: 2px;\n      margin-top: 3px;\n    }\n\n    /* Canvas contains the palette on the left and the paper on the right */\n\n    .paper {\n      padding: 0px;\n      background-color: #ffffff;\n      /* \theight: 100%;\n          width: 100%;\n          position: relative;\n          overflow: hidden;\n       *//* \tmargin-left: 400px; */\n    }\n\n    #sidebar-resizer {\n      background-color: #34302d;\n      position: absolute;\n      top: 0;\n      bottom: 0;\n      width: 6px;\n      cursor: e-resize;\n    }\n\n    #palette-container {\n      background-color: #EEE;\n      position: absolute;\n      top: 0;\n      bottom: 0;\n      left: 0;\n      overflow: auto;\n    }\n\n    #paper-container {\n      position: absolute;\n      top: 0;\n      bottom: 0;\n      right: 0;\n      overflow: auto;\n      color: #FFF;\n      background-color: #FFF;\n      outline: none;\n    }\n\n    /* Tooltip START */\n\n    .node-tooltip .tooltip-description {\n      margin-top: 5px;\n      margin-left: 0px;\n      margin-bottom: 5px;\n    }\n\n    .node-tooltip {\n      display:none;\n      position:absolute;\n      border:1px solid #333;\n      background-color:#34302d;/*#161616;*/\n      border-radius:5px;\n      padding:5px;\n      color:#fff;\n      /*\tfont-size:12px Arial;*/\n      font-family: \"Varela Round\",sans-serif;\n      font-size: 19px;\n      z-index: 100;\n    }\n\n    .tooltip-title-type {\n      font-size: 24px;\n      font-weight: bold;\n    }\n\n    .tooltip-title-group {\n      padding-left: 5px;\n      font-size: 20px;\n      font-style: italic;\n    }\n\n    .node-tooltip-option-name {\n      font-family: monospace;/*\"Varela Round\",sans-serif;*/\n      font-size: 17px;\n      font-weight: bold;\n      padding-right: 20px;\n\n    }\n\n    .node-tooltip-option-description {\n      font-family: \"Varela Round\",sans-serif;\n      font-size: 18px;\n    }\n\n    /* Tooltip END */\n\n\n    /* Validation Error Marker on Canvas START */\n\n    .error-tooltip p {\n      margin-top: 5px;\n      margin-left: 0px;\n      margin-bottom: 5px;\n      color:#fff;\n    }\n    .error-tooltip {\n      display:none;\n      position:absolute;\n      border:1px solid #333;\n      background-color:red;/*#161616;*/\n      border-radius:5px;\n      padding:5px;\n      color:#fff;\n      /*\tfont-size:12px Arial;*/\n      font-family: \"Varela Round\",sans-serif;\n      font-size: 20px;\n      z-index: 100;\n    }\n\n    /* Validation Error Marker on Canvas END */\n\n    /* Controls on Canvas START */\n\n    .canvas-controls-container {\n      position: absolute;\n      right: 15px;\n      top: 5px;\n    }\n\n    .canvas-control {\n      background: transparent;\n      font-family: \"Varela Round\",sans-serif;\n      font-size: 11px;\n      vertical-align: middle;\n      margin: 0px;\n    }\n\n    .zoom-canvas-control {\n      border: 0px;\n      padding: 0px;\n      margin: 0px;\n      outline: none;\n    }\n\n    .zoom-canvas-input {\n      text-align: right;\n      font-weight:bold;\n      color: black;\n      background-color: transparent;\n    }\n\n    .zoom-canvas-label {\n      padding-right: 4px;\n      color: black;\n    }\n\n    /* Controls on Canvas END */\n\n\n\n\n    /* START - FLO CANVAS STYLES - override joint js styles */\n\n    .highlighted {\n      outline: none;\n    }\n\n    .joint-type-handle {\n      cursor: pointer;\n    }\n\n    .available-magnet {\n      stroke-width: 3;\n    }\n\n    .link {\n      fill: none;\n      stroke: #ccc;\n      stroke-width: 1.5px;\n    }\n\n    .link-tools .tool-options {\n      display: none;       /* by default, we don't display link options tool */\n    }\n\n    /* Make transparent the circle around the link-tools (cog) icon. It'll allow shape to have a circle clicking area */\n    .link-tools .tool-options circle {\n      fill: transparent;\n      stroke: transparent;\n    }\n\n    .link-tools .tool-options path {\n      fill: black;\n      stroke: black;\n    }\n\n    .link-tools .tool-remove circle {\n      fill: red;\n      stroke: red;\n    }\n\n    .link-tools .tool-remove path {\n      fill: white;\n      stroke: white;\n    }\n\n    .link-tools-container {\n      stroke-width: 0;\n      fill: transparent;\n    }\n\n    /* END - FLO CANVAS STYLES */\n  "],
            encapsulation: ViewEncapsulation.None
        }),
        tslib_1.__metadata("design:paramtypes", [ElementRef])
    ], EditorComponent);
    return EditorComponent;
}());
export { EditorComponent };
//# sourceMappingURL=editor.component.js.map