declare type ParamItems = {
    [k: string]: string;
};

declare const quotePatterns: {
    '``': string;
    '[]': string;
    '""-qq': string;
    '""-bs': string;
    '""-qq-bs': string;
    '""-raw': string;
    "''-qq": string;
    "''-bs": string;
    "''-qq-bs": string;
    "''-raw": string;
    $$: string;
    "'''..'''": string;
    '""".."""': string;
    '{}': string;
    "q''": string;
};

/** Token type enum for all possible Token categories */
declare enum TokenType {
    QUOTED_IDENTIFIER = "QUOTED_IDENTIFIER",
    IDENTIFIER = "IDENTIFIER",
    STRING = "STRING",
    VARIABLE = "VARIABLE",
    RESERVED_DATA_TYPE = "RESERVED_DATA_TYPE",
    RESERVED_PARAMETERIZED_DATA_TYPE = "RESERVED_PARAMETERIZED_DATA_TYPE",
    RESERVED_KEYWORD = "RESERVED_KEYWORD",
    RESERVED_FUNCTION_NAME = "RESERVED_FUNCTION_NAME",
    RESERVED_PHRASE = "RESERVED_PHRASE",
    RESERVED_SET_OPERATION = "RESERVED_SET_OPERATION",
    RESERVED_CLAUSE = "RESERVED_CLAUSE",
    RESERVED_SELECT = "RESERVED_SELECT",
    RESERVED_JOIN = "RESERVED_JOIN",
    ARRAY_IDENTIFIER = "ARRAY_IDENTIFIER",
    ARRAY_KEYWORD = "ARRAY_KEYWORD",
    CASE = "CASE",
    END = "END",
    WHEN = "WHEN",
    ELSE = "ELSE",
    THEN = "THEN",
    LIMIT = "LIMIT",
    BETWEEN = "BETWEEN",
    AND = "AND",
    OR = "OR",
    XOR = "XOR",
    OPERATOR = "OPERATOR",
    COMMA = "COMMA",
    ASTERISK = "ASTERISK",
    PROPERTY_ACCESS_OPERATOR = "PROPERTY_ACCESS_OPERATOR",
    OPEN_PAREN = "OPEN_PAREN",
    CLOSE_PAREN = "CLOSE_PAREN",
    LINE_COMMENT = "LINE_COMMENT",
    BLOCK_COMMENT = "BLOCK_COMMENT",
    DISABLE_COMMENT = "DISABLE_COMMENT",
    NUMBER = "NUMBER",
    NAMED_PARAMETER = "NAMED_PARAMETER",
    QUOTED_PARAMETER = "QUOTED_PARAMETER",
    NUMBERED_PARAMETER = "NUMBERED_PARAMETER",
    POSITIONAL_PARAMETER = "POSITIONAL_PARAMETER",
    CUSTOM_PARAMETER = "CUSTOM_PARAMETER",
    DELIMITER = "DELIMITER",
    EOF = "EOF"
}
/** Struct to store the most basic cohesive unit of language grammar */
interface Token {
    type: TokenType;
    raw: string;
    text: string;
    key?: string;
    start: number;
    precedingWhitespace?: string;
}

interface IdentChars {
    first?: string;
    rest?: string;
    dashes?: boolean;
    allowFirstCharNumber?: boolean;
}
declare type PlainQuoteType = keyof typeof quotePatterns;
interface PrefixedQuoteType {
    quote: PlainQuoteType;
    prefixes: string[];
    requirePrefix?: boolean;
}
interface RegexPattern {
    regex: string;
}
declare type QuoteType = PlainQuoteType | PrefixedQuoteType | RegexPattern;
declare type VariableType = RegexPattern | PrefixedQuoteType;
interface ParamTypes {
    positional?: boolean;
    numbered?: ('?' | ':' | '$')[];
    named?: (':' | '@' | '$')[];
    quoted?: (':' | '@' | '$')[];
    custom?: CustomParameter[];
}
interface CustomParameter {
    regex: string;
    key?: (text: string) => string;
}
interface TokenizerOptions {
    reservedSelect: string[];
    reservedClauses: string[];
    supportsXor?: boolean;
    reservedSetOperations: string[];
    reservedJoins: string[];
    reservedPhrases?: string[];
    reservedFunctionNames: string[];
    reservedDataTypes: string[];
    reservedKeywords: string[];
    stringTypes: QuoteType[];
    identTypes: QuoteType[];
    variableTypes?: VariableType[];
    extraParens?: ('[]' | '{}')[];
    paramTypes?: ParamTypes;
    lineCommentTypes?: string[];
    nestedBlockComments?: boolean;
    identChars?: IdentChars;
    paramChars?: IdentChars;
    operators?: string[];
    propertyAccessOperators?: string[];
    postProcess?: (tokens: Token[]) => Token[];
}

declare type IndentStyle = 'standard' | 'tabularLeft' | 'tabularRight';
declare type KeywordCase = 'preserve' | 'upper' | 'lower';
declare type IdentifierCase = KeywordCase;
declare type DataTypeCase = KeywordCase;
declare type FunctionCase = KeywordCase;
declare type LogicalOperatorNewline = 'before' | 'after';
interface FormatOptions {
    tabWidth: number;
    useTabs: boolean;
    keywordCase: KeywordCase;
    identifierCase: IdentifierCase;
    dataTypeCase: DataTypeCase;
    functionCase: FunctionCase;
    indentStyle: IndentStyle;
    logicalOperatorNewline: LogicalOperatorNewline;
    expressionWidth: number;
    linesBetweenQueries: number;
    denseOperators: boolean;
    newlineBeforeSemicolon: boolean;
    params?: ParamItems | string[];
    paramTypes?: ParamTypes;
}

interface DialectFormatOptions {
    alwaysDenseOperators?: string[];
    onelineClauses: string[];
    tabularOnelineClauses?: string[];
}

interface DialectOptions {
    name: string;
    tokenizerOptions: TokenizerOptions;
    formatOptions: DialectFormatOptions;
}

declare const bigquery: DialectOptions;

declare const db2: DialectOptions;

declare const db2i: DialectOptions;

declare const hive: DialectOptions;

declare const mariadb: DialectOptions;

declare const mysql: DialectOptions;

declare const tidb: DialectOptions;

declare const n1ql: DialectOptions;

declare const plsql: DialectOptions;

declare const postgresql: DialectOptions;

declare const redshift: DialectOptions;

declare const spark: DialectOptions;

declare const sqlite: DialectOptions;

declare const sql: DialectOptions;

declare const trino: DialectOptions;

declare const transactsql: DialectOptions;

declare const singlestoredb: DialectOptions;

declare const snowflake: DialectOptions;

declare const allDialects_bigquery: typeof bigquery;
declare const allDialects_db2: typeof db2;
declare const allDialects_db2i: typeof db2i;
declare const allDialects_hive: typeof hive;
declare const allDialects_mariadb: typeof mariadb;
declare const allDialects_mysql: typeof mysql;
declare const allDialects_n1ql: typeof n1ql;
declare const allDialects_plsql: typeof plsql;
declare const allDialects_postgresql: typeof postgresql;
declare const allDialects_redshift: typeof redshift;
declare const allDialects_singlestoredb: typeof singlestoredb;
declare const allDialects_snowflake: typeof snowflake;
declare const allDialects_spark: typeof spark;
declare const allDialects_sql: typeof sql;
declare const allDialects_sqlite: typeof sqlite;
declare const allDialects_tidb: typeof tidb;
declare const allDialects_transactsql: typeof transactsql;
declare const allDialects_trino: typeof trino;
declare namespace allDialects {
  export { allDialects_bigquery as bigquery, allDialects_db2 as db2, allDialects_db2i as db2i, allDialects_hive as hive, allDialects_mariadb as mariadb, allDialects_mysql as mysql, allDialects_n1ql as n1ql, allDialects_plsql as plsql, allDialects_postgresql as postgresql, allDialects_redshift as redshift, allDialects_singlestoredb as singlestoredb, allDialects_snowflake as snowflake, allDialects_spark as spark, allDialects_sql as sql, allDialects_sqlite as sqlite, allDialects_tidb as tidb, allDialects_transactsql as transactsql, allDialects_trino as trino };
}

declare const dialectNameMap: Record<keyof typeof allDialects | 'tsql', keyof typeof allDialects>;
declare const supportedDialects: string[];
declare type SqlLanguage = keyof typeof dialectNameMap;
declare type FormatOptionsWithLanguage = Partial<FormatOptions> & {
    language?: SqlLanguage;
};
declare type FormatOptionsWithDialect = Partial<FormatOptions> & {
    dialect: DialectOptions;
};
/**
 * Format whitespace in a query to make it easier to read.
 *
 * @param {string} query - input SQL query string
 * @param {FormatOptionsWithLanguage} cfg Configuration options (see docs in README)
 * @return {string} formatted query
 */
declare const format: (query: string, cfg?: FormatOptionsWithLanguage) => string;
/**
 * Like the above format(), but language parameter is mandatory
 * and must be a Dialect object instead of a string.
 *
 * @param {string} query - input SQL query string
 * @param {FormatOptionsWithDialect} cfg Configuration options (see docs in README)
 * @return {string} formatted query
 */
declare const formatDialect: (query: string, { dialect, ...cfg }: FormatOptionsWithDialect) => string;

/**
 * Performs expandSinglePhrase() on array
 */
declare const expandPhrases: (phrases: string[]) => string[];

declare class ConfigError extends Error {
}

export { ConfigError, type DataTypeCase, type DialectOptions, type FormatOptions, type FormatOptionsWithDialect, type FormatOptionsWithLanguage, type FunctionCase, type IdentifierCase, type IndentStyle, type KeywordCase, type LogicalOperatorNewline, type ParamItems, type ParamTypes, type SqlLanguage, bigquery, db2, db2i, expandPhrases, format, formatDialect, hive, mariadb, mysql, n1ql, plsql, postgresql, redshift, singlestoredb, snowflake, spark, sql, sqlite, supportedDialects, tidb, transactsql, trino };
