import { Hash, elUniqId } from '../utils/dom';
import { camelize, lcFirst } from '../utils/text';
import { debounce } from '../utils/timing';
import { effect } from '../vendored/preact-core';
import { DSP, DSS } from './consts';
import { initErr, runtimeErr } from './errors';
import { SignalsRoot } from './signals';
import { PluginType, Requirement, } from './types';
const removalKey = (k, v) => `${k}${DSP}${v}`;
export class Engine {
    #signals;
    #plugins;
    #actions;
    #watchers;
    // Map of cleanup functions by element, keyed by the dataset key and value
    #removals;
    constructor() {
        this.aliasPrefix = '';
        this.#signals = new SignalsRoot();
        this.#plugins = [];
        this.#actions = {};
        this.#watchers = [];
        // Map of cleanup functions by element, keyed by the dataset key and value
        this.#removals = new Map();
        this.#debouncedApply = debounce(() => {
            this.#apply(document.body);
        }, 1);
        const datasetPrefix = 'data-';
        const ob = new MutationObserver((mutations) => {
            for (const { target, type, attributeName, oldValue, addedNodes, removedNodes, } of mutations) {
                switch (type) {
                    case 'childList':
                        {
                            for (const node of removedNodes) {
                                const el = node;
                                const elRemovals = this.#removals.get(el);
                                if (!elRemovals)
                                    continue;
                                for (const [_, removalFn] of elRemovals) {
                                    removalFn();
                                }
                                this.#removals.delete(el);
                            }
                            for (const node of addedNodes) {
                                const el = node;
                                this.#apply(el);
                            }
                        }
                        break;
                    case 'attributes': {
                        {
                            const requiredPrefix = datasetPrefix + (this.aliasPrefix ? `${this.aliasPrefix}-` : '');
                            if (!attributeName?.startsWith(requiredPrefix)) {
                                break;
                            }
                            const el = target;
                            const datasetKey = camelize(attributeName.slice(datasetPrefix.length));
                            // If the value has changed, clean up the old value
                            if (oldValue !== null && el.dataset[datasetKey] !== oldValue) {
                                const elRemovals = this.#removals.get(el);
                                if (elRemovals) {
                                    const rk = removalKey(datasetKey, oldValue);
                                    const removalFn = elRemovals.get(rk);
                                    if (removalFn) {
                                        removalFn();
                                        elRemovals.delete(rk);
                                    }
                                }
                            }
                            // Apply the plugin only if the dataset key exists
                            if (datasetKey in el.dataset) {
                                this.#applyAttributePlugin(el, datasetKey);
                            }
                        }
                        break;
                    }
                }
            }
        });
        ob.observe(document.body, {
            attributes: true,
            attributeOldValue: true,
            childList: true,
            subtree: true,
        });
    }
    get signals() {
        return this.#signals;
    }
    load(...pluginsToLoad) {
        for (const plugin of pluginsToLoad) {
            const that = this; // I hate javascript
            const ctx = {
                get signals() {
                    return that.#signals;
                },
                effect: (cb) => effect(cb),
                actions: this.#actions,
                plugin,
            };
            let globalInitializer;
            switch (plugin.type) {
                case PluginType.Watcher: {
                    const wp = plugin;
                    this.#watchers.push(wp);
                    globalInitializer = wp.onGlobalInit;
                    break;
                }
                case PluginType.Action: {
                    this.#actions[plugin.name] = plugin;
                    break;
                }
                case PluginType.Attribute: {
                    const ap = plugin;
                    this.#plugins.push(ap);
                    globalInitializer = ap.onGlobalInit;
                    break;
                }
                default: {
                    throw initErr('InvalidPluginType', ctx);
                }
            }
            if (globalInitializer) {
                globalInitializer(ctx);
            }
        }
        // Sort attribute plugins by descending length then alphabetically
        this.#plugins.sort((a, b) => {
            const lenDiff = b.name.length - a.name.length;
            if (lenDiff !== 0)
                return lenDiff;
            return a.name.localeCompare(b.name);
        });
        this.#debouncedApply();
    }
    #debouncedApply;
    // Apply all plugins to the element and its children
    #apply(rootElement) {
        this.#walkDownDOM(rootElement, (el) => {
            // Cleanup any existing removal functions
            const elRemovals = this.#removals.get(el);
            if (elRemovals) {
                for (const [, removalFn] of elRemovals) {
                    removalFn();
                }
                this.#removals.delete(el);
            }
            // Apply the plugins to the element in order of application
            // since DOMStringMap is ordered, we can be deterministic
            for (const datasetKey of Object.keys(el.dataset)) {
                this.#applyAttributePlugin(el, datasetKey);
            }
        });
    }
    #applyAttributePlugin(el, datasetKey) {
        // Extract the raw key from the dataset
        const rawKey = lcFirst(datasetKey.slice(this.aliasPrefix.length));
        // Find the plugin that matches, since the plugins are sorted by length descending and alphabetically
        // the first match will be the most specific
        const plugin = this.#plugins.find((p) => rawKey.startsWith(p.name));
        // Skip if no plugin is found
        if (!plugin)
            return;
        // Ensure the element has an id
        if (!el.id.length)
            el.id = elUniqId(el);
        // Extract the key and modifiers
        let [key, ...rawModifiers] = rawKey.slice(plugin.name.length).split(/\_\_+/);
        const hasKey = key.length > 0;
        if (hasKey) {
            // Keys starting with a dash are not converted to camel case in the dataset
            key = key.startsWith('-') ? key.slice(1) : lcFirst(key);
        }
        const value = el.dataset[datasetKey] || '';
        const hasValue = value.length > 0;
        // Create the runtime context
        const that = this; // I hate javascript
        const ctx = {
            get signals() {
                return that.#signals;
            },
            effect: (cb) => effect(cb),
            actions: this.#actions,
            genRX: () => this.#genRX(ctx, ...(plugin.argNames || [])),
            plugin,
            el,
            rawKey,
            key,
            value,
            mods: new Map(),
        };
        // Check the requirements
        const keyReq = plugin.keyReq || Requirement.Allowed;
        if (hasKey) {
            if (keyReq === Requirement.Denied) {
                throw runtimeErr(`${plugin.name}KeyNotAllowed`, ctx);
            }
        }
        else if (keyReq === Requirement.Must) {
            throw runtimeErr(`${plugin.name}KeyRequired`, ctx);
        }
        const valReq = plugin.valReq || Requirement.Allowed;
        if (hasValue) {
            if (valReq === Requirement.Denied) {
                throw runtimeErr(`${plugin.name}ValueNotAllowed`, ctx);
            }
        }
        else if (valReq === Requirement.Must) {
            throw runtimeErr(`${plugin.name}ValueRequired`, ctx);
        }
        // Check for exclusive requirements
        if (keyReq === Requirement.Exclusive || valReq === Requirement.Exclusive) {
            if (hasKey && hasValue) {
                throw runtimeErr(`${plugin.name}KeyAndValueProvided`, ctx);
            }
            if (!hasKey && !hasValue) {
                throw runtimeErr(`${plugin.name}KeyOrValueRequired`, ctx);
            }
        }
        for (const rawMod of rawModifiers) {
            const [label, ...mod] = rawMod.split('.');
            ctx.mods.set(camelize(label), new Set(mod.map((t) => t.toLowerCase())));
        }
        // Load the plugin and store any cleanup functions
        const removalFn = plugin.onLoad(ctx);
        if (removalFn) {
            let elRemovals = this.#removals.get(el);
            if (!elRemovals) {
                elRemovals = new Map();
                this.#removals.set(el, elRemovals);
            }
            elRemovals.set(removalKey(datasetKey, value), removalFn);
        }
        // Remove the attribute if required
        const removeOnLoad = plugin.removeOnLoad;
        if (removeOnLoad && removeOnLoad(rawKey) === true) {
            delete el.dataset[datasetKey];
        }
    }
    #genRX(ctx, ...argNames) {
        let userExpression = '';
        // This regex allows Datastar expressions to support nested
        // regex and strings that contain ; without breaking.
        //
        // Each of these regex defines a block type we want to match
        // (importantly we ignore the content within these blocks):
        //
        // regex            \/(\\\/|[^\/])*\/
        // double quotes      "(\\"|[^\"])*"
        // single quotes      '(\\'|[^'])*'
        // ticks              `(\\`|[^`])*`
        //
        // We also want to match the non delimiter part of statements
        // note we only support ; statement delimiters:
        //
        // [^;]
        //
        const statementRe = /(\/(\\\/|[^\/])*\/|"(\\"|[^\"])*"|'(\\'|[^'])*'|`(\\`|[^`])*`|[^;])+/gm;
        const statements = ctx.value.trim().match(statementRe);
        if (statements) {
            const lastIdx = statements.length - 1;
            const last = statements[lastIdx].trim();
            if (!last.startsWith('return')) {
                statements[lastIdx] = `return (${last});`;
            }
            userExpression = statements.join(';\n');
        }
        // Ignore any escaped values
        const escaped = new Map();
        const escapeRe = new RegExp(`(?:${DSP})(.*?)(?:${DSS})`, 'gm');
        for (const match of userExpression.matchAll(escapeRe)) {
            const k = match[1];
            const v = new Hash('dsEscaped').with(k).value;
            escaped.set(v, k);
            userExpression = userExpression.replace(DSP + k + DSS, v);
        }
        const fnCall = /@(\w*)\(/gm;
        const matches = userExpression.matchAll(fnCall);
        const methodsCalled = new Set();
        for (const match of matches) {
            methodsCalled.add(match[1]);
        }
        // Replace any action calls
        const actionsRe = new RegExp(`@(${Object.keys(this.#actions).join('|')})\\(`, 'gm');
        // Add ctx to action calls
        userExpression = userExpression.replaceAll(actionsRe, 'ctx.actions.$1.fn(ctx,');
        // Replace any signal calls
        const signalNames = ctx.signals.paths();
        if (signalNames.length) {
            // Match any valid `$signalName` followed by a non-word character or end of string
            const signalsRe = new RegExp(`\\$(${signalNames.join('|')})(\\W|$)`, 'gm');
            userExpression = userExpression.replaceAll(signalsRe, `ctx.signals.signal('$1').value$2`);
        }
        // Replace any escaped values
        for (const [k, v] of escaped) {
            userExpression = userExpression.replace(k, v);
        }
        const fnContent = `return (()=> {\n${userExpression}\n})()`; // Wrap in IIFE
        ctx.fnContent = fnContent;
        try {
            const fn = new Function('ctx', ...argNames, fnContent);
            return (...args) => {
                try {
                    return fn(ctx, ...args);
                }
                catch (error) {
                    throw runtimeErr('ExecuteExpression', ctx, {
                        error: error.message,
                    });
                }
            };
        }
        catch (error) {
            throw runtimeErr('GenerateExpression', ctx, {
                error: error.message,
            });
        }
    }
    #walkDownDOM(element, callback) {
        if (!element ||
            !(element instanceof HTMLElement || element instanceof SVGElement)) {
            return null;
        }
        const dataset = element.dataset;
        if ('starIgnore' in dataset) {
            return null;
        }
        if (!('starIgnore__self' in dataset)) {
            callback(element);
        }
        let el = element.firstElementChild;
        while (el) {
            this.#walkDownDOM(el, callback);
            el = el.nextElementSibling;
        }
    }
}
