import { BYTE } from './byte';
/**
 * Frame class represents a STOMP frame. Many of the callbacks pass the Frame received from
 * the STOMP broker. For advanced usage you might need to access [headers]{@link Frame#headers}.
 *
 * Part of `@stomp/stompjs`.
 *
 * {@link Message} is an extended Frame.
 */
var Frame = /** @class */ (function () {
    /**
     * Frame constructor. `command`, `headers` and `body` are available as properties.
     *
     * @internal
     */
    function Frame(params) {
        var command = params.command, headers = params.headers, body = params.body, binaryBody = params.binaryBody, escapeHeaderValues = params.escapeHeaderValues, skipContentLengthHeader = params.skipContentLengthHeader;
        this.command = command;
        this.headers = Object.assign({}, headers || {});
        if (binaryBody) {
            this._binaryBody = binaryBody;
            this.isBinaryBody = true;
        }
        else {
            this._body = body || '';
            this.isBinaryBody = false;
        }
        this.escapeHeaderValues = escapeHeaderValues || false;
        this.skipContentLengthHeader = skipContentLengthHeader || false;
    }
    Object.defineProperty(Frame.prototype, "body", {
        /**
         * body of the frame
         */
        get: function () {
            if (!this._body && this.isBinaryBody) {
                this._body = new TextDecoder().decode(this._binaryBody);
            }
            return this._body;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(Frame.prototype, "binaryBody", {
        /**
         * body as Uint8Array
         */
        get: function () {
            if (!this._binaryBody && !this.isBinaryBody) {
                this._binaryBody = new TextEncoder().encode(this._body);
            }
            return this._binaryBody;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * deserialize a STOMP Frame from raw data.
     *
     * @internal
     */
    Frame.fromRawFrame = function (rawFrame, escapeHeaderValues) {
        var headers = {};
        var trim = function (str) { return str.replace(/^\s+|\s+$/g, ''); };
        // In case of repeated headers, as per standards, first value need to be used
        for (var _i = 0, _a = rawFrame.headers.reverse(); _i < _a.length; _i++) {
            var header = _a[_i];
            var idx = header.indexOf(':');
            var key = trim(header[0]);
            var value = trim(header[1]);
            if (escapeHeaderValues && (rawFrame.command !== 'CONNECT') && (rawFrame.command !== 'CONNECTED')) {
                value = Frame.hdrValueUnEscape(value);
            }
            headers[key] = value;
        }
        return new Frame({
            command: rawFrame.command,
            headers: headers,
            binaryBody: rawFrame.binaryBody,
            escapeHeaderValues: escapeHeaderValues
        });
    };
    /**
     * @internal
     */
    Frame.prototype.toString = function () {
        return this.serializeCmdAndHeaders();
    };
    /**
     * serialize this Frame in a format suitable to be passed to WebSocket.
     * If the body is string the output will be string.
     * If the body is binary (i.e. of type Unit8Array) it will be serialized to ArrayBuffer.
     *
     * @internal
     */
    Frame.prototype.serialize = function () {
        var cmdAndHeaders = this.serializeCmdAndHeaders();
        if (this.isBinaryBody) {
            return Frame.toUnit8Array(cmdAndHeaders, this._binaryBody).buffer;
        }
        else {
            return cmdAndHeaders + this._body + BYTE.NULL;
        }
    };
    Frame.prototype.serializeCmdAndHeaders = function () {
        var lines = [this.command];
        if (this.skipContentLengthHeader) {
            delete this.headers['content-length'];
        }
        for (var _i = 0, _a = Object.keys(this.headers || {}); _i < _a.length; _i++) {
            var name_1 = _a[_i];
            var value = this.headers[name_1];
            if (this.escapeHeaderValues && (this.command !== 'CONNECT') && (this.command !== 'CONNECTED')) {
                lines.push(name_1 + ":" + Frame.hdrValueEscape("" + value));
            }
            else {
                lines.push(name_1 + ":" + value);
            }
        }
        if (this.isBinaryBody || (!this.isBodyEmpty() && !this.skipContentLengthHeader)) {
            lines.push("content-length:" + this.bodyLength());
        }
        return lines.join(BYTE.LF) + BYTE.LF + BYTE.LF;
    };
    Frame.prototype.isBodyEmpty = function () {
        return this.bodyLength() === 0;
    };
    Frame.prototype.bodyLength = function () {
        var binaryBody = this.binaryBody;
        return binaryBody ? binaryBody.length : 0;
    };
    /**
     * Compute the size of a UTF-8 string by counting its number of bytes
     * (and not the number of characters composing the string)
     */
    Frame.sizeOfUTF8 = function (s) {
        return s ? new TextEncoder().encode(s).length : 0;
    };
    Frame.toUnit8Array = function (cmdAndHeaders, binaryBody) {
        var uint8CmdAndHeaders = new TextEncoder().encode(cmdAndHeaders);
        var nullTerminator = new Uint8Array([0]);
        var uint8Frame = new Uint8Array(uint8CmdAndHeaders.length + binaryBody.length + nullTerminator.length);
        uint8Frame.set(uint8CmdAndHeaders);
        uint8Frame.set(binaryBody, uint8CmdAndHeaders.length);
        uint8Frame.set(nullTerminator, uint8CmdAndHeaders.length + binaryBody.length);
        return uint8Frame;
    };
    /**
     * Serialize a STOMP frame as per STOMP standards, suitable to be sent to the STOMP broker.
     *
     * @internal
     */
    Frame.marshall = function (params) {
        var frame = new Frame(params);
        return frame.serialize();
    };
    /**
     *  Escape header values
     */
    Frame.hdrValueEscape = function (str) {
        return str.replace(/\\/g, '\\\\').replace(/\r/g, '\\r').replace(/\n/g, '\\n').replace(/:/g, '\\c');
    };
    /**
     * UnEscape header values
     */
    Frame.hdrValueUnEscape = function (str) {
        return str.replace(/\\r/g, '\r').replace(/\\n/g, '\n').replace(/\\c/g, ':').replace(/\\\\/g, '\\');
    };
    return Frame;
}());
export { Frame };
//# sourceMappingURL=frame.js.map