"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
class Activatable {
    constructor() {
        this._state = 'deactivated';
    }
    get state() {
        return this._state;
    }
    activate() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            switch (this._state) {
                case 'activated':
                    return;
                case 'deactivated':
                    try {
                        this._state = 'isActivating';
                        yield this.doActivate();
                        this._state = 'activated';
                        return;
                    }
                    catch (e) {
                        this._state = 'deactivated';
                        throw e;
                    }
                default:
                    throw new Error(`Cannot call activate on an Activatable in state '${this._state}'`);
            }
        });
    }
    deactivate() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            switch (this._state) {
                case 'deactivated':
                    return;
                case 'activated':
                    try {
                        this._state = 'isDeactivating';
                        yield this.doDeactivate();
                        this._state = 'deactivated';
                        return;
                    }
                    catch (e) {
                        this._state = 'activated';
                        throw e;
                    }
                default:
                    throw new Error(`Cannot call deactivate on an Activatable in state '${this._state}'`);
            }
        });
    }
}
exports.Activatable = Activatable;
class ActivatableCollection extends Activatable {
    constructor() {
        super(...arguments);
        this.activatables = [];
    }
    doActivate() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const activated = [];
            try {
                for (const activatable of this.activatables) {
                    yield activatable.activate();
                    activated.push(activatable);
                }
            }
            catch (e) {
                activated.reverse();
                for (const activatable of activated) {
                    try {
                        yield activatable.deactivate();
                    }
                    catch (_) {
                    }
                }
                throw e;
            }
        });
    }
    doDeactivate() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            for (const activatable of [...this.activatables].reverse()) {
                yield activatable.deactivate();
            }
        });
    }
    push(activatables) {
        this.activatables.push(activatables);
    }
}
exports.ActivatableCollection = ActivatableCollection;
//# sourceMappingURL=activatable.js.map