"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Emitter = require("wolfy87-eventemitter");
const disposable_1 = require("../disposable");
class EventEmitter {
    constructor() {
        this._emitter = new Emitter();
        this._registeredListeners = new Map();
    }
    on(type, listener) {
        let registeredListeners = this._registeredListeners.get(type);
        if (registeredListeners === void 0) {
            registeredListeners = new WeakSet();
            this._registeredListeners.set(type, registeredListeners);
        }
        else if (registeredListeners.has(listener)) {
            throw new Error(`Double-registered for '${type}' event.`);
        }
        const wrappedListener = (...args) => {
            try {
                listener(...args);
            }
            catch (ex) {
                console.error(ex);
            }
        };
        registeredListeners.add(listener);
        this._emitter.on(String(type), wrappedListener);
        return disposable_1.createDisposable(() => {
            var _a;
            (_a = registeredListeners) === null || _a === void 0 ? void 0 : _a.delete(listener);
            this._emitter.off(String(type), wrappedListener);
        });
    }
    emit(type, ...args) {
        this._emitter.trigger(String(type), args);
    }
    get hasListeners() {
        const eventsToListeners = this._emitter.getListeners(/.*/);
        for (const ev in eventsToListeners) {
            if (!{}.hasOwnProperty.call(eventsToListeners, ev))
                continue;
            const l = eventsToListeners[ev];
            if (l !== undefined && l.length > 0) {
                return true;
            }
        }
        return false;
    }
    dispose() {
        this._registeredListeners.clear();
        this._emitter.removeAllListeners();
    }
    createEmitGroup() {
        const notifier = this;
        const eventQueue = [];
        let flushed = false;
        return {
            get queueCount() {
                return eventQueue.length;
            },
            emit(event, ...args) {
                if (flushed) {
                    notifier.emit(event, ...args);
                }
                else {
                    eventQueue.push([event, args]);
                }
            },
            flush() {
                for (const [event, args] of eventQueue) {
                    try {
                        notifier.emit(event, ...args);
                    }
                    catch (e) {
                    }
                }
                this.reset();
                flushed = true;
            },
            reset() {
                eventQueue.length = 0;
                flushed = false;
            },
        };
    }
}
exports.EventEmitter = EventEmitter;
//# sourceMappingURL=emitter.js.map