"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.markdown = void 0;
const tslib_1 = require("tslib");
const spectral_runtime_1 = require("@stoplight/spectral-runtime");
const utils_1 = require("./utils");
const types_1 = require("@stoplight/types");
const markdown_escape_1 = (0, tslib_1.__importDefault)(require("markdown-escape"));
const getDocumentationUrl_1 = require("./utils/getDocumentationUrl");
const markdown = (results, { failSeverity }, ctx) => {
    const groupedResults = (0, utils_1.groupBySource)(results);
    const lines = [];
    for (const [source, validationResults] of Object.entries(groupedResults)) {
        validationResults.sort((a, b) => a.range.start.line - b.range.start.line);
        if (validationResults.length > 0) {
            const filteredValidationResults = validationResults.filter(result => result.severity <= failSeverity);
            for (const result of filteredValidationResults) {
                const ruleDocumentationUrl = (0, getDocumentationUrl_1.getRuleDocumentationUrl)(result.code, ctx);
                const codeWithOptionalLink = ruleDocumentationUrl != null
                    ? `[${result.code.toString()}](${ruleDocumentationUrl})`
                    : result.code.toString();
                const escapedPath = (0, markdown_escape_1.default)((0, spectral_runtime_1.printPath)(result.path, spectral_runtime_1.PrintStyle.Dot));
                const escapedMessage = (0, markdown_escape_1.default)(result.message);
                const severityString = types_1.DiagnosticSeverity[result.severity];
                const start = `${result.range.start.line}:${result.range.start.character}`;
                const end = `${result.range.end.line}:${result.range.end.character}`;
                const escapedSource = (0, markdown_escape_1.default)(source);
                lines.push([codeWithOptionalLink, escapedPath, escapedMessage, severityString, start, end, escapedSource]);
            }
        }
    }
    const headers = ['Code', 'Path', 'Message', 'Severity', 'Start', 'End', 'Source'];
    return createMdTable(headers, lines);
};
exports.markdown = markdown;
function createMdTable(headers, lines) {
    const columnLengths = headers.map((_, i) => Math.max(...lines.map(line => line[i].length), headers[i].length));
    let string = '';
    string += '|';
    for (const header of headers) {
        string += ` ${header}`;
        string += ' '.repeat(columnLengths[headers.indexOf(header)] - header.length);
        string += ' |';
    }
    string += '\n|';
    for (const _ of headers) {
        string += ' ';
        string += '-'.repeat(columnLengths[headers.indexOf(_)]);
        string += ' |';
    }
    for (const line of lines) {
        string += '\n|';
        for (const cell of line) {
            string += ` ${cell}`;
            string += ' '.repeat(columnLengths[line.indexOf(cell)] - cell.length);
            string += ' |';
        }
    }
    return string;
}
//# sourceMappingURL=markdown.js.map