"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.messages = exports.ruleName = undefined;

exports.default = function (expectation) {

  var quoteMsg = function () {
    switch (expectation) {
      case "single":
        return "single quotes";
      case "double":
        return "double quotes";
      case "none":
        return "no quotes";

      case "always":
        return "quotes";
      case "never":
        return "no quotes";
    }
  }();

  var charDefiesExpectation = function () {
    switch (expectation) {
      case "single":
        return function (c) {
          return c !== "'";
        };
      case "double":
        return function (c) {
          return c !== "\"";
        };
      case "none":
        return function (c) {
          return c === "'" || c === "\"";
        };

      case "never":
        return function (c) {
          return c === "'" || c === "\"";
        };
      case "always":
        return function (c) {
          return c !== "'" && c !== "\"";
        };
    }
  }();

  function strDefiesExpectation(str) {
    return charDefiesExpectation(str[0]) || charDefiesExpectation(str[str.length - 1]);
  }

  return function (root, result) {
    var validOptions = (0, _utils.validateOptions)(result, ruleName, {
      actual: expectation,
      possible: ["single", "double", "none", "always", "never"]
    });
    if (!validOptions) {
      return;
    }

    if (expectation === "single" || expectation === "double" || expectation === "none") {
      result.warn("The '" + expectation + "' option for 'function-url-quotes' has been deprecated, " + "and will be removed in '7.0'. Instead, use the 'always' or 'never' options together with the 'string-quotes' rule.", {
        stylelintType: "deprecation",
        stylelintReference: "http://stylelint.io/user-guide/rules/function-url-quotes/"
      });
    }

    root.walkAtRules(check);
    root.walkRules(check);

    function check(statement) {
      if (statement.type === "atrule") {
        checkAtRuleParams(statement);
      }

      statement.walkDecls(function (decl) {
        (0, _utils.functionArgumentsSearch)(decl.toString().toLowerCase(), "url", function (args, index) {
          var trimLeftArgs = args.trimLeft();
          if (!strDefiesExpectation(trimLeftArgs.trimRight())) {
            return;
          }
          complain(messages.expected(quoteMsg), decl, index + args.length - trimLeftArgs.length);
        });
      });
    }

    function checkAtRuleParams(atRule) {
      var atRuleParamsLowerCase = atRule.params.toLowerCase();
      (0, _utils.functionArgumentsSearch)(atRuleParamsLowerCase, "url", function (args, index) {
        var trimLeftArgs = args.trimLeft();
        if (!strDefiesExpectation(trimLeftArgs.trimRight())) {
          return;
        }
        complain(messages.expected(quoteMsg), atRule, index + args.length - trimLeftArgs.length + (0, _utils.atRuleParamIndex)(atRule));
      });
      (0, _utils.functionArgumentsSearch)(atRuleParamsLowerCase, "url-prefix", function (args, index) {
        var trimLeftArgs = args.trimLeft();
        if (!strDefiesExpectation(trimLeftArgs.trimRight())) {
          return;
        }
        complain(messages.expected(quoteMsg, "url-prefix"), atRule, index + args.length - trimLeftArgs.length + (0, _utils.atRuleParamIndex)(atRule));
      });
      (0, _utils.functionArgumentsSearch)(atRuleParamsLowerCase, "domain", function (args, index) {
        var trimLeftArgs = args.trimLeft();
        if (!strDefiesExpectation(trimLeftArgs.trimRight())) {
          return;
        }
        complain(messages.expected(quoteMsg, "domain"), atRule, index + args.length - trimLeftArgs.length + (0, _utils.atRuleParamIndex)(atRule));
      });
    }

    function complain(message, node, index) {
      (0, _utils.report)({
        message: message,
        node: node,
        index: index,
        result: result,
        ruleName: ruleName
      });
    }
  };
};

var _utils = require("../../utils");

var ruleName = exports.ruleName = "function-url-quotes";

var messages = exports.messages = (0, _utils.ruleMessages)(ruleName, {
  expected: function expected(q) {
    var f = arguments.length <= 1 || arguments[1] === undefined ? "url" : arguments[1];
    return "Expected " + q + " around " + f + " argument";
  }
});