"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.messages = exports.ruleName = undefined;

exports.default = function (max) {
  return function (root, result) {
    var validOptions = (0, _utils.validateOptions)(result, ruleName, {
      actual: max,
      possible: [function (max) {
        return typeof max === "number" && max > 0;
      }]
    });
    if (!validOptions) {
      return;
    }

    // Finds actual selectors in selectorNode object and checks them
    function checkSelector(selectorNode, rule) {
      var compoundCount = 1;

      selectorNode.each(function (childNode) {
        // Only traverse inside actual selectors and :not()
        if (childNode.type === "selector" || childNode.value === ":not") {
          checkSelector(childNode, rule);
        }

        // Compund selectors are separated by combinators, so increase count when meeting one
        if (childNode.type === "combinator") {
          compoundCount++;
        }
      });

      if (selectorNode.type !== "root" && selectorNode.type !== "pseudo" && compoundCount > max) {
        (0, _utils.report)({
          ruleName: ruleName,
          result: result,
          node: rule,
          message: messages.expected(selectorNode, max),
          word: selectorNode
        });
      }
    }

    root.walkRules(function (rule) {
      if (!(0, _utils.isStandardRule)(rule)) {
        return;
      }
      if (!(0, _utils.isStandardSelector)(rule.selector)) {
        return;
      }

      // Nested selectors are processed in steps, as nesting levels are resolved.
      // Here we skip processing the intermediate parts of selectors (to process only fully resolved selectors)
      if (rule.nodes.some(function (node) {
        return node.type === "rule" || node.type === "atrule";
      })) {
        return;
      }

      // Using `rule.selectors` gets us each selector if there is a comma separated set
      rule.selectors.forEach(function (selector) {
        (0, _postcssResolveNestedSelector2.default)(selector, rule).forEach(function (resolvedSelector) {
          // Process each resolved selector with `checkSelector` via postcss-selector-parser
          (0, _postcssSelectorParser2.default)(function (s) {
            return checkSelector(s, rule);
          }).process(resolvedSelector);
        });
      });
    });
  };
};

var _postcssResolveNestedSelector = require("postcss-resolve-nested-selector");

var _postcssResolveNestedSelector2 = _interopRequireDefault(_postcssResolveNestedSelector);

var _postcssSelectorParser = require("postcss-selector-parser");

var _postcssSelectorParser2 = _interopRequireDefault(_postcssSelectorParser);

var _utils = require("../../utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var ruleName = exports.ruleName = "selector-max-compound-selectors";

var messages = exports.messages = (0, _utils.ruleMessages)(ruleName, {
  expected: function expected(selector, max) {
    return "Expected \"" + selector + "\" to have no more than " + max + " compound selectors";
  }
});