"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.messages = exports.ruleName = undefined;

var _postcssResolveNestedSelector = require("postcss-resolve-nested-selector");

var _postcssResolveNestedSelector2 = _interopRequireDefault(_postcssResolveNestedSelector);

var _utils = require("../../utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var ruleName = exports.ruleName = "selector-no-qualifying-type";

var messages = exports.messages = (0, _utils.ruleMessages)(ruleName, {
  rejected: "Unexpected selector qualified by type"
});

var selectorCharacters = ["#", ".", "["];

function isSelectorCharacters(value) {
  return selectorCharacters.some(function (char) {
    return value.indexOf(char) !== -1;
  });
}

function getLeftNodes(node) {
  var result = [];
  var leftNode = node;

  while (leftNode = leftNode.prev()) {
    if (leftNode.type === "combinator") {
      break;
    }
    if (leftNode.type !== "id" && leftNode.type !== "class" && leftNode.type !== "attribute") {
      continue;
    }

    result.push(leftNode);
  }

  return result;
}

function getRightNodes(node) {
  var result = [];
  var rightNode = node;

  while (rightNode = rightNode.next()) {
    if (rightNode.type === "combinator") {
      break;
    }
    if (rightNode.type !== "id" && rightNode.type !== "class" && rightNode.type !== "attribute") {
      continue;
    }

    result.push(rightNode);
  }

  return result;
}

exports.default = function (enabled, options) {
  return function (root, result) {
    var validOptions = (0, _utils.validateOptions)(result, ruleName, {
      actual: enabled,
      possible: [true, false]
    }, {
      actual: options,
      possible: {
        ignore: ["attribute", "class", "id"]
      },
      optional: true
    });
    if (!validOptions) {
      return;
    }

    root.walkRules(function (rule) {
      if (!(0, _utils.isStandardRule)(rule)) {
        return;
      }
      if ((0, _utils.isKeyframeRule)(rule)) {
        return;
      }
      if (!(0, _utils.isStandardSelector)(rule.selector)) {
        return;
      }
      if (!isSelectorCharacters(rule.selector)) {
        return;
      }

      function checkSelector(selectorAST) {
        selectorAST.walkTags(function (selector) {
          var selectorParent = selector.parent;

          if (selectorParent.nodes.length === 1) {
            return;
          }

          var leftNodes = getLeftNodes(selector);
          var rightNodes = getRightNodes(selector);
          var concatNodes = [].concat(leftNodes, rightNodes);
          var index = selector.sourceIndex;

          concatNodes.forEach(function (selectorNode) {
            if (selectorNode.type === "id" && !(0, _utils.optionsHaveIgnored)(options, "id")) {
              complain(index);
            }

            if (selectorNode.type === "class" && !(0, _utils.optionsHaveIgnored)(options, "class")) {
              complain(index);
            }

            if (selectorNode.type === "attribute" && !(0, _utils.optionsHaveIgnored)(options, "attribute")) {
              complain(index);
            }
          });
        });
      }

      (0, _postcssResolveNestedSelector2.default)(rule.selector, rule).forEach(function (resolvedSelector) {
        (0, _utils.parseSelector)(resolvedSelector, result, rule, checkSelector);
      });

      function complain(index) {
        (0, _utils.report)({
          ruleName: ruleName,
          result: result,
          node: rule,
          message: messages.rejected,
          index: index
        });
      }
    });
  };
};