import { SuperMap } from '../SuperMap';
import { CommonServiceBase } from './CommonServiceBase';

/**
 * @class SuperMap.GeoprocessingService
 * @category  iServer ProcessingAutomationService
 * @classdesc 处理自动化服务接口的基类。
 * @version 10.1.0
 * @extends {SuperMap.CommonServiceBase}
 * @param {string} url - 服务地址。
 * @param {Object} options - 参数。
 * @param {SuperMap.Events} options.events - 处理所有事件的对象。
 * @param {Object} [options.eventListeners] - 事件监听器对象。有 processCompleted 属性可传入处理完成后的回调函数。processFailed 属性传入处理失败后的回调函数。
 */
export class GeoprocessingService extends CommonServiceBase {
    constructor(url, options) {
        options = options || {};
        options.EVENT_TYPES = ['processCompleted', 'processFailed', 'processRunning'];
        super(url, options);
        this.CLASS_NAME = 'SuperMap.GeoprocessingService';
        this.headers = {};
        this.crossOrigin = true;
    }
    /**
     * @function SuperMap.GeoprocessingService.prototype.getTools
     * @description 获取处理自动化工具列表。
     */
    getTools() {
        this._get(`${this.url}/list`);
    }
    /**
     * @function SuperMap.GeoprocessingService.prototype.getTool
     * @description 获取处理自动化工具的ID、名称、描述、输入参数、环境参数和输出结果等相关参数。
     * @param {string} identifier - 处理自动化工具ID。
     */
    getTool(identifier) {
        this._get(`${this.url}/${identifier}`);
    }
    /**
     * @function SuperMap.GeoprocessingService.prototype.execute
     * @description 同步执行处理自动化工具。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {Object} parameter - 处理自动化工具的输入参数。
     * @param {Object} environment - 处理自动化工具的环境参数。
     */
    execute(identifier, parameter, environment) {
        parameter = parameter ? parameter : null;
        environment = environment ? environment : null;
        const executeParamter = { parameter, environment };
        this._get(`${this.url}/${identifier}/execute`, executeParamter);
    }
    /**
     * @function SuperMap.GeoprocessingService.prototype.submitJob
     * @description 异步执行处理自动化工具。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {Object} parameter - 处理自动化工具的输入参数。
     * @param {Object} environments - 处理自动化工具的环境参数。
     */
    submitJob(identifier, parameter, environments) {
        parameter = parameter ? parameter : null;
        environments = environments ? environments : null;
        const asyncParamter = { parameter: parameter, environments: environments };
        this.request({
            url: `${this.url}/${identifier}/jobs`,
            headers: { 'Content-type': 'application/json' },
            method: 'POST',
            data: JSON.stringify(asyncParamter),
            scope: this,
            success: this.serviceProcessCompleted,
            failure: this.serviceProcessFailed
        });
    }

    /**
     * @function SuperMap.GeoprocessingService.prototype.waitForJobCompletion
     * @description 获取处理自动化异步执行状态信息。
     * @param {string} jobId - 处理自动化任务ID。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {Object} options - 状态信息参数。
     * @param {number} options.interval - 定时器时间间隔。
     * @param {Callback} options.statusCallback - 任务状态的回调函数。
     */
    waitForJobCompletion(jobId, identifier, options) {
        const me = this;
        const timer = setInterval(function () {
            const serviceProcessCompleted = function (serverResult) {
                const state = serverResult.state.runState;
                if (options.statusCallback) {
                    options.statusCallback(state);
                }
                switch (state) {
                    case 'FINISHED':
                        clearInterval(timer);
                        me.events.triggerEvent('processCompleted', {
                            result: serverResult
                        });
                        break;
                    case 'FAILED':
                        clearInterval(timer);
                        me.events.triggerEvent('processFailed', {
                            result: serverResult
                        });
                        break;
                    case 'CANCELED':
                        clearInterval(timer);
                        me.events.triggerEvent('processFailed', {
                            result: serverResult
                        });
                        break;
                }
            };
            me._get(`${me.url}/${identifier}/jobs/${jobId}`, null, serviceProcessCompleted);
        }, options.interval);
    }

    /**
     * @function SuperMap.GeoprocessingService.prototype.getJobInfo
     * @description 获取处理自动化任务的执行信息。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {string} jobId - 处理自动化任务ID。
     */
    getJobInfo(identifier, jobId) {
        this._get(`${this.url}/${identifier}/jobs/${jobId}`);
    }

    /**
     * @function SuperMap.GeoprocessingService.prototype.cancelJob
     * @description 取消处理自动化任务的异步执行。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {string} jobId - 处理自动化任务ID。
     */
    cancelJob(identifier, jobId) {
        this._get(`${this.url}/${identifier}/jobs/${jobId}/cancel`);
    }
    /**
     * @function SuperMap.GeoprocessingService.prototype.getJobs
     * @description 获取处理自动化服务任务列表。
     * @param {string} identifier - 处理自动化工具ID。(传参代表identifier算子的任务列表，不传参代表所有任务的列表)
     */
    getJobs(identifier) {
        let url = `${this.url}/jobs`;
        if (identifier) {
            url = `${this.url}/${identifier}/jobs`;
        }
        this._get(url);
    }
    /**
     * @function SuperMap.GeoprocessingService.prototype.getResults
     * @description 处理自动化工具执行的结果等,支持结果过滤。
     * @param {string} identifier - 处理自动化工具ID。
     * @param {string} jobId - 处理自动化任务ID。
     * @param {string} filter - 输出异步结果的id。(可选，传入filter参数时对该处理自动化工具执行的结果进行过滤获取，不填参时显示所有的执行结果)
     */
    getResults(identifier, jobId, filter) {
        let url = `${this.url}/${identifier}/jobs/${jobId}/results`;
        if (filter) {
            url = `${url}/${filter}`;
        }
        this._get(url);
    }
    _get(url, paramter, serviceProcessCompleted, serviceProcessFailed) {
        this.request({
            url: url,
            method: 'GET',
            params: paramter,
            headers: { 'Content-type': 'application/json' },
            scope: this,
            success: serviceProcessCompleted ? serviceProcessCompleted : this.serviceProcessCompleted,
            failure: serviceProcessFailed ? serviceProcessFailed : this.serviceProcessFailed
        });
    }
}
SuperMap.GeoprocessingService = GeoprocessingService;
