/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
// Each message we send on the channel specifies an action we want the other side to cooperate with.
var Actions;
(function (Actions) {
    Actions["GET"] = "get";
    Actions["REPLY"] = "reply";
    Actions["EMIT"] = "emit";
    Actions["ERROR"] = "error";
})(Actions || (Actions = {}));
function isGet(message) {
    return message.switchboardAction === Actions.GET;
}
function isReply(message) {
    return message.switchboardAction === Actions.REPLY;
}
function isEmit(message) {
    return message.switchboardAction === Actions.EMIT;
}
function isError(message) {
    return message.switchboardAction === Actions.ERROR;
}
/**
 * A utility for communications between an iframe and its parent, used by the Superset embedded SDK.
 * This builds useful patterns on top of the basic functionality offered by MessageChannel.
 *
 * Both windows instantiate a Switchboard, passing in their MessagePorts.
 * Calling methods on the switchboard causes messages to be sent through the channel.
 */
export class Switchboard {
    port;
    name = '';
    methods = {};
    // used to make unique ids
    incrementor = 1;
    debugMode;
    isInitialised;
    constructor(params) {
        if (!params) {
            return;
        }
        this.init(params);
    }
    init(params) {
        if (this.isInitialised) {
            this.logError('already initialized');
            return;
        }
        const { port, name = 'switchboard', debug = false } = params;
        this.port = port;
        this.name = name;
        this.debugMode = debug;
        port.addEventListener('message', async (event) => {
            this.log('message received', event);
            const message = event.data;
            if (isGet(message)) {
                // find the method, call it, and reply with the result
                this.port.postMessage(await this.getMethodResult(message));
            }
            else if (isEmit(message)) {
                const { method, args } = message;
                // Find the method and call it, but no result necessary.
                // Should this multicast to a set of listeners?
                // Maybe, but that requires writing a bunch more code
                // and I haven't found a need for it yet.
                const executor = this.methods[method];
                if (executor) {
                    executor(args);
                }
            }
        });
        this.isInitialised = true;
    }
    async getMethodResult({ messageId, method, args, }) {
        const executor = this.methods[method];
        if (executor == null) {
            return {
                switchboardAction: Actions.ERROR,
                messageId,
                error: `[${this.name}] Method "${method}" is not defined`,
            };
        }
        try {
            const result = await executor(args);
            return {
                switchboardAction: Actions.REPLY,
                messageId,
                result,
            };
        }
        catch (err) {
            this.logError(err);
            return {
                switchboardAction: Actions.ERROR,
                messageId,
                error: `[${this.name}] Method "${method}" threw an error`,
            };
        }
    }
    /**
     * Defines a method that can be "called" from the other side by sending an event.
     */
    defineMethod(methodName, executor) {
        this.methods[methodName] = executor;
    }
    /**
     * Calls a method registered on the other side, and returns the result.
     *
     * How this is accomplished:
     * This switchboard sends a "get" message over the channel describing which method to call with which arguments.
     * The other side's switchboard finds a method with that name, and calls it with the arguments.
     * It then packages up the returned value into a "reply" message, sending it back to us across the channel.
     * This switchboard has attached a listener on the channel, which will resolve with the result when a reply is detected.
     *
     * Instead of an arguments list, arguments are supplied as a map.
     *
     * @param method the name of the method to call
     * @param args arguments that will be supplied. Must be serializable, no functions or other nonsense.
     * @returns whatever is returned from the method
     */
    get(method, args = undefined) {
        return new Promise((resolve, reject) => {
            if (!this.isInitialised) {
                reject(new Error('Switchboard not initialised'));
                return;
            }
            // In order to "call a method" on the other side of the port,
            // we will send a message with a unique id
            const messageId = this.getNewMessageId();
            // attach a new listener to our port, and remove it when we get a response
            const listener = (event) => {
                const message = event.data;
                if (message.messageId !== messageId)
                    return;
                this.port.removeEventListener('message', listener);
                if (isReply(message)) {
                    resolve(message.result);
                }
                else {
                    const errStr = isError(message)
                        ? message.error
                        : 'Unexpected response message';
                    reject(new Error(errStr));
                }
            };
            this.port.addEventListener('message', listener);
            this.port.start();
            const message = {
                switchboardAction: Actions.GET,
                method,
                messageId,
                args,
            };
            this.port.postMessage(message);
        });
    }
    /**
     * Emit calls a method on the other side just like get does.
     * But emit doesn't wait for a response, it just sends and forgets.
     *
     * @param method
     * @param args
     */
    emit(method, args = undefined) {
        if (!this.isInitialised) {
            this.logError('Switchboard not initialised');
            return;
        }
        const message = {
            switchboardAction: Actions.EMIT,
            method,
            args,
        };
        this.port.postMessage(message);
    }
    start() {
        if (!this.isInitialised) {
            this.logError('Switchboard not initialised');
            return;
        }
        this.port.start();
    }
    log(...args) {
        if (this.debugMode) {
            console.debug(`[${this.name}]`, ...args);
        }
    }
    logError(...args) {
        console.error(`[${this.name}]`, ...args);
    }
    getNewMessageId() {
        // eslint-disable-next-line no-plusplus
        return `m_${this.name}_${this.incrementor++}`;
    }
}
export default new Switchboard();
//# sourceMappingURL=switchboard.js.map