export type Params = {
    port: MessagePort;
    name?: string;
    debug?: boolean;
};
type Method<A extends {}, R> = (args: A) => R | Promise<R>;
/**
 * A utility for communications between an iframe and its parent, used by the Superset embedded SDK.
 * This builds useful patterns on top of the basic functionality offered by MessageChannel.
 *
 * Both windows instantiate a Switchboard, passing in their MessagePorts.
 * Calling methods on the switchboard causes messages to be sent through the channel.
 */
export declare class Switchboard {
    port: MessagePort;
    name: string;
    methods: Record<string, Method<any, unknown>>;
    incrementor: number;
    debugMode: boolean;
    private isInitialised;
    constructor(params?: Params);
    init(params: Params): void;
    private getMethodResult;
    /**
     * Defines a method that can be "called" from the other side by sending an event.
     */
    defineMethod<A extends {}, R = any>(methodName: string, executor: Method<A, R>): void;
    /**
     * Calls a method registered on the other side, and returns the result.
     *
     * How this is accomplished:
     * This switchboard sends a "get" message over the channel describing which method to call with which arguments.
     * The other side's switchboard finds a method with that name, and calls it with the arguments.
     * It then packages up the returned value into a "reply" message, sending it back to us across the channel.
     * This switchboard has attached a listener on the channel, which will resolve with the result when a reply is detected.
     *
     * Instead of an arguments list, arguments are supplied as a map.
     *
     * @param method the name of the method to call
     * @param args arguments that will be supplied. Must be serializable, no functions or other nonsense.
     * @returns whatever is returned from the method
     */
    get<T = unknown>(method: string, args?: unknown): Promise<T>;
    /**
     * Emit calls a method on the other side just like get does.
     * But emit doesn't wait for a response, it just sends and forgets.
     *
     * @param method
     * @param args
     */
    emit(method: string, args?: unknown): void;
    start(): void;
    private log;
    private logError;
    private getNewMessageId;
}
declare const _default: Switchboard;
export default _default;
//# sourceMappingURL=switchboard.d.ts.map