import Kapsule from 'kapsule';
import { range } from 'd3-array';
import { scaleLinear } from 'd3-scale';
import { select } from 'd3-selection';

var moveToFront = Kapsule({
  init: function init(el) {
    el.parentNode.appendChild(el);
  }
});

var gradient = Kapsule({
  props: {
    id: {},
    // Use with: .attr('fill', 'url(#<gradId>)');
    colorScale: {
      "default": scaleLinear().range(['black', 'white'])
    },
    angle: {
      "default": 0
    } // 0 (left-right), 90 (down-up))

  },
  init: function init(el, state) {
    state.id = "areaGradient".concat(Math.round(Math.random() * 10000));
    state.gradient = select(el).append('linearGradient');
  },
  update: function update(state) {
    var rad = Math.PI * state.angle / 180;
    state.gradient.attr('y1', Math.round(100 * Math.max(0, Math.sin(rad))) + '%').attr('y2', Math.round(100 * Math.max(0, -Math.sin(rad))) + '%').attr('x1', Math.round(100 * Math.max(0, -Math.cos(rad))) + '%').attr('x2', Math.round(100 * Math.max(0, Math.cos(rad))) + '%').attr('id', state.id);
    var stopsScale = scaleLinear().domain([0, 100]).range(state.colorScale.domain());
    var colorStops = state.gradient.selectAll('stop').data(range(0, 100.01, 20)); // 11 stops is sufficient to cover all noticeable color nuances

    colorStops.exit().remove();
    colorStops.merge(colorStops.enter().append('stop')).attr('offset', function (d) {
      return "".concat(d, "%");
    }).attr('stop-color', function (d) {
      return state.colorScale(stopsScale(d));
    });
  }
});

var dropshadow = Kapsule({
  // Use with: .attr('filter', 'url(#<shadowId>)'))
  props: {
    id: {
      "default": "areaGradient".concat(Math.round(Math.random() * 10000))
    }
  },
  init: function init(el, state) {
    state.filter = select(el).append('defs').append('filter').attr('height', '130%');
    state.filter.append('feGaussianBlur').attr('in', 'SourceAlpha').attr('stdDeviation', 3);
    state.filter.append('feOffset').attr('dx', 2).attr('dy', 2).attr('result', 'offsetblur');
    var feMerge = state.filter.append('feMerge');
    feMerge.append('feMergeNode');
    feMerge.append('feMergeNode').attr('in', 'SourceGraphic');
  },
  update: function update(state) {
    state.filter.attr('id', state.id);
  }
});

var throbber = Kapsule({
  props: {
    x: {
      "default": 0
    },
    y: {
      "default": 0
    },
    r: {
      "default": 8
    },
    color: {
      "default": 'darkblue'
    },
    duration: {
      "default": 0.7
    },
    angleFull: {
      "default": 120
    }
  },
  init: function init(el, state) {
    el = select(el);
    state.path = el.append('path');
    state.transform = state.path.append('animateTransform').attr('attributeName', 'transform').attr('attributeType', 'XML').attr('type', 'rotate').attr('begin', '0s').attr('fill', 'freeze').attr('repeatCount', 'indefinite');
  },
  update: function update(state) {
    state.path.attr('d', genDonutSlice(state.x, state.y, state.r, state.r / 3, 0, state.angleFull)).attr('fill', state.color);
    state.transform.attr('from', '0 ' + state.x + ' ' + state.y).attr('to', '360 ' + state.x + ' ' + state.y).attr('dur', state.duration + 's'); //

    function genDonutSlice(cx, cy, r, thickness, startAngle, endAngle) {
      startAngle = startAngle / 180 * Math.PI;
      endAngle = endAngle / 180 * Math.PI;
      var outerR = r;
      var innerR = r - thickness;
      var p = [[cx + outerR * Math.cos(startAngle), cy + outerR * Math.sin(startAngle)], [cx + outerR * Math.cos(endAngle), cy + outerR * Math.sin(endAngle)], [cx + innerR * Math.cos(endAngle), cy + innerR * Math.sin(endAngle)], [cx + innerR * Math.cos(startAngle), cy + innerR * Math.sin(startAngle)]];
      var angleDiff = endAngle - startAngle;
      var largeArc = angleDiff % (Math.PI * 2) > Math.PI ? 1 : 0;
      var path = [];
      path.push('M' + p[0].join());
      path.push('A' + [outerR, outerR, 0, largeArc, 1, p[1]].join());
      path.push('L' + p[2].join());
      path.push('A' + [innerR, innerR, 0, largeArc, 0, p[3]].join());
      path.push('z');
      return path.join(' ');
    }
  }
});

var image = Kapsule({
  props: {
    imgUrl: {},
    x: {
      "default": 0
    },
    y: {
      "default": 0
    },
    maxWidth: {
      "default": 20
    },
    maxHeight: {
      "default": 20
    },
    svgAlign: {
      "default": 'xMidYMid'
    }
  },
  methods: {
    show: function show(state) {
      state.img.attr('width', state.maxWidth).attr('height', state.maxHeight);
      return this;
    },
    hide: function hide(state) {
      state.img.attr('width', 0).attr('height', 0);
      return this;
    }
  },
  init: function init(el, state) {
    state.img = select(el).append('image');
  },
  update: function update(state) {
    state.img.attr('xlink:href', state.imgUrl).attr('x', state.x).attr('y', state.y).attr('width', state.maxW).attr('height', state.maxH).attr('preserveAspectRatio', state.svgAlign + ' meet');
  }
});

var zoomOverviewArea = Kapsule({
  props: {
    selection: {
      "default": {
        x: [null, null],
        // [start, end]
        y: [null, null] // [start, end]

      }
    },
    xDomain: {
      onChange: function onChange(xDomain, state) {
        state.xScale.domain(xDomain);
      }
    },
    yDomain: {
      onChange: function onChange(yDomain, state) {
        state.yScale.domain(yDomain);
      }
    },
    transitionDuration: 700
  },
  stateInit: {
    xScale: scaleLinear(),
    yScale: scaleLinear()
  },
  init: function init(el, state, _ref) {
    var width = _ref.width,
        height = _ref.height,
        _ref$margin = _ref.margin,
        margin = _ref$margin === void 0 ? {
      top: 2,
      right: 2,
      bottom: 2,
      left: 2
    } : _ref$margin;
    state.xScale.range([margin.left, width - state.margin.right]);
    state.yScale.range([margin.top, height - state.margin.bottom]); // Build dom

    state.svg = select(el).append('svg').attr('width', width).attr('height', height);
    state.outerBox = state.svg.append('rect').attr('x', state.xScale.range()[0]).attr('y', state.yScale.range()[0]).attr('rx', 2).attr('ry', 2).attr('width', state.xScale.range()[1]).attr('height', state.yScale.range()[1]).style('fill', '#EEE').style('stroke', 'grey');
    state.selectionBox = state.svg.append('rect').attr('rx', 1).attr('ry', 1).attr('width', 0).attr('height', 0).style('stroke', 'blue').style('stroke-opacity', 0.6).style('fill', 'blue').style('fill-opacity', 0.3);
  },
  update: function update(state) {
    state.outerBox.attr('x', state.xScale.range()[0]).attr('y', state.yScale.range()[0]).attr('width', state.xScale.range()[1]).attr('height', state.yScale.range()[1]);
    state.selectionBox.attr('x', state.xScale(state.selection.x[0])).attr('y', state.yScale(state.selection.y[0])).attr('width', state.xScale(state.selection.x[1] - state.selection.x[0])).attr('height', state.yScale(state.selection.y[1] - state.selection.y[0]));
  }
});

export { dropshadow, gradient, image, moveToFront, throbber, zoomOverviewArea };
