"use strict";

exports.__esModule = true;
exports.uriToPointer = exports.unescape = exports.parse = exports.evaluate = exports.escape = exports.compile = void 0;
var _ramda = require("ramda");
var _ramdaAdjunct = require("ramda-adjunct");
var _apidomCore = require("@swagger-api/apidom-core");
var _index = require("./errors/index.cjs");
exports.InvalidJsonPointerError = _index.InvalidJsonPointerError;
exports.EvaluationJsonPointerError = _index.EvaluationJsonPointerError;
/**
 * decodeURIComponent can throw URIError in certain cases like 'c%d'.
 * safeDecodeURIComponent is a safe variant of decodeURIComponent that never trows.
 *
 * {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Errors/Malformed_URI|More info about URIError}
 */
const safeDecodeURIComponent = encodedURIComponent => {
  try {
    return decodeURIComponent(encodedURIComponent);
  } catch {
    return encodedURIComponent;
  }
};

// escape :: String -> String
const escape = (0, _ramda.pipe)((0, _ramda.replace)(/~/g, '~0'), (0, _ramda.replace)(/\//g, '~1'), encodeURIComponent);

// unescape :: String -> String
exports.escape = escape;
const unescape = (0, _ramda.pipe)((0, _ramda.replace)(/~1/g, '/'), (0, _ramda.replace)(/~0/g, '~'), safeDecodeURIComponent);

// parse :: String -> String[]
exports.unescape = unescape;
const parse = pointer => {
  if ((0, _ramdaAdjunct.isEmptyString)(pointer)) {
    return [];
  }
  if (!(0, _ramda.startsWith)('/', pointer)) {
    throw new _index.InvalidJsonPointerError(pointer);
  }
  const tokens = (0, _ramda.pipe)((0, _ramda.split)('/'), (0, _ramda.map)(unescape))(pointer);
  return (0, _ramda.tail)(tokens);
};

// compile :: String[] -> String
exports.parse = parse;
const compile = tokens => {
  if (tokens.length === 0) {
    return '';
  }
  return `/${tokens.map(escape).join('/')}`;
};

// evaluates JSON Pointer against ApiDOM fragment
exports.compile = compile;
const evaluate = (pointer, element) => {
  const tokens = parse(pointer);
  return tokens.reduce((acc, token) => {
    if ((0, _apidomCore.isObjectElement)(acc)) {
      // @ts-ignore
      if (!acc.hasKey(token)) {
        throw new _index.EvaluationJsonPointerError(`Evaluation failed on token: "${token}"`);
      }
      // @ts-ignore
      return acc.get(token);
    }
    if ((0, _apidomCore.isArrayElement)(acc)) {
      if (!(token in acc.content) || !(0, _ramdaAdjunct.isInteger)(Number(token))) {
        throw new _index.EvaluationJsonPointerError(`Evaluation failed on token: "${token}"`);
      }
      // @ts-ignore
      return acc.get(Number(token));
    }
    throw new _index.EvaluationJsonPointerError(`Evaluation failed on token: "${token}"`);
  }, element);
};

/**
 * Returns the hash (URL fragment), of the given path.
 * If there is no hash, then the root hash ("#") is returned.
 */
exports.evaluate = evaluate;
const getHash = uri => {
  const hashIndex = uri.indexOf('#');
  if (hashIndex !== -1) {
    return uri.substring(hashIndex);
  }
  return '#';
};

// uriToPointer :: String -> String
const uriToPointer = uri => {
  const hash = getHash(uri);
  return (0, _ramdaAdjunct.trimCharsStart)('#', hash);
};
exports.uriToPointer = uriToPointer;