"use strict";

exports.__esModule = true;
exports.default = void 0;
var _ramda = require("ramda");
var _apidomCore = require("@swagger-api/apidom-core");
const removeSpaces = operationId => {
  return operationId.replace(/\s/g, '');
};
const replaceSpecialCharsWithUnderscore = operationId => {
  return operationId.replace(/\W/gi, '_');
};
const createNormalizedOperationId = (path, method) => {
  const normalizedMethod = replaceSpecialCharsWithUnderscore(removeSpaces(method.toLowerCase()));
  const normalizedPath = replaceSpecialCharsWithUnderscore(removeSpaces(path));
  return `${normalizedMethod}${normalizedPath}`;
};
const normalizeOperationId = (operationId, path, method) => {
  const withoutSpaces = removeSpaces(operationId);
  if (withoutSpaces.length > 0) {
    return replaceSpecialCharsWithUnderscore(withoutSpaces);
  }
  return createNormalizedOperationId(path, method);
};

/**
 * Normalization of Operation.operationId field.
 *
 * This normalization is not guided by OpenAPI 3.1 specification.
 *
 * Existing Operation.operationId fields are normalized into snake case form.
 *
 * Operation Objects, that do not define operationId field, are left untouched.
 *
 * Original operationId is stored in meta and as new `__originalOperationId` field.
 *
 * This plugin also guarantees the uniqueness of all defined Operation.operationId fields,
 * and make sure Link.operationId fields are pointing to correct and normalized Operation.operationId fields.
 *
 */
/* eslint-disable no-param-reassign */

const plugin = ({
  operationIdNormalizer = normalizeOperationId
} = {}) => ({
  predicates,
  namespace
}) => {
  const paths = [];
  const normalizedOperations = [];
  const links = [];
  return {
    visitor: {
      OpenApi3_1Element: {
        leave() {
          // group normalized operations by normalized operationId
          const normalizedOperationGroups = (0, _ramda.groupBy)(operationElement => {
            return (0, _apidomCore.toValue)(operationElement.operationId);
          }, normalizedOperations);

          // append incremental numerical suffixes to identical operationIds
          Object.entries(normalizedOperationGroups).forEach(([normalizedOperationId, operationElements]) => {
            if (operationElements.length <= 1) return;
            operationElements.forEach((operationElement, index) => {
              const indexedNormalizedOperationId = `${normalizedOperationId}${index + 1}`;
              // @ts-ignore
              operationElement.operationId = new namespace.elements.String(indexedNormalizedOperationId);
            });
          });

          // rectify possibly broken Link.operationId fields
          links.forEach(linkElement => {
            var _operationElement$ope;
            if (typeof linkElement.operationId === 'undefined') return;
            const linkOperationId = String((0, _apidomCore.toValue)(linkElement.operationId));
            const operationElement = normalizedOperations.find(normalizedOperationElement => {
              const originalOperationId = (0, _apidomCore.toValue)(normalizedOperationElement.meta.get('originalOperationId'));
              return originalOperationId === linkOperationId;
            });

            // Link Object doesn't need to be rectified
            if (typeof operationElement === 'undefined') return;
            linkElement.operationId = (_operationElement$ope = operationElement.operationId) === null || _operationElement$ope === void 0 ? void 0 : _operationElement$ope.clone();
            linkElement.meta.set('originalOperationId', linkOperationId);
            linkElement.set('__originalOperationId', linkOperationId);
          });

          // cleanup the references
          normalizedOperations.length = 0;
          links.length = 0;
        }
      },
      PathItemElement: {
        enter(pathItemElement) {
          // `path` meta may not be always available, e.g. in Callback Object or Components Object
          const path = (0, _ramda.defaultTo)('path', (0, _apidomCore.toValue)(pathItemElement.meta.get('path')));
          paths.push(path);
        },
        leave() {
          paths.pop();
        }
      },
      OperationElement: {
        enter(operationElement) {
          // operationId field is undefined, needs no normalization
          if (typeof operationElement.operationId === 'undefined') return;

          // cast operationId to string type
          const originalOperationId = String((0, _apidomCore.toValue)(operationElement.operationId));
          // perform operationId normalization
          const path = (0, _ramda.last)(paths);
          // `http-method` meta may not be always available, e.g. in Callback Object or Components Object
          const method = (0, _ramda.defaultTo)('method', (0, _apidomCore.toValue)(operationElement.meta.get('http-method')));
          const normalizedOperationId = operationIdNormalizer(originalOperationId, path, method);

          // normalization is not necessary
          if (originalOperationId === normalizedOperationId) return;

          // @ts-ignore
          operationElement.operationId = new namespace.elements.String(normalizedOperationId);
          operationElement.set('__originalOperationId', originalOperationId);
          operationElement.meta.set('originalOperationId', originalOperationId);
          normalizedOperations.push(operationElement);
        }
      },
      LinkElement: {
        leave(linkElement) {
          // make sure this Link elements doesn't come from base namespace
          if (!predicates.isLinkElement(linkElement)) return;
          // ignore Link Objects with undefined operationId
          if (typeof linkElement.operationId === 'undefined') return;
          links.push(linkElement);
        }
      }
    }
  };
};
/* eslint-enable */
var _default = plugin;
exports.default = _default;