import stampit from 'stampit';
import { defaultTo, propEq } from 'ramda';
import { createNamespace, visit } from '@swagger-api/apidom-core';
import openApi3_0Namespace, { getNodeType, isOpenApi3_0Element, keyMap, mediaTypes } from '@swagger-api/apidom-ns-openapi-3-0';
import DereferenceStrategy from "../DereferenceStrategy.js";
import Reference from "../../../Reference.js";
import ReferenceSet from "../../../ReferenceSet.js";
import OpenApi3_0DereferenceVisitor from "./visitor.js"; // @ts-ignore
const visitAsync = visit[Symbol.for('nodejs.util.promisify.custom')];

// eslint-disable-next-line @typescript-eslint/naming-convention
const OpenApi3_0DereferenceStrategy = stampit(DereferenceStrategy, {
  init() {
    this.name = 'openapi-3-0';
  },
  methods: {
    canDereference(file) {
      var _file$parseResult;
      // assert by media type
      if (file.mediaType !== 'text/plain') {
        return mediaTypes.includes(file.mediaType);
      }

      // assert by inspecting ApiDOM
      return isOpenApi3_0Element((_file$parseResult = file.parseResult) === null || _file$parseResult === void 0 ? void 0 : _file$parseResult.api);
    },
    async dereference(file, options) {
      const namespace = createNamespace(openApi3_0Namespace);
      const refSet = defaultTo(ReferenceSet(), options.dereference.refSet);
      let reference;
      if (!refSet.has(file.uri)) {
        reference = Reference({
          uri: file.uri,
          value: file.parseResult
        });
        refSet.add(reference);
      } else {
        // pre-computed refSet was provided as configuration option
        reference = refSet.find(propEq('uri', file.uri));
      }
      const visitor = OpenApi3_0DereferenceVisitor({
        reference,
        namespace,
        options
      });
      const dereferencedElement = await visitAsync(refSet.rootRef.value, visitor, {
        keyMap,
        nodeTypeGetter: getNodeType
      });

      /**
       * Release all memory if this refSet was not provided as an configuration option.
       * If provided as configuration option, then provider is responsible for cleanup.
       */
      if (options.dereference.refSet === null) {
        refSet.clean();
      }
      return dereferencedElement;
    }
  }
});
export default OpenApi3_0DereferenceStrategy;