import { isEmpty } from 'ramda';
import { isParseResultElement, ParseResultElement } from '@swagger-api/apidom-core';
import { merge as mergeOptions } from "../options/util.js";
import parse from "../parse/index.js";
import * as plugins from "../util/plugins.js";
import File from "../util/File.js";
import { ResolverError, UnmatchedResolveStrategyError } from "../util/errors/index.js";
import * as url from "../util/url.js";
/**
 * Resolves ApiDOM with all its external references.
 */
export const resolveApiDOM = async (element, options) => {
  // @ts-ignore
  let parseResult = element;

  // wrap element into parse result
  if (!isParseResultElement(element)) {
    // shallow clone of the element
    // @ts-ignore
    const elementClone = new element.constructor(element.content, element.meta.clone(), element.attributes);
    elementClone.classes.push('result');
    parseResult = new ParseResultElement([elementClone]);
  }
  const sanitizedURI = url.sanitize(url.stripHash(options.resolve.baseURI));
  const file = File({
    uri: sanitizedURI,
    parseResult,
    mediaType: options.parse.mediaType
  });
  const resolveStrategies = await plugins.filter('canResolve', file, options.resolve.strategies);

  // we couldn't find any resolver for this File
  if (isEmpty(resolveStrategies)) {
    throw new UnmatchedResolveStrategyError(file.uri);
  }
  try {
    const {
      result
    } = await plugins.run('resolve', [file, options], resolveStrategies);
    return result;
  } catch (error) {
    throw new ResolverError(`Error while resolving file "${file.uri}"`, {
      cause: error
    });
  }
};

/**
 * Resolves a file with all its external references.
 */
const resolve = async (uri, options) => {
  const parseResult = await parse(uri, options);
  const mergedOptions = mergeOptions(options, {
    resolve: {
      baseURI: uri
    }
  });
  return resolveApiDOM(parseResult, mergedOptions);
};
export default resolve;