'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

exports.default = function () {
  return function (css) {
    const classLookup = buildClassTable(css);

    css.walkRules(rule => {
      rule.walkAtRules('apply', atRule => {
        const mixins = _postcss2.default.list.space(atRule.params);

        /*
         * Don't wreck CSSNext-style @apply rules:
         * http://cssnext.io/features/#custom-properties-set-apply
         *
         * These are deprecated in CSSNext but still playing it safe for now.
         * We might consider renaming this at-rule.
         */
        const [customProperties, classes] = _lodash2.default.partition(mixins, mixin => {
          return _lodash2.default.startsWith(mixin, '--');
        });

        const decls = (0, _lodash2.default)(classes).reject(mixin => mixin === '!important').flatMap(mixin => {
          return findMixin(classLookup, normalizeClassName(mixin), message => {
            throw atRule.error(message);
          });
        }).value();

        _lodash2.default.tap(_lodash2.default.last(mixins) === '!important', important => {
          decls.forEach(decl => decl.important = important);
        });

        atRule.before(decls);

        atRule.params = customProperties.join(' ');

        if (_lodash2.default.isEmpty(customProperties)) {
          atRule.remove();
        }
      });
    });
  };
};

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _postcss = require('postcss');

var _postcss2 = _interopRequireDefault(_postcss);

var _escapeClassName = require('../util/escapeClassName');

var _escapeClassName2 = _interopRequireDefault(_escapeClassName);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function buildClassTable(css) {
  const classTable = {};

  css.walkRules(rule => {
    if (!_lodash2.default.has(classTable, rule.selector)) {
      classTable[rule.selector] = [];
    }
    classTable[rule.selector].push(rule);
  });

  return classTable;
}

function normalizeClassName(className) {
  return `.${(0, _escapeClassName2.default)(_lodash2.default.trimStart(className, '.'))}`;
}

function findMixin(classTable, mixin, onError) {
  const matches = _lodash2.default.get(classTable, mixin, []);

  if (_lodash2.default.isEmpty(matches)) {
    // prettier-ignore
    onError(`\`@apply\` cannot be used with \`${mixin}\` because \`${mixin}\` either cannot be found, or it's actual definition includes a pseudo-selector like :hover, :active, etc. If you're sure that \`${mixin}\` exists, make sure that any \`@import\` statements are being properly processed *before* Tailwind CSS sees your CSS, as \`@apply\` can only be used for classes in the same CSS tree.`);
  }

  if (matches.length > 1) {
    // prettier-ignore
    onError(`\`@apply\` cannot be used with ${mixin} because ${mixin} is included in multiple rulesets.`);
  }

  const [match] = matches;

  if (match.parent.type !== 'root') {
    // prettier-ignore
    onError(`\`@apply\` cannot be used with ${mixin} because ${mixin} is nested inside of an at-rule (@${match.parent.name}).`);
  }

  return match.clone().nodes;
}