'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

exports.default = function (config, generatedUtilities) {
  return function (css) {
    const classLookup = buildClassTable(css);
    const shadowLookup = buildShadowTable(generatedUtilities);

    css.walkRules(rule => {
      rule.walkAtRules('apply', atRule => {
        const classesAndProperties = _postcss2.default.list.space(atRule.params);

        /*
         * Don't wreck CSSNext-style @apply rules:
         * http://cssnext.io/features/#custom-properties-set-apply
         *
         * These are deprecated in CSSNext but still playing it safe for now.
         * We might consider renaming this at-rule.
         */
        const [customProperties, classes] = _lodash2.default.partition(classesAndProperties, classOrProperty => {
          return _lodash2.default.startsWith(classOrProperty, '--');
        });

        const decls = (0, _lodash2.default)(classes).reject(cssClass => cssClass === '!important').flatMap(cssClass => {
          return findClass(normalizeClassName(cssClass), classLookup, shadowLookup, message => {
            return atRule.error(message);
          });
        }).value();

        _lodash2.default.tap(_lodash2.default.last(classesAndProperties) === '!important', important => {
          decls.forEach(decl => decl.important = important);
        });

        atRule.before(decls);

        atRule.params = customProperties.join(' ');

        if (_lodash2.default.isEmpty(customProperties)) {
          atRule.remove();
        }
      });
    });
  };
};

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _postcss = require('postcss');

var _postcss2 = _interopRequireDefault(_postcss);

var _escapeClassName = require('../util/escapeClassName');

var _escapeClassName2 = _interopRequireDefault(_escapeClassName);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function buildClassTable(css) {
  const classTable = {};

  css.walkRules(rule => {
    if (!_lodash2.default.has(classTable, rule.selector)) {
      classTable[rule.selector] = [];
    }
    classTable[rule.selector].push(rule);
  });

  return classTable;
}

function buildShadowTable(generatedUtilities) {
  const utilities = _postcss2.default.root();

  _postcss2.default.root({ nodes: generatedUtilities }).walkAtRules('variants', atRule => {
    utilities.append(atRule.clone().nodes);
  });

  return buildClassTable(utilities);
}

function normalizeClassName(className) {
  return `.${(0, _escapeClassName2.default)(_lodash2.default.trimStart(className, '.'))}`;
}

function findClass(classToApply, classTable, shadowLookup, onError) {
  const matches = _lodash2.default.get(classTable, classToApply, []);

  if (_lodash2.default.isEmpty(matches)) {
    if (_lodash2.default.isEmpty(shadowLookup)) {
      // prettier-ignore
      throw onError(`\`@apply\` cannot be used with \`${classToApply}\` because \`${classToApply}\` either cannot be found, or it's actual definition includes a pseudo-selector like :hover, :active, etc. If you're sure that \`${classToApply}\` exists, make sure that any \`@import\` statements are being properly processed *before* Tailwind CSS sees your CSS, as \`@apply\` can only be used for classes in the same CSS tree.`);
    }

    return findClass(classToApply, shadowLookup, {}, onError);
  }

  if (matches.length > 1) {
    // prettier-ignore
    throw onError(`\`@apply\` cannot be used with ${classToApply} because ${classToApply} is included in multiple rulesets.`);
  }

  const [match] = matches;

  if (match.parent.type !== 'root') {
    // prettier-ignore
    throw onError(`\`@apply\` cannot be used with ${classToApply} because ${classToApply} is nested inside of an at-rule (@${match.parent.name}).`);
  }

  return match.clone().nodes;
}