import { PasscodeState } from "../state/users/PasscodeState";
import { Passcode } from "../Dto";
import { Client } from "./Client";
import { HttpClientOptions } from "./HttpClient";
/**
 * A class to handle passcodes.
 *
 * @constructor
 * @category SDK
 * @subcategory Clients
 * @extends {Client}
 */
declare class PasscodeClient extends Client {
    state: PasscodeState;
    constructor(api: string, options: HttpClientOptions);
    /**
     * Causes the API to send a new passcode to the user's email address.
     *
     * @param {string} userID - The UUID of the user.
     * @param {string=} emailID - The UUID of the email address. If unspecified, the email will be sent to the primary email address.
     * @param {boolean=} force - Indicates the passcode should be sent, even if there is another active passcode.
     * @return {Promise<Passcode>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @throws {TooManyRequestsError}
     * @see https://docs.hanko.io/api/public#tag/Passcode/operation/passcodeInit
     */
    initialize(userID: string, emailID?: string, force?: boolean): Promise<Passcode>;
    /**
     * Validates the passcode obtained from the email.
     *
     * @param {string} userID - The UUID of the user.
     * @param {string} code - The passcode digests.
     * @return {Promise<void>}
     * @throws {InvalidPasscodeError}
     * @throws {MaxNumOfPasscodeAttemptsReachedError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/Passcode/operation/passcodeFinal
     */
    finalize(userID: string, code: string): Promise<void>;
    /**
     * Returns the number of seconds the current passcode is active for.
     *
     * @param {string} userID - The UUID of the user.
     * @return {number}
     */
    getTTL(userID: string): number;
    /**
     * Returns the number of seconds the rate limiting is active for.
     *
     * @param {string} userID - The UUID of the user.
     * @return {number}
     */
    getResendAfter(userID: string): number;
}
export { PasscodeClient };
