import { User, UserInfo, UserCreated } from "../Dto";
import { Client } from "./Client";
/**
 * A class to manage user information.
 *
 * @category SDK
 * @subcategory Clients
 * @extends {Client}
 */
declare class UserClient extends Client {
    /**
     * Fetches basic information about the user identified by the given email address. Can be used while the user is logged out
     * and is helpful in deciding which type of login to choose. For example, if the user's email is not verified, you may
     * want to log in with a passcode, or if no WebAuthn credentials are registered, you may not want to use WebAuthn.
     *
     * @param {string} email - The user's email address.
     * @return {Promise<UserInfo>}
     * @throws {NotFoundError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/User-Management/operation/getUserId
     */
    getInfo(email: string): Promise<UserInfo>;
    /**
     * Creates a new user. Afterwards, verify the email address via passcode. If a 'ConflictError'
     * occurred, you may want to prompt the user to log in.
     *
     * @param {string} email - The email address of the user to be created.
     * @return {Promise<UserCreated>}
     * @throws {ConflictError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/User-Management/operation/createUser
     */
    create(email: string): Promise<UserCreated>;
    /**
     * Fetches the current user.
     *
     * @return {Promise<User>}
     * @throws {UnauthorizedError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/User-Management/operation/IsUserAuthorized
     * @see https://docs.hanko.io/api/public#tag/User-Management/operation/listUser
     */
    getCurrent(): Promise<User>;
    /**
     * Deletes the current user and expires the existing session cookie.
     *
     * @return {Promise<void>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @throws {UnauthorizedError}
     */
    delete(): Promise<void>;
    /**
     * Logs out the current user and expires the existing session cookie. A valid session cookie is required to call the logout endpoint.
     *
     * @return {Promise<void>}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     */
    logout(): Promise<void>;
}
export { UserClient };
