import { create as createWebauthnCredential, get as getWebauthnCredential } from "@github/webauthn-json";
import { Client } from "./Client";
import { PasscodeState } from "../state/users/PasscodeState";
import { WebauthnState } from "../state/users/WebauthnState";
import { User, WebauthnCredentials, WebauthnFinalized } from "../Dto";
import { HttpClientOptions } from "./HttpClient";
/**
 * A class that handles WebAuthn authentication and registration.
 *
 * @constructor
 * @category SDK
 * @subcategory Clients
 * @extends {Client}
 */
declare class WebauthnClient extends Client {
    webauthnState: WebauthnState;
    passcodeState: PasscodeState;
    controller: AbortController;
    _getCredential: typeof getWebauthnCredential;
    _createCredential: typeof createWebauthnCredential;
    constructor(api: string, options: HttpClientOptions);
    /**
     * Performs a WebAuthn authentication ceremony. When 'userID' is specified, the API provides a list of
     * allowed credentials and the browser is able to present a list of suitable credentials to the user.
     *
     * @param {string=} userID - The user's UUID.
     * @param {boolean=} useConditionalMediation - Enables autofill assisted login.
     * @return {Promise<void>}
     * @throws {WebauthnRequestCancelledError}
     * @throws {InvalidWebauthnCredentialError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/webauthnLoginInit
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/webauthnLoginFinal
     * @see https://www.w3.org/TR/webauthn-2/#authentication-ceremony
     * @return {WebauthnFinalized}
     */
    login(userID?: string, useConditionalMediation?: boolean): Promise<WebauthnFinalized>;
    /**
     * Performs a WebAuthn registration ceremony.
     *
     * @return {Promise<void>}
     * @throws {WebauthnRequestCancelledError}
     * @throws {RequestTimeoutError}
     * @throws {UnauthorizedError}
     * @throws {TechnicalError}
     * @throws {UserVerificationError}
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/webauthnRegInit
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/webauthnRegFinal
     * @see https://www.w3.org/TR/webauthn-2/#sctn-registering-a-new-credential
     */
    register(): Promise<void>;
    /**
     * Returns a list of all WebAuthn credentials assigned to the current user.
     *
     * @return {Promise<WebauthnCredentials>}
     * @throws {UnauthorizedError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/listCredentials
     */
    listCredentials(): Promise<WebauthnCredentials>;
    /**
     * Updates the WebAuthn credential.
     *
     * @param {string=} credentialID - The credential's UUID.
     * @param {string} name - The new credential name.
     * @return {Promise<void>}
     * @throws {UnauthorizedError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/updateCredential
     */
    updateCredential(credentialID: string, name: string): Promise<void>;
    /**
     * Deletes the WebAuthn credential.
     *
     * @param {string=} credentialID - The credential's UUID.
     * @return {Promise<void>}
     * @throws {UnauthorizedError}
     * @throws {RequestTimeoutError}
     * @throws {TechnicalError}
     * @see https://docs.hanko.io/api/public#tag/WebAuthn/operation/deleteCredential
     */
    deleteCredential(credentialID: string): Promise<void>;
    /**
     * Determines whether a credential registration ceremony should be performed. Returns 'true' when WebAuthn
     * is supported and the user's credentials do not intersect with the credentials already known on the
     * current browser/device.
     *
     * @param {User} user - The user object.
     * @return {Promise<boolean>}
     */
    shouldRegister(user: User): Promise<boolean>;
    _createAbortSignal(): AbortSignal;
}
export { WebauthnClient };
